/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import java.util.*;

/**
 * Headers keeps a list of HTTP headers and grants case-insensitiv access to
 * those. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Revision: 1.1 $
 */

public final class Headers {

  /**
   * TBD: Description of the class.
   */
  private static class Header {
    public final String name;
    private String value;
    private boolean added;

    public Header(String name, String value) {
      this.name = name;
      this.value = value;
    }

    public String getValue() {
      return this.value;
    }

    public void add(String value) {
      this.added = true;
      if (this.value == null) {
        this.value = value;
      }
      else {
        StringBuffer sb = new StringBuffer(this.value);
        sb.append(", ").append(value);
        this.value = sb.toString();
      }
    }

    public void set(String value) {
      this.value = value;
      this.added = false;
    }

    public boolean wasAdded() {
      return this.added;
    }
  }

  private final Map m_headers;

  public Headers() {
    m_headers = new HashMap();
  }

  public void clear() {
    m_headers.clear();
  }

  /**
   * Get the header value for name (case insensitive).
   *
   * @param name the header name to look for
   * @return the value of the header or <code>null</code>
   */
  public String get(String name) {
    Header header = (Header)m_headers.get(name.toLowerCase());
    return (header != null) ? header.getValue() : null;
  }

  /**
   * Enumerate all header names
   *
   * @return TBD: Description of the outgoing return value
   */
  public Iterator keys() {
    return
      new Iterator() {
        private Iterator m_iter = m_headers.values().iterator();

        public boolean hasNext() {
          return m_iter.hasNext();
        }

        public Object next() {
          Header header = (Header)m_iter.next();
          if (header != null) {
            return header.name;
          }
          return null;
        }

        public void remove() { }
      };
  }

  /**
   * Set the header field name with value. Will replace any already existing
   * value.
   *
   * @param name of header field
   * @param value of header field
   */
  public void set(String name, String value) {
    String key = name.toLowerCase();
    Header header = (Header)m_headers.get(key);
    if (header != null) {
      header.set(value);
    }
    else {
      m_headers.put(key, new Header(name, value));
    }
  }

  /**
   * Add the header field name with value. Will add to any already existing
   * value.
   *
   * @param name of header field
   * @param value of header field
   */
  public void add(String name, String value) {
    String key = name.toLowerCase();
    Header header = (Header)m_headers.get(key);
    if (header != null) {
      header.add(value);
    }
    else {
      m_headers.put(key, new Header(name, value));
    }
  }

  /**
   * Removes the header field name.
   *
   * @param name of header to remove
   */
  public void remove(String name) {
    m_headers.remove(name.toLowerCase());
  }

  /**
   * @param name TBD: Description of the incoming method parameter
   * @return if this header was constructed using add methods.
   */
  public boolean wasAdded(String name) {
    Header header = (Header)m_headers.get(name);
    if (header != null) {
      return header.wasAdded();
    }
    return false;
  }
}

