/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * Utility class for handling date conversions as needed in HTTP headers and
 * WEBDAV response documents. <p>
 *
 * Copyright (c) SAP AG 2001-2004
 * @author julian.reschke@greenbytes.de
 * @author stefan.eissing@greenbytes.de
 */
public class DateFormat {

  private final static String FORMAT_ISO8601 = "yyyy-MM-dd'T'HH:mm:ss'Z'";
  private final static String FORMAT_ISO8601_MS = "yyyy-MM-dd'T'HH:mm:ss.SSS'Z'";
  private final static String FORMAT_ISO8601_SHORT = "yyyy-MM-dd'T'HH:mm:ss";
  private final static String FORMAT_ISO8601_SHORT_MS = "yyyy-MM-dd'T'HH:mm:ss.SSS";
  private final static String FORMAT_HTTP = "EEE, dd MMM yyyy HH:mm:ss 'GMT'";

  private final static int FORMAT_ISO8601_LEN = FORMAT_ISO8601.length() - 4;
  private final static int FORMAT_ISO8601_SHORT_LEN = FORMAT_ISO8601_SHORT.length() - 2;
  private final static int FORMAT_ISO8601_SHORT_MS_LEN = FORMAT_ISO8601_SHORT_MS.length() - 2;

  private final static TimeZone GMT = TimeZone.getTimeZone("GMT");

  private SimpleDateFormat sdfHTTP;
  private SimpleDateFormat sdfISO8601;
  private SimpleDateFormat sdfISO8601ms;
  private SimpleDateFormat sdfISO8601short;
  private SimpleDateFormat sdfISO8601shortms;

  /**
   * Convert date to RFC1123 format for use in HTTP headers
   *
   * @param date to convert
   * @return converted date
   */
  public String formatHTTP(Date date) {
    checkHTTP();
    return sdfHTTP.format(date);
  }

  /**
   * Convert date to ISO8601 format for use in WEBDAV responses
   *
   * @param date to convert
   * @return converted date
   */
  public String formatISO8601(Date date) {
    checkISO();
    return sdfISO8601.format(date);
  }

  /**
   * Convert string in RFC1123 format to java date.
   *
   * @param text to convert
   * @return converted date
   * @throws java.text.ParseException on wrong format
   */
  public Date parseHTTP(String text)
    throws java.text.ParseException {
    checkHTTP();
    return sdfHTTP.parse(text);
  }

  /**
   * Convert string in ISO8601 format to java date.
   *
   * @param text to convert
   * @return converted date
   * @throws java.text.ParseException on wrong format
   */
  public Date parseISO8601(String text)
    throws java.text.ParseException {
    int len = text.length();
    if (len > 0 && text.charAt(len - 1) == 'Z') {
      if (len > FORMAT_ISO8601_LEN) {
        checkISOms();
        return sdfISO8601ms.parse(text);
      }
      else {
        checkISO();
        return sdfISO8601.parse(text);
      }
    }
    else {
      Date date;
      int slen;
      if (len == FORMAT_ISO8601_SHORT_MS_LEN + 6) {
        checkISOshortms();
        slen = FORMAT_ISO8601_SHORT_MS_LEN;
        date = sdfISO8601shortms.parse(text.substring(0, slen));
      }
      else if (len == FORMAT_ISO8601_SHORT_LEN + 6) {
        checkISOshort();
        slen = FORMAT_ISO8601_SHORT_LEN;
        date = sdfISO8601short.parse(text.substring(0, slen));
      }
      else {
        throw new java.text.ParseException("timezone wrong format: (" +
          text + ") expected " + (FORMAT_ISO8601_SHORT_LEN + 6), len - 1);
      }

      
      char sign = text.charAt(slen);
      if (sign != '+' && sign != '-') {
        throw new java.text.ParseException("timezone offset wrong in " + text, slen);
      }
      
      boolean positive = sign == '+';
      
      if (text.charAt(slen + 3) != ':') {
        throw new java.text.ParseException("timezone offset wrong in " + text, slen + 3);
      }

      int hours = 0;
      int minutes = 0;
      try {
        hours = Integer.parseInt(text.substring(slen + 1, slen + 3));
        minutes = Integer.parseInt(text.substring(slen + 4));
      }
      catch (NumberFormatException ex) {
            //$JL-EXC$        
        throw new java.text.ParseException("timezone offset not number in " + text, slen + 1);
      }

      long offset = (minutes * 60 * 1000L) + (hours * 60 * 60 * 1000L);
      if (positive) {
        offset = -offset;
      }
      return new Date(date.getTime() + offset);
    }
  }

  // ------------------------------- private ------------------------------

  private void checkISO() {
    if (sdfISO8601 == null) {
      sdfISO8601 = new SimpleDateFormat(FORMAT_ISO8601, Locale.ENGLISH);
      sdfISO8601.setTimeZone(GMT);
    }
  }

  private void checkISOms() {
    if (sdfISO8601ms == null) {
      sdfISO8601ms = new SimpleDateFormat(FORMAT_ISO8601_MS, Locale.ENGLISH);
      sdfISO8601ms.setTimeZone(GMT);
    }
  }

  private void checkISOshort() {
    if (sdfISO8601short == null) {
      sdfISO8601short = new SimpleDateFormat(FORMAT_ISO8601_SHORT, Locale.ENGLISH);
      sdfISO8601short.setTimeZone(GMT);
    }
  }

  private void checkISOshortms() {
    if (sdfISO8601shortms == null) {
      sdfISO8601shortms = new SimpleDateFormat(FORMAT_ISO8601_SHORT_MS, Locale.ENGLISH);
      sdfISO8601shortms.setTimeZone(GMT);
    }
  }

  private void checkHTTP() {
    if (sdfHTTP == null) {
      sdfHTTP = new SimpleDateFormat(FORMAT_HTTP, Locale.ENGLISH);
      sdfHTTP.setTimeZone(GMT);
    }
  }

}
