/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.http;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;

/**
 * HTTP Content Type: mime type + optional encoding. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: ContentType.java,v 1.4 2003/02/17 14:24:04 jre Exp $
 */
public final class ContentType {

  public final static ContentType APPLICATION_OCTET_STREAM = new ContentType("application/octet-stream");
  public final static ContentType APPLICATION_XML = new ContentType("application/xml");
  public final static ContentType APPLICATION_PDF = new ContentType("application/pdf");
  public final static ContentType TEXT_HTML = new ContentType("text/html");
  public final static ContentType TEXT_PLAIN = new ContentType("text/plain");
  public final static ContentType TEXT_XML = new ContentType("text/xml");
  public final static ContentType TEXT_XML_UTF8 = new ContentType("text/xml; charset=UTF-8", "text/xml", "UTF-8");

  private final static Map ctypes;

  /**
   * Map of encoding names and their Java equivalents
   */
  private final static Map ENCODINGS;

  static {
    Map encodings = new HashMap(31);

    encodings.put("utf-8", "UTF-8");
    encodings.put("utf8", "UTF-8");
    encodings.put("unicode-1-1-utf-8", "UTF-8");
    encodings.put("unicode-2-0-utf-8", "UTF-8");
    encodings.put("x-unicode-2-0-utf-8", "UTF-8");

    encodings.put("iso88591", "ISO-8859-1");
    encodings.put("iso8859_1", "ISO-8859-1");
    encodings.put("iso8859-1", "ISO-8859-1");
    encodings.put("iso88591", "ISO-8859-1");

    encodings.put("x-sjis", "x-sjis");
    encodings.put("ms_kanji", "x-sjis");
    encodings.put("csshiftjis", "x-sjis");
    encodings.put("shift_jis", "x-sjis");
    encodings.put("cswindow31j", "x-sjis");
    encodings.put("x-ms-cp932", "x-sjis");

    ENCODINGS = Collections.unmodifiableMap(encodings);

    Map map = new HashMap(17);
    map.put(APPLICATION_OCTET_STREAM.ctype, APPLICATION_OCTET_STREAM);
    map.put(APPLICATION_XML.ctype, APPLICATION_XML);
    map.put(APPLICATION_PDF.ctype, APPLICATION_PDF);
    map.put(TEXT_HTML.ctype, TEXT_HTML);
    map.put(TEXT_PLAIN.ctype, TEXT_PLAIN);
    map.put(TEXT_XML.ctype, TEXT_XML);
    map.put(TEXT_XML_UTF8.ctype, TEXT_XML_UTF8);

    ctypes = Collections.unmodifiableMap(map);
  }

  public final static String mapEncoding(String encoding) {
    if (encoding != null && encoding.length() > 0) {
      char c = encoding.charAt(0);
      if (c == '\"' || c == '\'') {
        encoding = encoding.substring(1);
      }
      if (encoding.length() > 0) {
        int end = encoding.length() - 1;
        c = encoding.charAt(end);
        if (c == '\"' || c == '\'') {
          encoding = encoding.substring(0, end);
        }

        if (encoding.length() > 0) {
          String mappedEncoding = (String)ENCODINGS.get(encoding.toLowerCase());
          if (mappedEncoding != null) {
            return mappedEncoding;
          }
          return encoding;
        }
      }
    }
    return null;
  }

  public static ContentType valueOf(String contentType) {
    if (contentType == null || contentType.length() == 0) {
      return APPLICATION_OCTET_STREAM;
    }

    ContentType ct = (ContentType)ctypes.get(contentType);
    if (ct == null) {
      if (contentType.indexOf(";") > 0) {
        String mtype = null;
        String encoding = null;
        StringTokenizer st = new StringTokenizer(contentType, "; ");
        if (st.hasMoreTokens()) {
          mtype = st.nextToken();
        }
        while (st.hasMoreTokens() && encoding == null) {
          String token = st.nextToken();
          if (token.startsWith("charset=")) {
            encoding = token.substring("charset=".length());
            encoding = mapEncoding(encoding);
          }
        }

        if (mtype != null) {
          ct = new ContentType(contentType, mtype, encoding);
        }
        else {
          ct = new ContentType(contentType);
        }
      }
      else {
        ct = new ContentType(contentType);
      }
    }

    return ct;
  }

  public static ContentType valueOf(String mimeType, String encoding) {
    if (encoding == null) {
      return valueOf(mimeType);
    }
    else {
      String contentType = mimeType + "; charset=" + encoding;
      ContentType ct = (ContentType)ctypes.get(contentType);
      if (ct == null) {
        ct = new ContentType(contentType, mimeType, encoding);
      }

      return ct;
    }
  }

  private final String ctype;
  private final String mtype;
  private final String encoding;

  private ContentType(String ctype) {
    this.ctype = ctype;
    this.mtype = ctype;
    this.encoding = null;
  }

  private ContentType(String ctype, String mtype, String encoding) {
    this.ctype = ctype;
    this.mtype = mtype;
    this.encoding = encoding;
  }

  /**
   * Get the HTTP Content-Type header value.
   *
   * @return HTTP Content-Type header value
   */
  public String getContentType() {
    return this.ctype;
  }

  /**
   * Get the Mime Type.
   *
   * @return the mime type
   */
  public String getMimeType() {
    return this.mtype;
  }

  /**
   * Get the Encoding/Charset if defined.
   *
   * @return encoding/charset or <code>null</code> if not defined
   */
  public String getEncoding() {
    return this.encoding;
  }

  /**
   * @param other TBD: Description of the incoming method parameter
   * @return if this mime type equals other's mime type
   */
  public boolean isSameMimeType(ContentType other) {
    return this.mtype.equalsIgnoreCase(other.mtype);
  }

  /**
   * Get a string representation, suitable for HTTP Content-Type header field.
   *
   * @return string representation of content type
   */
  public String toString() {
    return this.ctype;
  }

  public int hashCode() {
    return this.ctype.hashCode();
  }

  public boolean equals(Object obj) {
    if (obj instanceof ContentType) {
      ContentType oc = (ContentType)obj;
      return this.ctype.equals(oc.ctype);
    }
    return false;
  }
}
