/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.html;

/**
 * IHTMLContentHandler receives events from a {@link
 * com.sapportals.wcm.util.html.IHTMLReader}. <p>
 *
 * For each event the {@link com.sapportals.wcm.util.html.IHTMLReader} calls the
 * corresponding method in the content handler. This maps as follows:
 *
 * <tableborder="1">
 *
 *   <tr>
 *
 *     <th>
 *       Document Part
 *     </th>
 *
 *     <th>
 *       Event
 *     </th>
 *
 *     <th>
 *       Notes
 *     </th>
 *
 *   </tr>
 *
 *   <tr>
 *
 *     <td>
 *       (start of input)
 *     </td>
 *
 *     <td>
 *       startDocument
 *     </td>
 *
 *     <td>
 *       This event precedes any other event.
 *     </td>
 *
 *   </tr>
 *
 *   <tr>
 *
 *     <td>
 *       (end of input)
 *     </td>
 *
 *     <td>
 *       endDocument
 *     </td>
 *
 *     <td>
 *       This is the last event for a document.
 *     </td>
 *
 *   </tr>
 *
 *   <tr>
 *
 *     <td>
 *       &lt;tag...&gt;, &lt;tag.../&gt;
 *     </td>
 *
 *     <td>
 *       startElement
 *     </td>
 *
 *     <td>
 *       This event is generated for all tags which are not an end tag (e.g.
 *       start with '&lt;/'. Note that for empty tags (e.g. ending with '/&gt;')
 *       startElement is the only event generated. There will never be an
 *       endElement event.
 *     </td>
 *
 *   </tr>
 *
 *   <tr>
 *
 *     <td>
 *       &lt;/tag&gt;
 *     </td>
 *
 *     <td>
 *       endElement
 *     </td>
 *
 *     <td>
 *       A tag starting with '&lt;/' will generate this event.
 *     </td>
 *
 *   </tr>
 *
 *   <tr>
 *
 *     <td>
 *       otherwise
 *     </td>
 *
 *     <td>
 *       characters
 *     </td>
 *
 *     <td>
 *       The following parts of a HTML document will generate this event:
 *       characters between tags, DOCTYPE declarations, processing instructions,
 *       CDATA sections, any characters between start/end of <code>script</code>
 *       or <code>pre</code> tags.
 *     </td>
 *
 *   </tr>
 *
 * </table>
 * <p>
 *
 * <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: IHTMLContentHandler.java,v 1.2 2001/12/12 17:19:58 sei Exp $
 */
public interface IHTMLContentHandler {

  /**
   * Notification of a character event. The characters of the event are found in
   * <code>ch</code> at offset <code>start</code> . There are <code>length
   * </code> number of characters.<p>
   *
   * The content of the buffer before <code>start</code> or after <code>start +
   * length</code> is undefined. Modification of the character array is strictly
   * forbidden. The content of the array is undefined after this method returns.
   *
   * @param ch array holding characters of event
   * @param start where in the array the characters begin
   * @param length number of characters in event
   * @throws HTMLException to indicate error in event handling
   */
  public void characters(char[] ch, int start, int length)
    throws HTMLException;

  /**
   * Notification that the document is finished.
   *
   * @throws HTMLException to indicate error in event handling
   */
  public void endDocument()
    throws HTMLException;

  /**
   * Notification that an end tag was encountered (e.g. starting with '&lt;/').
   * The element paramter is only valid for the duration of the call. The
   * content of element are undefined when the method returns. See {@link
   * IHTMLElement} for further information.
   *
   * @param element TBD: Description of the incoming method parameter
   * @throws HTMLException to indicate error in event handling
   */
  public void endElement(IHTMLElement element)
    throws HTMLException;

  /**
   * Notification that the document is about to start.
   *
   * @throws HTMLException to indicate error in event handling
   */
  public void startDocument()
    throws HTMLException;

  /**
   * Notification that a tag was encountered. The element paramter is only valid
   * for the duration of the call. The content of element are undefined when the
   * method returns. See {@link IHTMLElementStart} for further information.
   *
   * @param element TBD: Description of the incoming method parameter
   * @throws HTMLException to indicate error in event handling
   */
  public void startElement(IHTMLElementStart element)
    throws HTMLException;

}
