/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.html;
import com.sapportals.wcm.WcmException;

import com.sapportals.wcm.WcmObject;

import java.lang.StringBuffer;
import java.util.Enumeration;
import java.util.Properties;

/**
 * <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author roland.preussmann@sapportals.com
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HtmlTag.java,v 1.2 2001/09/27 13:17:04 sei Exp $
 * @see HtmlTokenizer
 */
public class HtmlTag extends WcmObject {

  private final static char SINGLE_QUOTE = '\'';
  private final static char DOUBLE_QUOTE = '"';
  private final static char EQUAL_SIGN = '=';
  private final static char SLASH = '/';
  private final static char LESS_SIGN = '<';
  private final static char GREATER_SIGN = '>';
  private final static char SPACE = ' ';

  private String m_name;
  private String m_token;
  private boolean m_changed = false;
  private boolean m_endTag = false;
  private Properties m_params = new Properties();

  public HtmlTag() { }

  public HtmlTag(String token)
    throws WcmException {
    parse(token);
  }

  public void parse(String token)
    throws WcmException {
    empty(token);

    int len = token.length();
    int idx = 0;
    int begin = 0;

    // parse tag
    idx = eatSpace(token, idx);

    if (idx == len) {
      throw new WcmException("parse empty tag");
    }

    // end tag ??
    if (token.charAt(idx) == SLASH) {
      m_endTag = true;
      idx++;
    }

    // get tag name
    begin = idx;
    while (idx < len && !isSpace(token.charAt(idx))) {
      idx++;
    }
    m_name = token.substring(begin, idx).toUpperCase();

    // get tag parameters
    idx = eatSpace(token, idx);
    if (idx == len) {
      return;
    }

    while (idx < len) {
      idx = eatSpace(token, idx);
      if (idx == len) {
        continue;
      }

      begin = idx;
      if (token.charAt(idx) == DOUBLE_QUOTE) {
        idx++;
        while (idx < len && token.charAt(idx) != DOUBLE_QUOTE) {
          idx++;
        }
        if (idx == len) {
          continue;
        }// bad name
        idx++;
      }
      else if (token.charAt(idx) == SINGLE_QUOTE) {
        idx++;
        while (idx < len && token.charAt(idx) != SINGLE_QUOTE) {
          idx++;
        }
        if (idx == len) {
          continue;
        }// bad name
        idx++;
      }
      else {
        while (idx < len && !isSpace(token.charAt(idx)) && token.charAt(idx) != EQUAL_SIGN) {
          idx++;
        }
      }

      String name = token.substring(begin, idx).toUpperCase();

      if (idx < len && isSpace(token.charAt(idx))) {
        idx = eatSpace(token, idx);
      }

      if (idx == len || token.charAt(idx) != EQUAL_SIGN) {
        // name with empty value
        // JRE: HACK prevent report of \n as property (see www.w3.org META tag)
        if (!name.equals("/")) {
          m_params.setProperty(name, "");
        }
        continue;
      }
      idx++;

      if (idx == len) {
        continue;
      }

      if (token.charAt(idx) == ' ') {
        // special case: if value is surrounded by quotes
        // then it can have a space after the '='
        while (idx < len && isSpace(token.charAt(idx))) {
          idx++;
        }

        if (idx == len || (token.charAt(idx) != DOUBLE_QUOTE && token.charAt(idx) != SINGLE_QUOTE)) {
          continue;
        }
      }

      begin = idx;
      int end = begin;
      if (token.charAt(idx) == DOUBLE_QUOTE) {
        idx++;
        begin = idx;
        while (idx < len && token.charAt(idx) != DOUBLE_QUOTE) {
          idx++;
        }
        if (idx == len) {
          continue;
        }// bad value
        end = idx;
        idx++;
      }
      else if (token.charAt(idx) == SINGLE_QUOTE) {
        idx++;
        begin = idx;
        while (idx < len && token.charAt(idx) != SINGLE_QUOTE) {
          idx++;
        }
        if (idx == len) {
          continue;
        }// bad value
        end = idx;
        idx++;
      }
      else {
        while (idx < len && !isSpace(token.charAt(idx))) {
          idx++;
        }
        end = idx;
      }

      String value = token.substring(begin, end);
      m_params.setProperty(name, value);
    }
  }

  public String getName() {
    return m_name;
  }

  public boolean isEndtag() {
    return m_endTag;
  }

  public boolean hasParam(String name) {
    return getParam(name) != null;
  }

  public String getParam(String name) {
    return (String)m_params.getProperty(name);
  }

  public void setParam(String name, String value) {
    m_changed = true;
    m_params.setProperty(name, value);
  }

  public String toString() {
    StringBuffer tag = new StringBuffer();
    tag.append(LESS_SIGN);
    if (m_changed) {
      if (m_endTag) {
        tag.append(SLASH);
      }
      tag.append(m_name);
      Enumeration keys = m_params.propertyNames();
      while (keys.hasMoreElements()) {
        String name = (String)keys.nextElement();
        tag.append(SPACE);
        tag.append(name);
        String value = m_params.getProperty(name);
        if (value.length() > 0) {
          tag.append("=\"" + value + "\"");
        }
      }
    }
    else {
      tag.append(m_token);
    }
    tag.append(GREATER_SIGN);
    return tag.toString();
  }

  private void empty(String token) {
    m_name = "";
    m_endTag = false;
    m_params.clear();
    m_token = token;
    m_changed = false;
  }

  private int eatSpace(String token, int idx) {
    int len = token.length();
    while (idx < len && isSpace(token.charAt(idx))) {
      idx++;
    }
    return idx;
  }

  private boolean isSpace(int c) {
    return c == ' ' || c == '\r' || c == '\n' || c == '\t';
  }

  // HACK
  public Properties getParams() {
    return m_params;
  }
}
