/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.html;

import java.io.*;

/**
 * Writes events from a {@link IHTMLReader} onto a stream. <p>
 *
 * A HTMLStreamWriter can be installed as a {@link IHTMLContentHandler} in a
 * reader and will write all events generated by the reader onto the given
 * stream. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HTMLStreamWriter.java,v 1.5 2003/03/20 18:41:30 jre Exp $
 */
public class HTMLStreamWriter implements IHTMLContentHandler {

  private final Writer m_writer;
  private final OutputStream m_os;
  private final String m_encoding;

  /**
   * Will write all events onto <code>out</code> using the default Java
   * encoding.
   *
   * @param out the stream to write on
   */
  public HTMLStreamWriter(OutputStream out) {
    m_os = out;
    m_encoding = null;
    m_writer = new OutputStreamWriter(m_os);
  }

  /**
   * Will write all events onto <code>out</code> using the given Java encoding.
   *
   * @param out the stream to write on
   * @param encoding to use
   * @exception UnsupportedEncodingException Exception raised in failure
   *      situation
   */
  public HTMLStreamWriter(OutputStream out, String encoding)
    throws UnsupportedEncodingException {
    m_os = out;
    m_encoding = encoding;
    m_writer = new OutputStreamWriter(m_os, m_encoding);
  }

  /**
   * Will write all events onto <code>out</code> . Encoding does not matter.
   *
   * @param out the Writer to write on
   */
  public HTMLStreamWriter(Writer out) {
    m_os = null;
    m_encoding = null;
    m_writer = out;
  }

  public void characters(char[] ch, int start, int length)
    throws HTMLException {
    try {
      m_writer.write(ch, start, length);
    }
    catch (IOException ex) {
      throw new HTMLException(ex);
    }
  }

  public void endDocument()
    throws HTMLException {
    try {
      m_writer.close();
    }
    catch (IOException ex) {
      throw new HTMLException(ex);
    }
  }

  public void endElement(IHTMLElement element)
    throws HTMLException {
    writeElement(element);
  }

  public void startDocument()
    throws HTMLException { }

  public void startElement(IHTMLElementStart element)
    throws HTMLException {
    writeElement(element);
  }

  // -------------------- private --------------------------------------

  private void writeElement(IHTMLElement element)
    throws HTMLException {
    try {
      element.writeTo(m_writer);
    }
    catch (IOException ex) {
      throw new HTMLException(ex);
    }
  }
}
