/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.html;

import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;

/**
 * Default Implementation of {@link IHTMLFilter}. <p>
 *
 * Provides a default implmentation which is the null filter. It forwards all
 * events unchanged to its content handler. <p>
 *
 * Filters only interested in a subset of the events can extend this class to
 * ease their implementation effort. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HTMLFilterImpl.java,v 1.2 2001/12/12 17:19:55 sei Exp $
 */
public class HTMLFilterImpl implements IHTMLFilter, IHTMLContentHandler {

  private IHTMLReader m_reader;
  private IHTMLContentHandler m_content;

  /**
   * Empty filter with a parent reader installed.
   */
  public HTMLFilterImpl() { }

  /**
   * Filter which receives its events from the given reader.
   *
   * @param reader to get events from
   */
  public HTMLFilterImpl(IHTMLReader reader) {
    setParent(reader);
  }

  public IHTMLReader getParent() {
    return m_reader;
  }

  public void setParent(IHTMLReader reader) {
    m_reader = reader;
    if (m_content == null) {
      m_content = reader.getContentHandler();
    }
    reader.setContentHandler(this);
  }

  public IHTMLContentHandler getContentHandler() {
    return m_content;
  }

  public void setSource(InputStream input)
    throws HTMLException, IOException {
    ensureParent();
    m_reader.setSource(input);
  }

  public void setSource(InputStream input, String encoding)
    throws HTMLException, IOException {
    ensureParent();
    m_reader.setSource(input, encoding);
  }

  public void setSource(Reader input)
    throws HTMLException, IOException {
    ensureParent();
    m_reader.setSource(input);
  }

  public void setContentHandler(IHTMLContentHandler handler) {
    m_content = handler;
  }

  public String getEncoding()
    throws HTMLException, IOException {
    ensureParent();
    return m_reader.getEncoding();
  }

  public void parse()
    throws HTMLException, IOException {
    ensureParent();
    m_reader.parse();
  }

  public boolean parseNextEvent()
    throws HTMLException, IOException {
    ensureParent();
    return m_reader.parseNextEvent();
  }

  public void discard() {
    if (m_reader != null) {
      m_reader.discard();
    }
  }

  // ------------ IHTMLContentHandler ---------------------------------

  public void characters(char[] ch, int start, int length)
    throws HTMLException {
    if (m_content != null) {
      m_content.characters(ch, start, length);
    }
  }

  public void endDocument()
    throws HTMLException {
    if (m_content != null) {
      m_content.endDocument();
    }
  }

  public void endElement(IHTMLElement element)
    throws HTMLException {
    if (m_content != null) {
      m_content.endElement(element);
    }
  }

  public void startDocument()
    throws HTMLException {
    if (m_content != null) {
      m_content.startDocument();
    }
  }

  public void startElement(IHTMLElementStart element)
    throws HTMLException {
    if (m_content != null) {
      m_content.startElement(element);
    }
  }

  // ------------------------- private --------------------------------

  private void ensureParent()
    throws HTMLException {
    if (m_reader == null) {
      throw new HTMLException("no parent to read from");
    }
  }
}
