/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.html;

import com.sap.tc.logging.Location;

import java.io.IOException;
import java.io.Writer;

/**
 * Default Implementation. <p>
 *
 * Copyright (c) SAP AG 2001-2003
 *
 * @author stefan.eissing@greenbytes.de
 * @version $Id: HTMLElementImpl.java,v 1.14 2003/03/20 18:41:30 jre Exp $
 */
class HTMLElementImpl implements IHTMLElementStart {

  private final static Location log = Location.getLocation(com.sapportals.wcm.util.html.HTMLElementImpl.class);

  private final static int ATTR_INC = 10;
  private final static int ATTR_START = 10;

  private final static int BUFFER_INC = ATTR_INC * 128;

  private char[] m_buffer;
  private int m_length;

  private String m_name;
  private int m_nameStart;
  private int m_nameLength;

  private int m_attrCount;
  private int[] m_attrNStart;
  private int[] m_attrNLength;
  private int[] m_attrVStart;
  private int[] m_attrVLength;
  private String[] m_attrName;
  private String[] m_attrValue;

  private boolean m_isEmpty;
  private boolean m_isEndTag;

  public HTMLElementImpl() {
    m_buffer = new char[HTMLInput.INPUT_BUFFER_SIZE];
    m_attrNStart = new int[ATTR_START];
    m_attrNLength = new int[ATTR_START];
    m_attrVStart = new int[ATTR_START];
    m_attrVLength = new int[ATTR_START];
    m_attrName = new String[ATTR_START];
    m_attrValue = new String[ATTR_START];
  }

  public void parse(HTMLInput in)
    throws HTMLException, IOException {
    reset();
    int len = in.getLength();
    if (len == 2 && in.charAt(1) == '/') {
      m_isEndTag = true;
    }
    m_nameStart = len;

    if (!in.readNonWSUntil('>', '/')) {
      throw new HTMLException("cannot parse tag name: " + in.location());
    }
    m_nameLength = in.getLength() - m_nameStart;

    in.readWS();

    if (!m_isEndTag) {
      // No End Tag, might have attributes
      //
      loop :
      while (true) {
        int c = in.next();
//        if (log.beDebug()) {
//          log.debugT("parse element at: "+in.location());
//        }
        if (c == -1) {
          throw new HTMLException("element unexpected end of input: " + in.location());
        }
        else if (Character.isWhitespace((char)c)) {
          in.readWS();
        }
        else {
          switch (c) {
            case '/':
              c = in.next();
              if (c == '>') {
                m_isEmpty = true;
                break loop;
              }
              break;
            case '>':
              break loop;
            default:
              in.pushBack();
              if (parseAttribute(in)) {
                in.readWS();
              }
              else {
                if (log.beDebug()) {
                  log.debugT("parse(117)", "did not parse attribue at: " + in.location());
                }
              }
              break;
          }
        }
      }
    }
    else {
      // end tag, should not have attributes, but of course there
      // are HTML pages which do so. Sigh!
      //
      in.readUntil('>');
      int c = in.next();
      switch (c) {
        case '>':
          break;
        case -1:
          // unterminated end tag. report anyway
          if (log.beInfo()) {
            log.infoT("parse(137)", "unexpected end of input: " + in.location());
          }
          break;
        default:
          if (log.beInfo()) {
            log.infoT("parse(142)", "should not happen in end tag " + in.location());
          }
          break;
      }
    }

    len = in.getLength();
    ensureWriteable(len);
    in.copyTo(m_buffer, 0);
    m_length = len;
  }

  // ------------------- IHTMLElement -------------------------------------

  public String getName() {
    if (m_name == null) {
      if (m_nameStart > 0) {
        m_name = new String(m_buffer, m_nameStart, m_nameLength);
      }
      else {
        m_name = "";
      }
    }
    return m_name;
  }

  public int getNameLength() {
    return m_nameLength;
  }

  public boolean hasName(String name) {
    if (m_name != null) {
      return m_name.equalsIgnoreCase(name);
    }
    else {
      return equalsIgnoreCase(name, m_nameStart, m_nameLength);
    }
  }

  public int copyTo(char[] ch, int offset) {
    System.arraycopy(m_buffer, 0, ch, offset, m_length);
    return m_length;
  }

  public int writeTo(Writer writer)
    throws IOException {
    writer.write(m_buffer, 0, m_length);
    return m_length;
  }

  public int length() {
    return m_length;
  }

  // ------------------- IHTMLElementStart --------------------------------

  public boolean isEmpty() {
    return m_isEmpty;
  }

  public int getNumberOfAttributes() {
    return m_attrCount;
  }

  public void setValueOf(int index, String value) {
    checkIndex(index);
    m_attrValue[index] = value;

    int pos = m_attrVStart[index];
    int len = m_attrVLength[index];
    if (pos == 0) {
      // had no value before
      pos = m_attrNStart[index] + m_attrNLength[index];
      value = quote(value, "=");
      replace(pos, 0, value);
      m_attrVStart[index] = pos + 1;
      m_attrVLength[index] = value.length() - 3;
    }
    else if (m_buffer[pos - 1] == '\"') {
      if (value.indexOf('\"') >= 0) {
        m_buffer[pos - 1] = '\'';
        if (m_buffer[pos + len] == '\"') {
          m_buffer[pos + len] = '\'';
        }
      }
      replace(pos, len, value);
      m_attrVLength[index] = value.length();
    }
    else if (m_buffer[pos - 1] == '\'') {
      if (value.indexOf('\'') >= 0) {
        m_buffer[pos - 1] = '\"';
        if (m_buffer[pos + len] == '\"') {
          m_buffer[pos + len] = '\'';
        }
      }
      replace(pos, len, value);
      m_attrVLength[index] = value.length();
    }
    else {
      // was not quoted before, better quote now
      value = quote(value);
      replace(pos, len, value);
      m_attrVStart[index] = pos + 1;
      m_attrVLength[index] = value.length() - 2;
    }
  }

  public int getIndexOf(String attributeName) {
    for (int i = 0; i < m_attrCount; ++i) {
      String name = m_attrName[i];
      if (name != null) {
        if (name.equalsIgnoreCase(attributeName)) {
          return i;
        }
      }
      else {
        if (equalsIgnoreCase(attributeName, m_attrNStart[i], m_attrNLength[i])) {
          return i;
        }
      }
    }
    return -1;
  }

  public String getNameOf(int index) {
    checkIndex(index);
    String name = m_attrName[index];
    if (name == null) {
      name = new String(m_buffer, m_attrNStart[index], m_attrNLength[index]);
      m_attrName[index] = name;
    }
    return name;
  }

  public void removeAttribute(int index) {
    checkIndex(index);
    int pos = m_attrNStart[index];
    int len;
    if (m_attrVStart[index] > 0) {
      len = m_attrVStart[index] - pos + m_attrVLength[index];
      char c = m_buffer[pos + len];
      if (c == '\'' || c == '\"') {
        ++len;
      }
    }
    else {
      len = m_attrNLength[index];
    }

    for (int i = pos + len; i < m_length; ++i) {
      if (!Character.isWhitespace(m_buffer[i])) {
        break;
      }
      ++len;
    }

    replace(pos, len, "");

    if (index < m_attrCount - 1) {
      int csrc = index + 1;
      int clen = m_attrCount - csrc;
      System.arraycopy(m_attrNStart, csrc, m_attrNStart, index, clen);
      System.arraycopy(m_attrNLength, csrc, m_attrNLength, index, clen);
      System.arraycopy(m_attrVStart, csrc, m_attrVStart, index, clen);
      System.arraycopy(m_attrVLength, csrc, m_attrVLength, index, clen);
      System.arraycopy(m_attrName, csrc, m_attrName, index, clen);
      System.arraycopy(m_attrValue, csrc, m_attrValue, index, clen);
    }
    --m_attrCount;
  }

  public void addAttribute(String name, String value)
    throws HTMLException {
    if (getIndexOf(name) >= 0) {
      throw new HTMLException("addAttribute: attribute " + name + " already exists");
    }

    ensureAttributeSpace();

    int nlen = name.length();
    int vlen = value.length();
    StringBuffer sb = new StringBuffer(nlen + vlen + 5);
    sb.append(' ').append(name).append('=');
    if (value.indexOf('\"') >= 0) {
      sb.append('\'').append(value).append('\'');
    }
    else {
      sb.append('\"').append(value).append('\"');
    }

    int offset;
    if (m_attrCount > 0) {
      int i = m_attrCount - 1;
      if (m_attrVStart[i] > 0) {
        offset = m_attrVStart[i] + m_attrVLength[i];
        int c = m_buffer[offset];
        if (c == '\'' || c == '\"') {
          ++offset;
        }
      }
      else {
        offset = m_attrNStart[i] + m_attrNLength[i];
      }
    }
    else {
      offset = m_nameStart + m_nameLength;
    }

    replace(offset, 0, sb.toString());

    int index = m_attrCount++;
    m_attrNStart[index] = offset;
    m_attrNLength[index] = nlen;
    m_attrVStart[index] = offset + nlen + 2;
    m_attrVLength[index] = vlen;
    m_attrName[index] = name;
    m_attrValue[index] = value;
  }

  public String getValueOf(int index) {
    checkIndex(index);
    String value = m_attrValue[index];
    if (value == null) {
      value = new String(m_buffer, m_attrVStart[index], m_attrVLength[index]);
      m_attrValue[index] = value;
    }
    return value;
  }

  public String toString() {
    return new String(m_buffer, 0, m_length);
  }

  // ---------------------- private -------------------------------------

  private void reset() {
    m_isEmpty = false;
    m_length = 0;
    m_name = null;
    m_nameStart = m_nameLength = 0;
    m_isEndTag = false;
    for (int i = 0; i < m_attrCount; ++i) {
      m_attrName[i] = null;
      m_attrValue[i] = null;
    }
    m_attrCount = 0;
  }

  private void checkIndex(int index) {
    if (index < 0 || index >= m_attrCount) {
      throw new IndexOutOfBoundsException();
    }
  }

  private boolean equalsIgnoreCase(String s, int offset, int len) {
    if (s.length() != len) {
      return false;
    }

    for (int i = 0; i < len; ++i) {
      char c1 = m_buffer[offset + i];
      char c2 = s.charAt(i);
      if (c1 != c2) {
        if (Character.toUpperCase(c1) != Character.toUpperCase(c2)) {
          if (Character.toLowerCase(c1) != Character.toLowerCase(c2)) {
            return false;
          }
        }
      }
    }
    return true;
  }

  private boolean parseAttribute(HTMLInput in)
    throws HTMLException, IOException {
    // Try to parse an element attribute. They come in weird shape and sizes.
    // Be prepared to handle:
    // a | a= | a=b | =b  where b can be xxx, "xxx" or 'xxx' with x any char
    //
    ensureAttributeSpace();

    int nstart = in.getLength();
    boolean gotName = in.readNonWSUntil("=>/");
    int nlen = in.getLength() - nstart;
    int vstart = 0;
    int vlen = 0;
    int c = in.next();
    if (gotName || c == '=') {
      if (Character.isWhitespace((char)c)) {
        in.readWS();
        c = in.next();
      }

      if (c == '=') {
        vstart = in.getLength();
        c = in.next();
        if (Character.isWhitespace((char)c)) {
          in.readWS();
          vstart = in.getLength();
          c = in.next();
        }
        switch (c) {
          case -1:
            return false;
          case '\'':
          case '\"':
            ++vstart;
            in.readUntil((char)c);
            vlen = in.getLength() - vstart;
            in.next();// consume quote end
            break;
          case ' ':
          case '\t':
          case '\n':
          case '\r':
            in.pushBack();
            vlen = in.getLength() - vstart;
            break;
          default:
            in.readNonWSUntil('>');
            vlen = in.getLength() - vstart;
            break;
        }
      }
      else {
        in.pushBack();
      }
      m_attrNStart[m_attrCount] = nstart;
      m_attrNLength[m_attrCount] = nlen;
      m_attrVStart[m_attrCount] = vstart;
      m_attrVLength[m_attrCount] = vlen;
      m_attrName[m_attrCount] = null;
      m_attrValue[m_attrCount] = null;
      ++m_attrCount;
      return true;
    }
    else {
      if (c == -1) {
        throw new HTMLException("unexpected end of input: " + in.location());
      }
      in.pushBack();
      return false;
    }
  }

  private void replace(int pos, int len, String s) {
    int nlen = s.length();
    int offset = nlen - len;

    ensureWriteable(m_length + offset);
    if (offset == 0) {
    }
    else if (offset > 0) {
      System.arraycopy(m_buffer, pos, m_buffer, pos + offset, m_length - pos);
    }
    else if (offset < 0) {
      System.arraycopy(m_buffer, pos - offset, m_buffer, pos, m_length - pos + offset);
    }

    m_length += offset;
    if (nlen > 0) {
      s.getChars(0, nlen, m_buffer, pos);
    }
    for (int i = 0; i < m_attrCount; ++i) {
      int opos = m_attrNStart[i];
      if (opos > pos) {
        m_attrNStart[i] = opos + offset;
      }
      opos = m_attrVStart[i];
      if (opos > pos) {
        m_attrVStart[i] = opos + offset;
      }
    }
  }

  /**
   * Ensure that size bytes of writeable buffer space exists. When m_localBuffer
   * != m_buffer, then m_buffer is HTMLInput's own buffer and we have to make a
   * copy to local buffer space. When m_localBuffer == m_buffer, we have a
   * writeable local copy already and just need to check the size.
   *
   * @param size to garantuee
   */
  private void ensureWriteable(int size) {
    if (m_buffer.length < size) {
      int newsize = m_buffer.length + BUFFER_INC;
      if (newsize < size) {
        newsize = size;
      }
      char[] tmp = new char[newsize];
      if (m_length > 0) {
        System.arraycopy(m_buffer, 0, tmp, 0, m_length);
      }
      m_buffer = tmp;
    }
  }

  private void ensureAttributeSpace() {
    if (m_attrCount >= m_attrName.length) {
      int newSize = m_attrName.length + ATTR_INC;

      int[] attrNStart = new int[newSize];
      int[] attrNLength = new int[newSize];
      int[] attrVStart = new int[newSize];
      int[] attrVLength = new int[newSize];
      String[] attrName = new String[newSize];
      String[] attrValue = new String[newSize];
      for (int i = 0; i < m_attrCount; ++i) {
        System.arraycopy(m_attrNStart, 0, attrNStart, 0, m_attrCount);
        System.arraycopy(m_attrNLength, 0, attrNLength, 0, m_attrCount);
        System.arraycopy(m_attrVStart, 0, attrVStart, 0, m_attrCount);
        System.arraycopy(m_attrVLength, 0, attrVLength, 0, m_attrCount);
        System.arraycopy(m_attrName, 0, attrName, 0, m_attrCount);
        System.arraycopy(m_attrValue, 0, attrValue, 0, m_attrCount);
      }
      m_attrNStart = attrNStart;
      m_attrNLength = attrNLength;
      m_attrVStart = attrVStart;
      m_attrVLength = attrVLength;
      m_attrName = attrName;
      m_attrValue = attrValue;
    }
  }

  private String quote(String s) {
    return quote(s, "");
  }

  private String quote(String s, String prefix) {
    if (s.indexOf('\"') < 0) {
      return prefix + '\"' + s + '\"';
    }
    else {
      return prefix + '\'' + s + '\'';
    }
  }
}
