/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.factories;


import java.util.Date;

/**
 * Some tools for starting threads. <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author paul.goetz@sapportals.com
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/factories/ThreadUtils.java#5
 *      $
 */
public class ThreadUtils {

  static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.factories.ThreadUtils.class);


  // -------------------------
  // Inner Class InternalTimer ------------------------------------------------
  // -------------------------

  /**
   * TBD: Description of the class.
   */
  static class InternalFixedTimer
     implements Runnable {

    Runnable m_Task;
    Date m_Next;
    long m_Interval;

    protected InternalFixedTimer(Runnable task,
      Date start,
      long interval) {

      m_Task = task;
      m_Next = start;
      m_Interval = interval;

    }

    public void run() {

      Date now = new Date();
      long delay = 0;

      if (m_Next != null) {
        delay = m_Next.getTime() - now.getTime();
        if (delay <= 0) {
          m_Next = now;
        }
      }
      else {
        m_Next = now;
      }

      m_Next.setTime(m_Next.getTime() + m_Interval);

      try {
        if (delay > 0) {
          Thread.sleep(delay);
        }
      }
      catch (InterruptedException e) {
                    //$JL-EXC$
        g_Log.warningT("run(84)", "InternalTimer: caught interrupt for " + m_Task.getClass().getName());
        return;
      }

      while (true) {
        delay = m_Next.getTime() - System.currentTimeMillis();
        m_Next.setTime(m_Next.getTime() + m_Interval);
        m_Task.run();
        try {
          if (delay > 0) {
            Thread.sleep(delay);
          }
        }
        catch (InterruptedException e) {
            //$JL-EXC$          
          g_Log.warningT("run(98)", "InternalTimer: caught interrupt for " + m_Task.getClass().getName());
        }
      }

    }

  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  private static Thread doDaemonize(Runnable task,
    String name) {

    Thread thread = null;

    if (task == null) {
      return thread;
    }

    if (name == null) {
      thread = new Thread(getRootThreadGroup(), task);
    }
    else {
      thread = new Thread(getRootThreadGroup(), task, name);
    }

    if (thread == null) {
      return thread;
    }

    thread.setDaemon(true);
    thread.start();

    return thread;
  }


  private static Thread doDaemonizeFixedTimer(Runnable task,
    long delay,
    long interval,
    String name) {

    Date start = new Date();

    start.setTime(start.getTime() + delay);

    return doDaemonize(new InternalFixedTimer(task, start, interval), name);
  }


  private static Thread doDaemonizeFixedTimer(Runnable task,
    Date start,
    long interval,
    String name) {

    return doDaemonize(new InternalFixedTimer(task, start, interval), name);
  }


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  /**
   * Get the root thread group (the system thread group).
   *
   * @return the <code>ThreadGroup</code> at top level.
   */
  public static ThreadGroup getRootThreadGroup() {

    ThreadGroup currentGroup = Thread.currentThread().getThreadGroup();
    ThreadGroup parentGroup = currentGroup;

    while (parentGroup != null) {
      parentGroup = currentGroup.getParent();
      if (parentGroup != null) {
        currentGroup = parentGroup;
      }
    }

    return currentGroup;
  }


  /**
   * Create a new daemon thread for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonize(Runnable task) {

    return doDaemonize(task, null);
  }


  /**
   * Create a new daemon thread for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @param name a <code>String</code> with the name of the thread.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonize(Runnable task,
    String name) {

    return doDaemonize(task, name);
  }


  /**
   * Create a new daemon thread with a fixed timer for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @param start a <code>Date</code> with the start date for the timer.
   * @param interval a <code>long</code> with the delay between each call to the
   *      task in milliseconds.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonizeFixedTimer(Runnable task,
    Date start,
    long interval) {

    return doDaemonizeFixedTimer(task, start, interval, null);
  }


  /**
   * Create a new daemon thread with a fixed timer for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @param start a <code>Date</code> with the start date for the timer.
   * @param interval a <code>long</code> with the delay between each call to the
   *      task in milliseconds.
   * @param name a <code>String</code> with the name of the thread.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonizeFixedTimer(Runnable task,
    Date start,
    long interval,
    String name) {

    return doDaemonizeFixedTimer(task, start, interval, name);
  }


  /**
   * Create a new daemon thread with a fixed timer for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @param delay a <code>long</code> with the delay for the first call to the
   *      task in milliseconds.
   * @param interval a <code>long</code> with the delay between each call to the
   *      task in milliseconds.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonizeFixedTimer(Runnable task,
    long delay,
    long interval) {

    return doDaemonizeFixedTimer(task, delay, interval, null);
  }


  /**
   * Create a new daemon thread with a fixed timer for a <code>Runnable</code> .
   *
   * @param task the <code>Runnable</code> to execute by the task.
   * @param delay a <code>long</code> with the delay for the first call to the
   *      task in milliseconds.
   * @param interval a <code>long</code> with the delay between each call to the
   *      task in milliseconds.
   * @param name a <code>String</code> with the name of the thread.
   * @return the <code>Thread</code> , marked as daemon and already started.
   */
  public static Thread daemonizeFixedTimer(Runnable task,
    long delay,
    long interval,
    String name) {

    return doDaemonizeFixedTimer(task, delay, interval, name);
  }

}
