/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.factories;

import java.lang.reflect.Constructor;
import java.lang.reflect.Method;
import java.util.Enumeration;
import java.util.Properties;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;

/**
 * A wrapper for reflection methods with an <code>WcmException</code> . <p>
 *
 * Copyright (c) SAP AG 2001-2002
 *
 * @author paul.goetz@sapportals.com
 * @version $Id:$
 */
public class ReflectionUtils {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * Property tag for the classname parameter in the configuration data.
   */
  public final static String CONFIG_CLASSNAME_TAG = "class";

  // ------------------------------------------------------------------------

  /**
   * Default log name.
   */
  private final static String LOGNAME = "ReflectionUtils";

  /**
   * Default name for the <code>getInstance()</code> method.
   */
  private final static String DEFAULT_INSTANCE_METHOD_NAME = "getInstance";

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.factories.ReflectionUtils.class);


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  // ------------------------------------------------------------------------
  /**
   * Get the default name for the <code>getInstance</code> method.
   *
   * @return a <code>String</code> with the default name of the <code>
   *      getInstance()</code> method.
   */
  public static String getDefaultGetInstanceMethodName() {

    return DEFAULT_INSTANCE_METHOD_NAME;
  }


  // ------------------------------------------------------------------------
  /**
   * Load the class for a given name.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param className a <code>String</code> with the name of the class to load.
   * @return the <code>Class</code> object loaded.
   * @exception WcmException Exception raised in failure situation
   * @deprecated as of NW04. Use {@link com.sapportals.wcm.crt.CrtClassLoaderRegistry}
   *      instead.
   * @throws WcmException if the class cannot be loaded.
   */
  public static Class loadClass(String logName,
    String className)
    throws WcmException {

    if (logName == null) {
      logName = LOGNAME;
    }

    if (className == null) {
      throw new WcmException(logName + ": classname <null> not allowed");
    }

    try {

      Class classObject = CrtClassLoaderRegistry.forName(className);
      if (g_Log.beInfo()) {
        g_Log.infoT("loadClass(115)", logName + ": loaded class " + className);
      }
      return classObject;
    }
    catch (ClassNotFoundException e) {
            //$JL-EXC$      
      throw new WcmException(logName + ": class " + className + " not found");      
    }
    catch (Exception e) {
      throw new WcmException(logName + ": exception loading class " + className
         + ":" + e.getMessage(), e);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Get a classes' constructor.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param classObject the <code>Class</code> object of the class to get the
   *      constructor for.
   * @param parameterTypes an array of <code>Class</code> es defining the
   *      constructor's signature.
   * @return a <code>Constructor</code> object with the specified constructor.
   * @throws WcmException if the constructor cannot be loaded.
   */
  public static Constructor loadConstructor(String logName,
    Class classObject,
    Class[] parameterTypes)
    throws WcmException {

    if (logName == null) {
      logName = LOGNAME;
    }

    if (classObject == null) {
      throw new WcmException(logName + ": class <null> not allowed");
    }

    try {

      return classObject.getConstructor(parameterTypes);
    }
    catch (Exception e) {
      throw new WcmException(logName + ": execption loading constructor for "
         + classObject.getName() + ": " + e.getMessage(), e);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Get a classes' method.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param classObject the <code>Class</code> object of the class to get the
   *      method for.
   * @param methodName a <code>String</code> s with the name of the method.
   * @param parameterTypes an array of <code>Class</code> es defining the
   *      method's signature.
   * @return a <code>Method</code> object with the specified method.
   * @throws WcmException if the method cannot be loaded.
   */
  public static Method loadMethod(String logName,
    Class classObject,
    String methodName,
    Class[] parameterTypes)
    throws WcmException {

    if (logName == null) {
      logName = LOGNAME;
    }

    if (classObject == null) {
      throw new WcmException(logName + ": class <null> not allowed");
    }

    if (methodName == null) {
      throw new WcmException(logName + ": method <null> not allowed");
    }

    try {

      return classObject.getMethod(methodName, parameterTypes);
    }
    catch (Exception e) {
      throw new WcmException(logName + ": execption loading method " + methodName
         + " for " + classObject.getName() + ": "
         + e.getMessage(), e);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Call an object's / classes' method.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param object the <code>Object</code> to call the method for or <code>null
   *      </code> if it's a static method.
   * @param method the <code>Method</code> to call.
   * @param parameters an array of <code>Object</code> s with the parameters to
   *      pass to the method.
   * @return an <code>Object</code> with the result of the method or <code>null
   *      </code> if void.
   * @throws WcmException if the method cannot be called.
   */
  public static Object callMethod(String logName,
    Object object,
    Method method,
    Object[] parameters)
    throws WcmException {

    if (logName == null) {
      logName = LOGNAME;
    }

    if (method == null) {
      throw new WcmException(logName + ": method <null> not allowed");
    }

    try {

      return method.invoke(object, parameters);
    }
    catch (Exception e) {
      throw new WcmException(logName + ": exception calling " + method.getName()
         + " for " + method.getDeclaringClass().getName()
         + ": " + e.getMessage(), e);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Call a classes' static method.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param method the <code>Method</code> to call.
   * @param parameters an array of <code>Object</code> s with the parameters to
   *      pass to the method.
   * @return an <code>Object</code> with the result of the method or <code>null
   *      </code> if void.
   * @throws WcmException if the method cannot be called.
   */
  public static Object callStaticMethod(String logName,
    Method method,
    Object[] parameters)
    throws WcmException {

    return callMethod(logName, null, method, parameters);
  }


  // ------------------------------------------------------------------------
  /**
   * Create a new object from a class.
   *
   * @param logName a <code>String</code> with a short name for logging.
   * @param constructor the <code>Constructor</code> to use.
   * @param parameters an array of <code>Object</code> s with the parameters to
   *      pass to the constructor.
   * @return an <code>Object</code> with the newly created object.
   * @exception WcmException Exception raised in failure situation
   * @throws WcmException if the object cannot be created.
   */
  public static Object newObject(String logName,
    Constructor constructor,
    Object[] parameters)
    throws WcmException {

    if (logName == null) {
      logName = LOGNAME;
    }

    if (constructor == null) {
      throw new WcmException(logName + ": constructor <null> not allowed");
    }

    try {

      return constructor.newInstance(parameters);
    }
    catch (Exception e) {
      throw new WcmException(logName + ": unable to create a new instance with "
         + constructor.getDeclaringClass().getName() + ": "
         + e.getMessage(), e);
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Get a part of the configuration for specific base-prefix from the
   * properties (extract only those properties prefixed by <i>base</i> ).
   *
   * @param base a <code>String</code> with the prefix for the desired
   *      properties.
   * @param properties TBD: Description of the incoming method parameter
   * @return the <code>Properties</code> from configuration for the given <i>
   *      base</i> <code>.</code> -prefix. The keys of the returned properties
   *      are <i>without</i> the base-prefix.
   */
  public static Properties getSubProperties(Properties properties,
    String base) {

    Properties subProperties = new Properties();

    if (properties == null) {
      return subProperties;
    }

    if (base == null) {
      base = "";
    }

    String prefix = base + ".";
    int prefixLength = prefix.length();

    Enumeration propertyKeys = properties.propertyNames();
    while (propertyKeys.hasMoreElements()) {
      String key = (String)propertyKeys.nextElement();
      if (key.startsWith(prefix)) {
        subProperties.put(key.substring(prefixLength), properties.get(key));
      }
      else if (key.equals(base)) {
        subProperties.put("", properties.get(key));
      }
    }

    return subProperties;
  }

}
