/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.factories;

import com.sap.tc.logging.Location;

import com.sapportals.config.fwk.*;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.util.config.*;
import com.sapportals.wcm.util.string.StrUtil;

import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Properties;

/**
 * A factory which produces configurable objects. <p>
 *
 * The classes to load are specified by the classname from the configuration
 * properties.<br>
 * The objects classes are specified by a commata separated list in the
 * configuration, e.g.:<br>
 * <i>factory</i> <code>.list = </code> <i>id</i> <code>, </code> <i>id</i>
 * <code>, </code> ...<br>
 * For each <i>id</i> at least a classname has to be specified and an optional
 * singleton flag, e.g.:<br>
 * <i>factory</i> <code>.</code> <i>id</i> <code>.class = </code> <i>classname
 * </i> <br>
 * <i>factory</i> <code>.</code> <i>id</i> <code>.singleton = true</code> <br>
 * Additional parameters in the form<br>
 * <i>factory</i> <code>.</code> <i>id</i> <code>.</code> <i>xxx</i> <code> =
 * </code> <i>...</i> <br>
 * can be passed to the object's constructor. <p>
 *
 *
 *
 * @todo Proper handling of configuration changes (implementing a configuration
 *      listener and a reload mechanism).
 */
public abstract class AbstractObjectFactory {

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  /**
   * A short name for logging.
   */
  protected String m_LogName = AbstractObjectFactory.class.getName();

  /**
   * The id of the factory.
   */
  protected String m_ID = "";

  /**
   * The properties (with prefix <i>factory</i> -id) from the config file.
   */
  protected Properties m_Properties = null;

  /**
   * The map of <code>ObjectClassEntry<code>s, indexed by object class id.
   */
  protected HashMap m_ObjectMap = null;

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * Configuration tag for the object id list.
   */
  protected final static String CONFIG_OBJECTLIST_TAG = "list";

  /**
   * Configuration value for the object id list separator.
   */
  protected final static String CONFIG_OBJECTLIST_SEPARATOR = ",";

  /**
   * Configuration tag for the singleton flag.
   */
  protected final static String CONFIG_SINGLETON_TAG = "singleton";

  /**
   * Default configuration value for the singleton flag.
   */
  protected final static String CONFIG_SINGLETON_DEFAULT = "false";

  /**
   * Code for a singleton <code>AbstractObjectFactory</code> (singleton per id).
   */
  protected final static int SINGLETON = 0;

  /**
   * Code for a multi-instance <code>AbstractObjectFactory</code> .
   */
  protected final static int MULTI = 1;

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static Location g_Log = Location.getLocation(com.sapportals.wcm.util.factories.AbstractObjectFactory.class);

  /**
   * log name.
   */
  static String LOGNAME = "AbstractObjectFactory";

  /**
   * A table for the singleton instances, using the ids as keys.
   */
  private static HashMap g_Instances = new HashMap();

  /**
   * The default parameter types for the factory constructor.
   */
  private static Class[] g_FactoryConstructorParameterTypes = new Class[]{String.class};

  /**
   * The default parameter types for the object class constructor.
   */
  private static Class[] g_ObjectConstructorParameterTypes = new Class[]{String.class, Properties.class};


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create an <code>AbstractObjectFactory</code> for a given <i>factory-id</i>
   * . Should call <code>super(</code> <i> logName</i> <code>,</code> <i>id</i>
   * <code>)</code> . <i>Note</i> : This constructor has to be public (although
   * it should be protected), to allow the abstract factory the creation of a
   * new derived factories via reflection.
   *
   * @param id a <code>String</code> with the unique <i>id</i> of the factory
   *      used as a prefix for the configuration parameters.
   * @exception WcmException Exception raised in failure situation
   */
  public AbstractObjectFactory(String id)
    throws WcmException {

    this(id, id);

  }


  // ------------------------------------------------------------------------
  /**
   * Create an <code>AbstractObjectFactory</code> for a given <i>factory-id</i>
   * . <i>Note</i> : This constructor has to be public (although it should be
   * protected), to allow the <code>newFactory()</code> -method the creation of
   * new factories.
   *
   * @param logName a <code>String</code> with the factory's short name (for
   *      logging).
   * @param id a <code>String</code> with the unique <i>id</i> of the factory
   *      used as a prefix for the configuration parameters.
   * @exception WcmException Exception raised in failure situation
   */
  protected AbstractObjectFactory(String logName,
    String id)
    throws WcmException {

    if (logName != null) {
      m_LogName = logName;
    }

    if (id != null) {
      m_ID = id;
    }

  }


  // ------------------------------------------------------------------------
  /**
   * Get a list with one instance for each available object class.
   *
   * @return a <code>Collection</code> of <code>Object</code> s, which may be
   *      empty if no object class exists.
   * @throws WcmException if an error occured while creating the
   *      list.
   */
  public Collection getAllObjectInstances()
    throws WcmException {

    LinkedList objects = new LinkedList();

    Collection entries = m_ObjectMap.values();
    Iterator iterator = entries.iterator();
    while (iterator.hasNext()) {
      ObjectClassEntry entry = (ObjectClassEntry)iterator.next();
      objects.add(entry.getObjectInstance());
    }

    return objects;
  }


  // ------------------------------------------------------------------------
  /**
   * Get the parameter types for the object classes' constructor.
   *
   * @param id a <code>String</code> with the id of the object class to get the
   *      parameter types for.
   * @return The ObjectConstructorParameterTypes value
   * @eturn an array of <code>Class</code> es with the parameter types for the
   *      object classes' constructor.
   */
  protected Class[] getObjectConstructorParameterTypes(String id) {

    return g_ObjectConstructorParameterTypes;
  }


  // ------------------------------------------------------------------------
  /**
   * Get the actual parameter for the object's constructor.
   *
   * @param id a <code>String</code> with the id of the object to get the
   *      parameter for.
   * @param properties
   * @return The ObjectConstructorParameters value
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the incoming method parameter
   * @eturn an array of <code>Object</code> s with the actual parameters for the
   *      object's constructor.
   */
  protected Object[] getObjectConstructorParameters(String id,
    Properties properties)
    throws WcmException {

    return new Object[]{id, properties};
  }


  // ------------------------------------------------------------------------
  /**
   * Create a new instance from the <code>ObjectClassEntry</code> for the given
   * id.
   *
   * @param id a <code>String</code> with the id of the object class.
   * @return an <code>Object</code> with an instance of the requested object
   *      class specified by the id or <code>null</code> if the id is not found.
   * @throws WcmException if an object cannot be created.
   */
  protected Object getObjectInstance(String id)
    throws WcmException {

    ObjectClassEntry entry = (ObjectClassEntry)m_ObjectMap.get(id);

    if (entry == null) {
      return null;
    }

    return entry.getObjectInstance();
  }

  // ------------------------------------------------------------------------
  /**
   * Load the object classes into a map with pairs of <i>id</i> , <code>
   * ObjectEntry</code> s.
   *
   * @param list a <code>String</code> with a list of id strings for the object
   *      classes to load.
   * @param listSeparator a <code>String</code> with the separators for the list
   *      string.
   * @param properties the <code>Properties</code> to use for searching the
   *      object classes class name and to pass to those object classes.
   * @return TBD: Description
   * @throws WcmException if an error occured.
   */
  protected HashMap loadObjectMap(String list,
    String listSeparator,
    Properties properties)
    throws WcmException {

    String[] ids = StrUtil.extractStringComponentsToArray(list, listSeparator, true);
    Properties[] objectProperties = new Properties[ids.length];
    for (int i = 0; i < ids.length; i++) {
      objectProperties[i] = ReflectionUtils.getSubProperties(properties, ids[i]);
    }
    return loadObjectMap(ids, objectProperties);
  }

  /**
   * @param configurables
   * @return
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  protected HashMap loadObjectMap(IConfigurable[] configurables)
    throws WcmException {

    Properties[] objectProperties = new Properties[configurables.length];
    String[] ids = new String[configurables.length];
    for (int i = 0; i < configurables.length; i++) {
      ids[i] = configurables[i].getIdValue();
      objectProperties[i] = ConfigCrutch.getConfigurableProperties(configurables[i]);
    }
    return loadObjectMap(ids, objectProperties);
  }

  // ------------------------------------------------------------------------
  /**
   * Create a new object class entry for the object class list.
   *
   * @param id a <code>String</code> with the id of the object class.
   * @param constructor the <code>Constructor</code> for the object class.
   * @param properties the <code>Properties</code> for the object class.
   * @param singletonFlag a <code>boolean</code> <code>true</code> if this
   *      object class should have only one instance, <code>false</code> if not.
   * @return
   * @todo: Description of the outgoing return value
   * @throws <code> WcmException</code> if an error occured.
   */
  /**
   * Create a new object class entry for the object class list.
   *
   * Create a new object class entry for the object class list. Create a new
   * object class entry for the object class list. Create a new object class
   * entry for the object class list. Create a new object class entry for the
   * object class list. Create a new object class entry for the object class
   * list. Create a new object class entry for the object class list. Create a
   * new object class entry for the object class list. Create a new object class
   * entry for the object class list. create a new object class entry.
   *
   * @param id a <code>String</code> with the id of the object class.
   * @param constructor the <code>Constructor</code> for the object class.
   * @param properties the <code>Properties</code> for the object class.
   * @param singletonFlag a <code>boolean</code> <code>true</code> if this
   *      object class should have only one instance, <code>false</code> if not.
   * @return TDB: Description of the outgoing return value
   */
  protected ObjectClassEntry newObjectListEntry(String id,
    Constructor constructor,
    Properties properties,
    boolean singletonFlag) {

    return new ObjectClassEntry(id, this, constructor, properties, singletonFlag);
  }

  /**
   * @param ids
   * @param properties
   * @return
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  private HashMap loadObjectMap(String ids[], Properties[] properties)
    throws WcmException {

    m_ObjectMap = new HashMap();

    for (int i = 0; i < properties.length; i++) {
      String classname = properties[i].getProperty(ReflectionUtils.CONFIG_CLASSNAME_TAG);
      if (classname == null) {
        g_Log.warningT("loadObjectMap(399)", m_LogName + ": caught exception while loading " + ids[i] + "." + ReflectionUtils.CONFIG_CLASSNAME_TAG);
      }
      else {
        try {
          Class classobject = CrtClassLoaderRegistry.forName(classname);
          Constructor constructor = ReflectionUtils.loadConstructor(ids[i], classobject, getObjectConstructorParameterTypes(ids[i]));
          boolean singletonFlag = properties[i].getProperty(CONFIG_SINGLETON_TAG, CONFIG_SINGLETON_DEFAULT).equalsIgnoreCase("true");
          ObjectClassEntry entry = newObjectListEntry(ids[i], constructor, properties[i], singletonFlag);
          m_ObjectMap.put(ids[i], entry);
        }
        catch (Exception e) {
            //$JL-EXC$          
          g_Log.warningT("loadObjectMap(410)", "onInitialize: caught exception creating entry for " + ids[i]
             + " class " + classname + ": " + e.getMessage());
        }
      }
    }

    return m_ObjectMap;
  }


  // -----------------
  // Protected Methods --------------------------------------------------------
  // -----------------

  // ------------------------------------------------------------------------
  /**
   * Get an initialized instance of a factory for the given <i>id</i> . This
   * method will handle the <code>type</code> code for the singleton/multi
   * instance behavior.
   *
   * @param id a <code>String</code> with the <i>id</i> for the factory.
   * @param classObject the <code>Class</code> of the (derived) factory.
   * @param instanceType an <code>int</code> with the instance type (<code>
   *      SINGLETON</code> or <code>MULTI</code> ).
   * @return an <code>AbstractObjectFactory</code> with an instance of the
   *      (derived) factory as.
   * @throws WcmException if the factory cannot be initialized.
   */
  protected static AbstractObjectFactory getFactoryInstance(String id,
    Class classObject,
    int instanceType)
    throws WcmException {

    if (id == null) {
      throw new WcmException("AbstractObjectFactory: <null> id not valid");
    }

    if (classObject == null) {
      return null;
    }

    AbstractObjectFactory factoryInstance = null;

    switch (instanceType) {

      case SINGLETON:
      {
        synchronized (g_Instances) {
          factoryInstance = (AbstractObjectFactory)g_Instances.get(id);
          if (factoryInstance == null) {
            factoryInstance = newFactory(id, classObject);
            g_Instances.put(id, factoryInstance);
          }
        }
      }
        break;
      case MULTI:
      {
        factoryInstance = newFactory(id, classObject);
      }
        break;
      default:
      {
        throw new WcmException("AbstractObjectFactory: instance type "
           + instanceType + " not valid for " + id);
      }

    }

    return factoryInstance;
  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------


  /**
   * @param id
   * @param classObject
   * @return
   * @exception WcmException Exception raised in failure situation
   * @todo: Description of the Method.
   * @todo: Description of the incoming method parameter
   * @todo: Description of the incoming method parameter
   * @todo: Description of the outgoing return value
   */
  private static AbstractObjectFactory newFactory(String id,
    Class classObject)
    throws WcmException {

    Constructor factoryConstructor = ReflectionUtils.loadConstructor(LOGNAME, classObject, g_FactoryConstructorParameterTypes);

    Object[] parameters = new Object[]{id};
    AbstractObjectFactory instance = (AbstractObjectFactory)ReflectionUtils.newObject(LOGNAME, factoryConstructor, parameters);

    return instance;
  }


  // -----------------------
  // Inner Class ObjectEntry --------------------------------------------------
  // -----------------------

  /**
   * An object class entry represents the information about the several objects
   * to be created by the factory.
   */
  protected class ObjectClassEntry {

    /**
     * the object classes' id.
     */
    protected String m_ID;

    /**
     * the object classes' factory.
     */
    protected AbstractObjectFactory m_Factory;

    /**
     * the constructor to the object class.
     */
    protected Constructor m_Constructor;

    /**
     * the properties for this object class.
     */
    protected Properties m_Properties;

    /**
     * flag, if this object class should be treated as a singleton.
     */
    protected boolean m_IsSingleton;

    /**
     * the single instance, if it's a singleton.
     */
    protected Object m_Instance;


    /**
     * create a new object class entry.
     *
     * @param id
     * @param factory
     * @param constructor
     * @param properties
     * @param singletonFlag
     * @todo: Description of the incoming method parameter
     * @todo: Description of the incoming method parameter
     * @todo: Description of the incoming method parameter
     * @todo: Description of the incoming method parameter
     * @todo: Description of the incoming method parameter
     */
    protected ObjectClassEntry(String id,
      AbstractObjectFactory factory,
      Constructor constructor,
      Properties properties,
      boolean singletonFlag) {

      m_Factory = factory;
      m_ID = id;
      m_Constructor = constructor;
      m_Properties = properties;
      m_IsSingleton = singletonFlag;

      m_Instance = null;

    }

    /**
     * get an instance of an object from this object class.
     *
     * @return The ObjectInstance value
     * @exception WcmException Exception raised in failure situation
     */
    public Object getObjectInstance()
      throws WcmException {

      if (m_IsSingleton) {
        synchronized (this) {
          if (m_Instance == null) {
            m_Instance = newObject();
          }
        }
        return m_Instance;
      }
      else {
        return newObject();
      }

    }

    /**
     * equality operator.
     *
     * @param object
     * @return
     * @todo: Description of the incoming method parameter
     * @todo: Description of the outgoing return value
     */
    public boolean equals(Object object) {

      if (!(object instanceof ObjectClassEntry)) {
        return false;
      }

      ObjectClassEntry other = (ObjectClassEntry)object;

      return (m_ID.equals(other.m_ID));
    }

    /**
     * create a new object from this object class.
     *
     * @return
     * @exception WcmException Exception raised in failure situation
     * @todo: Description of the outgoing return value
     */
    private Object newObject()
      throws WcmException {

      return ReflectionUtils.newObject(
        LOGNAME,
        m_Constructor,
        m_Factory.getObjectConstructorParameters(m_ID, m_Properties)
        );
    }

  }

}
