/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.controlstatus;
import java.util.*;

import com.sapportals.config.fwk.*;
import com.sapportals.wcm.util.cache.*;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.uuid.UUID;

/**
 * @author Andreas Heix This service manages the lifetime and persistence of
 *      <code>IControlStatus</code> instances. It is a singleton and might use
 *      other utilities such as as the <code>CacheFactory</code> to manage
 *      persistency. Depending on the usage of this service and the clean-up
 *      intervals, elements might be deleted before their calling application is
 *      deleted. This will lead to a <code>null</code> as return value in the
 *      <code>getControlStatus</code> method. Calling classes must be aware of
 *      this. If no persistence service is available, the service will store
 *      elements in an internal <code>HashMap</code> . This mode is unsafe, as
 *      only a rudimentary persistence-management is available: If the map's
 *      size exceeds the size specified in the classes <code>MAX_SIZE</code>
 *      parameter, all entries in this map will be deleted to avoid memory
 *      problems.
 */
public class ControlStatusService {

  private static ControlStatusService instance = null;
  private ICache cache = null;
  private HashMap map = new HashMap();
  public final static String CSS = "ControlStatusService";
  public final static String CACHE = "cache";
  private com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.controlstatus.ControlStatusService.class);
  private boolean protecedMode = false;
  public final static int MAX_SIZE = 2000;

  //control status service
  private final static String CFG_PLUGIN_CM_UTILITIES_CSS = "/cm/utilities/control_status";


  private ControlStatusService() {
    this.initialize();
    //that's all we have to do
  }

  /**
   * returns the singleton instance of this service. If this method is called
   * for the first time, the service will read its configuration to initialize
   * its persistence layer, using the <code>CacheFactory</code> . If no
   * configuration can be found or the configuration is invalid, the service
   * will use a simple <code>HashMap</code> to store <code>IControlStatus</code>
   * elements. This mode is unsafe and error messages will be provided in the
   * log files.
   *
   * @return the singleton instance of this service
   */
  public static ControlStatusService getInstance() {
		return ServiceHolder.instance;
  }

  /**
   * returns the <code>IControlStatus</code> that has been saved with this key.
   *
   * @param key TBD: Description of the incoming method parameter
   * @return an <code>IControlStatus</code> object or <code>null</code> .
   * @throws ControlStatusException if an exception in the persistence layer
   *      occurs. The original exception is nested within the thrown exception.
   */
  public IControlStatus getControlStatus(String key)
    throws ControlStatusException {
    try {
    	IControlStatus result = null;
      if (key != null) {
        if (this.protecedMode) {
          result = (IControlStatus)this.map.get(key);
        }
        else {
          ICacheEntry entry = this.cache.getEntry(key);
          if (entry != null) {
            result = (IControlStatus)entry.getObject();
          }
        }
      }
      if (log.beDebug()) {
      	log.debugT("Retrieve entry <" + result + " > for the given key <" + key +">");
      }
      return result;
    }
    catch (Exception e) {
      throw new ControlStatusException(e);
    }
  }

	public List getCacheEntries(Locale locale) {
		List result = new ArrayList();

		if (this.protecedMode) {
			log.errorT("Service is running in protected mode - no cache overview available");
		} else {
			try {
	      CacheEntryList list = this.cache.elements();
	      for (CacheEntryListIterator iter = list.iterator(); iter.hasNext();) {
	      	ICacheEntry entry = iter.next();
	      	result.add(new Field(new FieldIdentifier(entry.getKey()), entry.getObject().toString(), new MetaInfoMap(entry.getKey()), true, null, locale));
	      }
	    } catch (CacheException e) {
	      log.errorT("Could not read out cache entries " + LoggingFormatter.extractCallstack(e));
	    }
		} 
    return result;
	}

  /**
   * adds the given object and returns the key to retrieve the object again
   * through the <code>getControlStatus</code> method. This method is
   * synchronized because execution of this method might change the structure of
   * the underlying persistence. The returned key is also added as an attribute
   * to the status.
   *
   * @param status the object to store
   * @return the key to retrieve the given object at a later time
   * @throws ControlStatusException if an exception in the persistence layer
   *      occurs. The original exception is nested within the thrown exception.
   */
  public synchronized String addControlStatus(IControlStatus status)
    throws ControlStatusException {
    try {
      String key = new UUID().toString();
      status.setID(key);
      if (this.protecedMode) {
        this.log.errorT("addControlStatus(124)", "No cache available for Service - adding entry to internal HashMap");
        if (this.map.size() > MAX_SIZE) {
          this.map.clear();
          this.log.errorT("addControlStatus(127)", "The internal persistence was becoming too big - deleting all entries");
        }
        this.map.put(key, status);
      }
      else {
        this.cache.addEntry(key, status);
      }
      if (log.beDebug()) {
      	log.debugT("Added status <" + status + "> with key <" + key +">");
      }
      return key;
    }
    catch (Exception e) {
      throw new ControlStatusException(e);
    }
  }

  /**
   * deletes an <code>IControlStatus</code> object with the given key from the
   * service's persistence.
   *
   * @param key the key of the object to delete
   * @return flag stating if a document was found and consequently deleted (
   *      <code>true</code> ) or if no element could be found (<code>false
   *      </code>).
   * @throws ControlStatusException if an exception in the persistence layer
   *      occurs. The original exception is nested within the thrown exception.
   *      If the exception is thrown, it is unknown, if the object has been
   *      deleted or not.
   */
  public synchronized boolean deleteControlStatus(String key)
    throws ControlStatusException {
    try {
    	boolean result = false;
      if (this.protecedMode) {
        result = (this.map.remove(key) != null);
      }
      else {
        result = this.cache.removeEntry(key);
      }
      if (log.beDebug()) {
      	if (result) {
					log.debugT("Deleted entry with key <" + key +">");
      	} else {
					log.debugT("Did not delete an entry, because no entry was found for given key <" + key +">");
      	}
      	
      }
      return result;
    }
    catch (Exception e) {
      throw new ControlStatusException(e);
    }
  }


  /**
   * reads out configuration
   */
  private void initialize() {
    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_CSS);
      
      IConfigurable config = plugin.getConfigurable(CSS);
      if (config == null) {
        this.log.errorT("initialize(179)", "No configuration available for Service - running in protected mode with less stability and performance");
        this.protecedMode = true;
        return;
      }
      String cacheID = config.getPropertyValue(CACHE);
      if (cacheID == null) {
        this.log.errorT("initialize(185)", "No cache configured for Service - running in protected mode with less stability and performance");
        this.protecedMode = true;
        return;
      }
      this.cache = CacheFactory.getInstance().getCache(cacheID);
      if (this.cache == null) {
        this.log.errorT("initialize(191)", "No cache with ID < " + cacheID + " > configured in CacheService - running in protected mode with less stability and performance");
        this.protecedMode = true;
        return;
      }
      this.log.infoT("initialize(195)", "Service properly instantiated - using cache < " + cacheID + ">");
    }
    catch (Exception e) {
      this.protecedMode = true;
      this.log.errorT("initialize(199)", "Exception when trying to instantiate cache - service is running in proteced mode" + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
    }
  }

  private static class ServiceHolder {
    static final ControlStatusService instance = new ControlStatusService();
  }

}
