/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.config;

import java.io.*;
import java.util.*;

//=============================================================================
/**
 * PropertiesFile is a helper class to read and write a properties file
 * hirarchically. This class might be replaced by a visitor/builder patern on
 * ConfigCollection. Copyright (c) SAP AG 2001-2002
 *
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/config/PropertiesFile.java#6
 *      $
 */
//=============================================================================
public class PropertiesFile {
  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.config.PropertiesFile.class);

  String m_filename;

  //---------------------------------------------------------------------------
  /**
   * Create a new PropertiesFile helper.
   *
   * @param filename The filename of the properties file.
   */
  //---------------------------------------------------------------------------
  public PropertiesFile(String filename) {//---------------------------------------------------------------------------

    m_filename = filename;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the filename of the configfile.
   *
   * @return The filename of the configfile
   */
  //---------------------------------------------------------------------------
  public String getFilename() {//---------------------------------------------------------------------------

    return m_filename;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Read the configuration file.
   *
   * @return TBD: Description of the outgoing return value
   */
  //---------------------------------------------------------------------------
  public ConfigCollection read() {//---------------------------------------------------------------------------

    Properties properties = new Properties();
    FileInputStream fis = null;

    try {
      fis = new FileInputStream(m_filename);

      properties.load(fis);
    }
    catch (IOException e) {
            //$JL-EXC$      
      log.errorT("read(83)", "Caught IOException in read(): " + e);
    }
    finally {
      try {
        if (fis != null) {
          fis.close();
        }
      }
      catch (Exception e) {
            //$JL-EXC$        
        com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
      }
    }

    ConfigCollection root = new ConfigCollection(null);
    Iterator iter = properties.keySet().iterator();

    while (iter.hasNext()) {
      String key = iter.next().toString();
      String value = properties.getProperty(key);

      root.putProperty(key, value);
    }

    return root;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Write a new configuration to the configuration file.
   *
   * @param root TBD: Description of the incoming method parameter
   */
  //---------------------------------------------------------------------------
  public void write(ConfigCollection root) {//---------------------------------------------------------------------------

    String newFilename = m_filename + ".new";
    String backupBase = m_filename + ".bak.";
    File file1;
    File file2;
    PrintWriter out = null;

    try {
      out = new PrintWriter(new FileWriter(newFilename));

      out.println("###########################################################");
      out.println("# General");
      out.println("###########################################################");
      out.println();

      Iterator iter = root.getPropertyNames().iterator();

      while (iter.hasNext()) {
        String key = (String)iter.next();
        String value = root.getProperty(key);
        StringBuffer buffer = new StringBuffer();
        int idx1 = 0;
        int idx2;

        while ((idx2 = value.indexOf("\\", idx1)) > 0) {
          buffer.append(value.substring(idx1, idx2));
          buffer.append("\\\\");
          idx1 = idx2 + 1;
        }
        buffer.append(value.substring(idx1));

        out.println(key + " = " + buffer.toString());
      }

      iter = root.getChildren().iterator();

      while (iter.hasNext()) {
        ConfigCollection collection = (ConfigCollection)iter.next();

        out.println();
        out.println("###########################################################");
        out.println("# " + collection.getName());
        out.println("###########################################################");
        out.println();

        writeCollection(out, collection.getName(), collection);
      }
      out.flush();
      out.close();

      file1 = new File(backupBase + "3");
      if (file1.exists()) {
        file1.delete();
      }
      file2 = new File(backupBase + "2");
      if (file2.exists()) {
        file2.renameTo(file1);
      }
      file1 = file2;
      file2 = new File(backupBase + "1");
      if (file2.exists()) {
        file2.renameTo(file1);
      }
      file1 = file2;
      file2 = new File(m_filename);
      if (file2.exists()) {
        file2.renameTo(file1);
      }
      file1 = file2;
      file2 = new File(newFilename);
      if (file2.exists()) {
        file2.renameTo(file1);
      }
    }
    catch (IOException e) {
            //$JL-EXC$      
      if (out != null) {
        out.close();
      }
      log.errorT("write(196)", "Caught IOException in write(): " + e);
    }
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  //---------------------------------------------------------------------------
  void writeCollection(PrintWriter out, String path, ConfigCollection collection)
    throws IOException {//---------------------------------------------------------------------------

    Iterator iter = collection.getPropertyNames().iterator();

    while (iter.hasNext()) {
      String key = (String)iter.next();
      String value = collection.getProperty(key);
      StringBuffer buffer = new StringBuffer();
      int idx1 = 0;
      int idx2;

      while ((idx2 = value.indexOf("\\", idx1)) > 0) {
        buffer.append(value.substring(idx1, idx2));
        buffer.append("\\\\");
        idx1 = idx2 + 1;
      }
      buffer.append(value.substring(idx1));

      out.println(path + "." + key + " = " + buffer.toString());
    }

    iter = collection.getChildren().iterator();

    while (iter.hasNext()) {
      ConfigCollection child = (ConfigCollection)iter.next();

      writeCollection(out, path + "." + child.getName(), child);
    }
  }
  //---------------------------------------------------------------------------
}
