/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.config;

import java.io.File;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import com.sapportals.config.event.IConfigEventListener;
import com.sapportals.config.event.IConfigEventService;
import com.sapportals.config.fwk.CannotAccessConfigException;
import com.sapportals.config.fwk.ConfigException;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigItem;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigUri;
import com.sapportals.config.fwk.IConfigUriResolver;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.config.fwk.IMutableConfigurable;
import com.sapportals.config.fwk.InitialConfigException;
import com.sapportals.config.fwk.InvalidEntryException;
import com.sapportals.portal.security.usermanagement.IServiceUserFactory;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.portal.security.usermanagement.UserManagementException;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

/**
 * Helper functions for the configuration framework
 *
 * @author D039230
 * @created 6. Mai 2003
 * 
 * @deprecated since EP 6.0 SP3. Use the Config Fraemwork directly
 */
public class ConfigurationProxy {
  // plugin names
  public final static String CFG_PLUGIN_CM = "/cm";

  public final static String CFG_PLUGIN_CM_CONTROLS = "/cm/controls";
  public final static String CFG_PLUGIN_CM_CONTROLS_MAIL_CONTROL = "/cm/controls/mail_control";

  public final static String CFG_PLUGIN_CM_PROTOCOLS = "/cm/protocols";
  public final static String CFG_PLUGIN_CM_PROTOCOLS_WEBDAV_PROTOCOL = "/cm/protocols/webdav_protocol";

  public final static String CFG_PLUGIN_CM_REPOSITORY_FILTERS = "/cm/repository_filters";

  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS = "/cm/repository_managers";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_WEB_SETUPS = "/cm/repository_managers/web_setups";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_WEB_SETUP_WEB_HTML_PROPERTIES = "/cm/repository_managers/web_setups/web_html_properties";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_WEB_SYSTEMS = "/cm/repository_managers/web_systems";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_LOTUS_DISPLAY_ITEMS = "/cm/repository_managers/lotus_display_items";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_REPORTS = "/cm/repository_managers/reports";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_RIP_SERVICES = "/cm/repository_managers/rip_services";
  public final static String CFG_PLUGIN_CM_REPOSITORY_MANAGERS_SECURITY_MANAGERS = "/cm/repository_managers/security_managers";

  public final static String CFG_PLUGIN_CM_REPOSITORY_SERVICES = "/cm/repository_services";
  public final static String CFG_PLUGIN_CM_REPOSITORY_SERVICES_EVENT_MAPPINGS = "/cm/repository_services/event_mappings";

  public final static String CFG_CLASS_CM_SERVICES_URLGENERATOR = "/cm/services/URLGeneratorService";
  public final static String CFG_CLASS_CM_SERVICES_PLACEHOLDER_RESOLVERS = "/cm/services/placeholder_resolvers/PlaceholderResolver";

  public final static String CFG_PLUGIN_CM_SERVICES = "/cm/services";
  public final static String CFG_PLUGIN_CM_SERVICES_CRAWLERS = "/cm/services/crawlers";
  public final static String CFG_PLUGIN_CM_SERVICES_CRAWLER_PROFILES = "/cm/services/crawler_profiles";
  public final static String CFG_PLUGIN_CM_SERVICES_ICE_SERVICE = "/cm/services/ice";
  public final static String CFG_PLUGIN_CM_SERVICES_INDEX_SERVICES = "/cm/services/index_services";
  public final static String CFG_PLUGIN_CM_SERVICES_PIPELINE_ADDONS = "/cm/services/pipeline_addons";
  public final static String CFG_PLUGIN_CM_SERVICES_CUSTOM_PROPERTIES = "/cm/services/custom_properties";
  public final static String CFG_PLUGIN_CM_SERVICES_LANDSCAPE = "/cm/services/landscape";
  public final static String CFG_PLUGIN_CM_SERVICES_LANDSCAPE_SYSTEMS = "/cm/services/landscape/systems";
  public final static String CFG_PLUGIN_CM_SERVICES_LANDSCAPE_CREDENTIALS = "/cm/services/landscape/credentials";
  public final static String CFG_PLUGIN_CM_SERVICES_LANDSCAPE_CREDENTIALS_CREDENTIAL = "/cm/services/landscape/credentials/credential";
  public final static String CFG_PLUGIN_CM_SERVICES_NETWORK_PATHS = "/cm/services/network_paths";
  public final static String CFG_PLUGIN_CM_SERVICES_SCHEDULERTIMETABLES = "/cm/services/schedulertimetables";
  public final static String CFG_PLUGIN_CM_SERVICES_SCHEDULERTASKS = "/cm/services/schedulertasks";
  public final static String CFG_PLUGIN_CM_SERVICES_LISTFILTER = "/cm/services/resourcelistfilter";
  public final static String CFG_PLUGIN_CM_SERVICES_HOUSEKEEPINGTASKS = "/cm/services/housekeepingtasks";
  public final static String CFG_PLUGIN_CM_SERVICES_URIMAPPER = "/cm/services/urimapper";

  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA = "/cm/services/properties_metadata";
  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA_NAMESPACES = "/cm/services/properties_metadata/namespaces";
  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA_GROUPS = "/cm/services/properties_metadata/groups";
  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA_EXTENSION = "/cm/services/properties_metadata/metadata_extension";
  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA_RENDERER = "/cm/services/properties_metadata/renderer";
  public final static String CFG_PLUGIN_CM_SERVICES_PROPERTIES_METADATA_SERVICE = "/cm/services";

  public final static String CFG_PLUGIN_CM_USERMANAGEMENT = "/cm/usermanagement";

  //next generation ui
  public final static String CFG_PLUGIN_CM_UI_COLLECTIONRENDERER = "/cm/ui/collectionrenderer";
  public final static String CFG_PLUGIN_CM_UI_COLLECTIONRENDERER_SETTINGS = "/cm/ui/settings/collectionrenderer";
  public final static String CFG_PLUGIN_CM_UI_DETAILS_OVERVIEW = "/cm/ui/detailsoverview";
  public final static String CFG_PLUGIN_CM_UI_DETAILS_PROPERTIES = "/cm/ui/detailsproperties";
  public final static String CFG_PLUGIN_CM_UI_IMAGEMAPS = "/cm/ui/imagemaps";
  public final static String CFG_PLUGIN_CM_UI_IMAGEMAPS_AREAS = "/cm/ui/imagemaps/areas";
  public final static String CFG_PLUGIN_CM_UI_MAPPING = "/cm/ui/mapping";
  public final static String CFG_PLUGIN_CM_UI_PREVIEW = "/cm/ui/preview";
  public final static String CFG_PLUGIN_CM_UI_PROXY_SETTING = "/cm/ui/proxy";
  public final static String CFG_PLUGIN_CM_UI_RESOURCERENDERER = "/cm/ui/resourcerenderer";
  public final static String CFG_PLUGIN_CM_UI_RESOURCERENDERER_SETTINGS = "/cm/ui/settings/resourcerenderer";
  public final static String CFG_PLUGIN_CM_UI_SETTINGS = "/cm/ui/settings";
  public final static String CFG_PLUGIN_CM_UI_UICOMMAND = "/cm/ui/uicommand";
  public final static String CFG_PLUGIN_CM_UI_COMPONENT = "/cm/ui/component";
  public final static String CFG_PLUGIN_CM_UI_CONTROL = "/cm/ui/control";

  //control status service
  public final static String CFG_PLUGIN_CM_UTILITIES_CSS = "/cm/utilities/control_status";

  public final static String CFG_PLUGIN_CM_UTILITIES = "/cm/utilities";
  public final static String CFG_PLUGIN_CM_UTILITIES_ACL_FACTORIES = "/cm/utilities/acl_factories";
  public final static String CFG_PLUGIN_CM_UTILITIES_ACL_MANAGERS = "/cm/utilities/acl_managers";
  public final static String CFG_PLUGIN_CM_UTILITIES_CACHES = "/cm/utilities/caches";
  public final static String CFG_PLUGIN_CM_UTILITIES_CHANNELS = "/cm/utilities/channels";
  public final static String CFG_PLUGIN_CM_UTILITIES_CONNECTION_POOLS = "/cm/utilities/connection_pools";
  public final static String CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS = "/cm/utilities/content_access";
  public final static String CFG_PLUGIN_CM_UTILITIES_CONTENT_ACCESS_PLUGINS = "/cm/utilities/content_access/content_access_plugins";
  public final static String CFG_PLUGIN_CM_UTILITIES_ICONS = "/cm/utilities/icons";
  public final static String CFG_PLUGIN_CM_UTILITIES_MIMES = "/cm/utilities/mimes";
  public final static String CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG = "/cm/utilities/system_config";
  public final static String CFG_PLUGIN_CM_UTILITIES_SYSTEM_CONFIG_MAPPINGS = "/cm/utilities/system_config/permission_mappings";
  public final static String CFG_PLUGIN_CM_UTILITIES_CHANNEL_ORIGINATORS = "/cm/utilities/channel_originators";
  public final static String CFG_PLUGIN_CM_UTILITIES_EDITING = "/cm/utilities/editing";

  public final static String CFG_PLUGIN_CM_XML_FORMS = "/cm/xml_forms";

  public final static String REPLACE_KEY_SERVLET_PATH = "servletpath";
  public final static String REPLACE_KEY_HOSTNAME = "hostname";
  public final static String REPLACE_KEY_DB_SERVER = "dbserver";
  public final static String REPLACE_KEY_DB_SID = "dbsid";
  public final static String REPLACE_KEY_DB_PORT = "dbport";
  public final static String REPLACE_KEY_PROTOCOL = "protocolserver";
  public final static String REPLACE_KEY_SYSTEMID = "systemid";
  public final static String REPLACE_KEY_TREX_SERVER = "trexserver";

  private final static String CONFIG_SERVICE_USER = "index_service";

  private final static String CONFIG_FILE_PROP = "cm.config.root";
  private final static String CONFIG_MANAGER_PROP = "cm.config.manager";
  private final static String CONFIG_SERVLETPATH_PROP = "servletpath";
  private final static String CONFIG_HOSTNAME_PROP = "hostname";
  private final static String CONFIG_DB_SERVER_PROP = "dbserver";
  private final static String CONFIG_DB_SID_PROP = "dbsid";
  private final static String CONFIG_DB_PORT_PROP = "dbport";
  private final static String CONFIG_PROTOCOL_PROP = "protocolserver";
  private final static String CONFIG_SYSTEMID_PROP = "systemid";
  private final static String CONFIG_TREX_SERVER_PROP = "trexserver";


  private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.config.ConfigurationProxy.class);
  private static ConfigurationProxy s_instance = null;

  private String m_configManagerName = null;
  private boolean m_configurationWasInitializedByMe = false;

  /**
   * Set the servlet path (the path to the etc directory where the config is
   * located)
   *
   * @param servletPath The new servletPath value
   * @deprecated since EP6.0 SP2
   */
  public static void setServletPath(String servletPath) {
  }

  /**
   * Set the path to the config.properties file
   *
   * @param path The new path value
   * @deprecated since EP6.0 SP2
   */
  public static void setPath(String path) {
  }

  /**
   * @param configurable Description of the Parameter
   * @return the name/value pairs of the attributes of a configurable in a
   *      property bag
   */
  public static Properties getConfigurableProperties(IConfigurable configurable) {
    Properties configProperties = map2properties(configurable.getProperties(true));
    return configProperties;
  }

  private static Properties map2properties(Map map) {
    Properties properties = new Properties();
    Iterator i = map.keySet().iterator();
    while (i.hasNext()) {
      Object key = i.next();
      Object value = map.get(key);
      if (value != null) {
        properties.setProperty(key.toString(), value.toString());
      }
    }
    return properties;
  }

  /**
   * @return the File object for the config-property file
   * @deprecated since EP6.0 SP2
   */
  public static File getFile() {
    return null;
  }

  /**
   * @return an instance of the ConfigurationProxy (singleton)
   * @exception InitialConfigException Description of the Exception
   * @exception CannotAccessConfigException Description of the Exception
   */
  public static synchronized ConfigurationProxy getInstance() throws InitialConfigException, CannotAccessConfigException {

    if (s_instance == null) {
      s_instance = new ConfigurationProxy();
    }
    return s_instance;
  }

  /**
   * Construct
   *
   * @exception InitialConfigException Description of the Exception
   * @exception CannotAccessConfigException Description of the Exception
   */
  private ConfigurationProxy() throws InitialConfigException, CannotAccessConfigException {
  	m_configManagerName = "local";
  }

  /**
   * @return the name of the CM config manager
   */
  public String getConfigManagerName() {
    return m_configManagerName;
  }

  /**
   * @return the CM config manager (read only)
   */
  public IConfigManager getConfigManager() {
    IConfigClientContext context = IConfigClientContext.createContext(null, m_configManagerName);
    try {
      return com.sapportals.config.fwk.Configuration.getInstance().getConfigManager(context);
    } catch (Exception ex) {
            //$JL-EXC$      
      s_log.fatalT("getConfigManager(367)", LoggingFormatter.extractCallstack(ex));
      throw new RuntimeException(ex.getMessage());
    }
  }

  /**
   * registeres a config event listener for the domaines specified
   *
   * @param listener the <code> IConfigEventListener </code>
   * @param domains config uri's to watch
   * @exception InitialConfigException Description of the Exception
   * @exception InvalidEntryException Description of the Exception
   */
  public void addConfigEventListener(IConfigEventListener listener, String[] domains) throws InitialConfigException, InvalidEntryException {
    if (domains != null) {
      // retrieve the configService
      IConfigEventService eventService = com.sapportals.config.fwk.Configuration.getInstance().getConfigEventServiceInstance();
      // subscribe to the service
      eventService.addConfigEventListener(listener, domains);
    }
  }

  /**
   * deregisters the specified eventlistener.
   *
   * @param listener Description of the Parameter
   * @exception InitialConfigException Description of the Exception
   * @exception InvalidEntryException Description of the Exception
   */
  public void terminateListener(IConfigEventListener listener) throws InitialConfigException, InvalidEntryException {
    // retrieve the configService
    IConfigEventService eventService = com.sapportals.config.fwk.Configuration.getInstance().getConfigEventServiceInstance();
    // unsubscribe to the service (all domains are removed)
    eventService.removeConfigEventListener(listener);
  }

  /**
   * returns the source of the event as <code> IConfigItem </code> . Depending
   * on the type of the event, this item can be casted to a <code>
   * IConfigManager </code> or to an <code>IConfigurable </code>
   *
   * @param e Description of the Parameter
   * @return The configItemFromEvent value
   * @exception InitialConfigException Description of the Exception
   */
  public IConfigItem getConfigItemFromEvent(com.sapportals.config.event.ConfigEvent e) throws InitialConfigException {
    IConfigItem returnItem = null;
    // retrieve the item on the uri resolver
    IConfigUriResolver configUriResolver = com.sapportals.config.fwk.Configuration.getInstance().getConfigUriResolverInstance();

    // retrieve the source of the event depending of the event type. Remark: operator == is used instead of equals()
    if (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGMANAGER_INITIALIZED) {

      // WARNING: as it is an asynchronous mechanism, the state of the source configmanager could have change!
      returnItem = (IConfigManager)configUriResolver.getItem(e.getSourceUri(), IConfigManager.CONFIGMANAGER);
    } else if (
      (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_ADDED)
        || (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_DELETED)
        || (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_UPDATED)) {

      // WARNING: as it is an asynchronous mechanism, the state of the configurable item could have change!
      returnItem = (IConfigurable)configUriResolver.getItem(e.getSourceUri(), IConfigurable.CONFIGURABLE);
    }
    return returnItem;
  }

  /**
   * returns the source from the event as <code> IConfigManager </code> , if
   * possible, otherwise null is returned
   *
   * @param e Description of the Parameter
   * @return The configManagerFromEvent value
   * @exception InitialConfigException Description of the Exception
   */
  public IConfigManager getConfigManagerFromEvent(com.sapportals.config.event.ConfigEvent e) throws InitialConfigException {
    IConfigManager returnItem = null;
    if (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGMANAGER_INITIALIZED) {
      returnItem = (IConfigManager)this.getConfigItemFromEvent(e);
    }
    return returnItem;
  }

  /**
   * returns the source from the event as <code> IConfigurable </code> , if
   * possible, otherwise null is returned
   *
   * @param e Description of the Parameter
   * @return The configurableFromEvent value
   * @exception InitialConfigException Description of the Exception
   */
  public IConfigurable getConfigurableFromEvent(com.sapportals.config.event.ConfigEvent e) throws InitialConfigException {
    IConfigurable returnItem = null;
    if ((e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_ADDED)
      || (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_DELETED)
      || (e.getType() == com.sapportals.config.event.ConfigEvent.CONFIGURABLE_UPDATED)) {
      returnItem = (IConfigurable)this.getConfigItemFromEvent(e);
    }
    return returnItem;
  }

  /**
   * @param configPluginName Description of the Parameter
   * @return all configurables of the specified plugin, or <code>null</code> if
   *      the plugin was not found.
   * @exception CannotAccessConfigException Description of the Exception
   */
  public IMutableConfigurable[] getPluginConfigurables(String configPluginName) throws CannotAccessConfigException {

    IConfigPlugin plugin = this.getConfigPlugin(configPluginName);
    if (plugin == null) {
      throw new CannotAccessConfigException("plugin not found: " + configPluginName);
    }
    return plugin.getConfigurables();
  }

  /**
   * @param configPluginName Description of the Parameter
   * @param configclassname Description of the Parameter
   * @param subclasses Description of the Parameter
   * @return all configurables of the specified plugin of the specified class,
   *      or <code>null</code> if the plugin was not found.
   * @exception CannotAccessConfigException Description of the Exception
   */
  public IMutableConfigurable[] getPluginConfigurables(String configPluginName, String configclassname, boolean subclasses)
    throws CannotAccessConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPluginName);
    if (plugin == null) {
      throw new CannotAccessConfigException("plugin not found: " + configPluginName);
    }
    return plugin.getConfigurables(configclassname, subclasses);
  }

  /**
   * @param configPluginName Description of the Parameter
   * @return the first configurable of the specified plugin (used to get a
   *      singleton configurable)
   * @exception CannotAccessConfigException Description of the Exception
   */
  public IConfigurable getFirstPluginConfigurable(String configPluginName) throws CannotAccessConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPluginName);
    IMutableConfigurable[] configurables = plugin.getConfigurables();
    if (configurables.length < 1) {
      return null;
    }
    return configurables[0];
  }

  /**
   * @param configPluginName Description of the Parameter
   * @param configurableName Description of the Parameter
   * @return a specific configurable from the specified plugin
   * @exception CannotAccessConfigException Description of the Exception
   */
  public IMutableConfigurable getPluginConfigurable(String configPluginName, String configurableName) throws CannotAccessConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPluginName);
    if (plugin == null) {
      throw new CannotAccessConfigException(configPluginName + " configurable not found");
    }
    return plugin.getConfigurable(configurableName);
  }

  public IConfigurable getConfigurableByUri(IConfigUri uri) throws CannotAccessConfigException {
    return this.getConfigManager().getConfigurable(uri);
  }

  /**
   * @param configPluginName Description of the Parameter
   * @param configurableName Description of the Parameter
   * @return the name/value pairs of the attributes of the specified
   *      configurable in a property bag
   * @exception CannotAccessConfigException Description of the Exception
   */
  public Properties getPluginConfigurableProperties(String configPluginName, String configurableName) throws CannotAccessConfigException {

    IConfigurable configurable = getPluginConfigurable(configPluginName, configurableName);
    if (configurable == null) {
      throw new CannotAccessConfigException("configurable not found");
    }

    return getConfigurableProperties(configurable);
  }

  /**
   * Save a configurable
   *
   * @param configPlugin Description of the Parameter
   * @param configurable Description of the Parameter
   * @exception ConfigException Description of the Exception
   */
  public void savePluginConfigurable(String configPlugin, IMutableConfigurable configurable) throws ConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPlugin);
    if (plugin == null) {
      throw new CannotAccessConfigException("plugin not found");
    }

    if (getPluginConfigurable(configPlugin, configurable.getIdValue()) == null) {
      plugin.addConfigurable(configurable);
    } else {
      plugin.replaceConfigurable(configurable);
    }
  }

  /**
   * Remove a configurable
   *
   * @param configPluginName Description of the Parameter
   * @param configurableName Description of the Parameter
   * @exception ConfigException Description of the Exception
   */
  public void removePluginConfigurable(String configPluginName, String configurableName) throws ConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPluginName);
    IMutableConfigurable configurable = plugin.getConfigurable(configurableName);
    if (configurable == null) {
      throw new CannotAccessConfigException("configurable not found");
    }
    plugin.removeConfigurable(configurable);
  }

  /**
   * Create a new configurable
   *
   * @param configPlugin Description of the Parameter
   * @param configurableName Description of the Parameter
   * @param properties Description of the Parameter
   * @return Description of the Return Value
   * @exception ConfigException Description of the Exception
   */
  public IMutableConfigurable createPluginConfigurable(String configPlugin, String configurableName, Properties properties) throws ConfigException {

    IConfigPlugin plugin = getConfigPlugin(configPlugin);
    if (plugin == null) {
      throw new CannotAccessConfigException("plugin not found");
    }
    return plugin.createConfigurable(properties, configurableName);
  }

  /**
   * Reload config.properties, reinitialize the config manager (for xtests)
   *
   * @exception InitialConfigException Description of the Exception
   * @exception CannotAccessConfigException Description of the Exception
   * @deprecated since EP6.0 SP2
   */
  public void reloadConfiguration() throws InitialConfigException, CannotAccessConfigException {  }

  /**
   * @param configPluginName Description of the Parameter
   * @return the specified plugin
   * @exception CannotAccessConfigException Description of the Exception
   */
  private synchronized IConfigPlugin getConfigPlugin(String configPluginName) throws CannotAccessConfigException {

    return this.getConfigManager().getConfigPlugin(configPluginName);
  }

  /**
   * @return the contents of the config.properties file as property bag
   * @exception CannotAccessConfigException Description of the Exception
   */
  public Properties readProxyConfiguration() throws CannotAccessConfigException {

    // ds: don't read config.properties any more
    //     temporary solution: return prefilled property bag
    //     final solution: remove ConfigurationProxy - job of base ;-)

    Properties result=new Properties();

    // this might still be used (???)
    result.setProperty(CONFIG_MANAGER_PROP,"local"); 

    // the other properties contain dummy values
    result.setProperty(CONFIG_FILE_PROP,"driss");
    result.setProperty(CONFIG_SERVLETPATH_PROP,"driss");
    result.setProperty(CONFIG_HOSTNAME_PROP,"driss");
    result.setProperty(CONFIG_DB_SERVER_PROP,"driss");
    result.setProperty(CONFIG_DB_SID_PROP,"driss");
    result.setProperty(CONFIG_DB_PORT_PROP,"driss");
    result.setProperty(CONFIG_PROTOCOL_PROP,"driss");
    result.setProperty(CONFIG_SYSTEMID_PROP,"driss");
    result.setProperty(CONFIG_TREX_SERVER_PROP,"driss");

    return result;

  }

  /**
   * Converts configurable properties to attributes of CRT component
   * configuration. Applies "replace strings", such as <%servletpath%>.
   *
   * @param configurable Description of the Parameter
   * @deprecated since EP6.0 SP2
   */
  public static HashMap convertConfigurable(IConfigurable configurable) {
    // we need a hash map as return value :-(
    HashMap cfg = new HashMap(configurable.getProperties(true));

    return cfg;
  }


  /**
   * @return an authenticated user for locking
   * @exception UserManagementException Description of the Exception
   */
  public IUser getConfigServiceUser() throws UserManagementException {

    IServiceUserFactory factory = WPUMFactory.getServiceUserFactory();
    return factory.getServiceUser(CONFIG_SERVICE_USER);
  }

}
