/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.config;

import java.io.*;
import java.util.*;

/**
 * TBD: Description of the class.
 */
public class ConfigurationManager {

  private final static String CONFIG_INCLUDE = "include";

  private static ConfigurationManager s_configurationManager;

  public File getFile(String configurationName) {
    return null;
  }

  /**
   * clear the container hash table m_configurations
   */
  public void clearAllProperties() { }

  public Properties loadProperties(String configurationName)
    throws Exception {
    return loadProperties(configurationName, (Properties)null);
  }

  public Properties loadProperties(Object caller, String configurationName)
    throws Exception {
    return loadProperties(caller, configurationName, null);
  }

  public Properties loadProperties(String configurationName, Properties replaceStrings)
    throws Exception {
    return loadProperties(this, configurationName, replaceStrings);
  }

  public Properties loadProperties(
    Object caller,
    String configurationName,
    Properties replaceStrings)
    throws Exception {
    return lowLoadProperties(caller, configurationName, replaceStrings);
  }

  public Properties loadPropertiesFromPath(String path, String configurationName)
    throws Exception {
    return loadPropertiesFromPath(path, configurationName, null);
  }

  public Properties loadPropertiesFromPath(
    String path,
    String configurationName,
    Properties replaceStrings)
    throws Exception {

    File configurationFile = new File(path, configurationName);

    Properties properties = new Properties();
    InputStream input = new FileInputStream(configurationFile);
    try {
      properties.load(input);
    }
    finally {
      input.close();
    }

    properties = trimProperties(properties);

    if (replaceStrings != null) {
      replacePropertyStrings(properties, replaceStrings);
    }
    properties = importProperties(configurationFile, replaceStrings, properties);

    return properties;
  }

  protected Properties lowLoadProperties(
    Object caller,
    String configurationFile,
    Properties replaceStrings)
    throws Exception {
    return readConfigurationFile(caller, configurationFile, replaceStrings);
  }

  private String getParentDirectory(File file) {
    File parentDirectoryFile = file.getParentFile();
    if (parentDirectoryFile != null) {
      return parentDirectoryFile.getAbsolutePath();
    }
    return "/";
  }

  private Properties readConfigurationFile(File configurationFile, Properties replaceStrings)
    throws Exception {
    Properties properties = new Properties();

    InputStream input = new FileInputStream(configurationFile);
    try {
      properties.load(input);
    }
    catch (Exception e) {
      throw e;
    }
    finally {
      input.close();
    }

    properties = trimProperties(properties);

    if (replaceStrings != null) {
      replacePropertyStrings(properties, replaceStrings);
    }
    properties = importProperties(configurationFile, replaceStrings, properties);

    return properties;
  }

  private Properties readConfigurationFile(
    Object caller,
    String configurationFile,
    Properties replaceStrings)
    throws Exception {
    Properties properties = new Properties();

    File file = new File(findConfigurationFilePath(getClassPath(caller.getClass()), configurationFile), configurationFile);
    InputStream input = null;

    if (file.exists()) {
      input = new FileInputStream(file);
    }
    else {
      input = caller.getClass().getClassLoader().getResourceAsStream(configurationFile);
    }
    if (input == null) {
      throw new FileNotFoundException("failed to load " + configurationFile);
    }

    properties.load(input);
    properties = trimProperties(properties);

    if (replaceStrings != null) {
      replacePropertyStrings(properties, replaceStrings);
    }
    properties = importProperties(caller, configurationFile, replaceStrings, properties);

    return properties;
  }

  private Properties trimProperties(Properties properties) {
    Enumeration keys = properties.keys();
    while (keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      properties.setProperty(key, properties.getProperty(key).trim());
    }

    return properties;
  }

  private void replacePropertyStrings(Properties properties, Properties replaceStrings) {
    final String REPLACE_STRING_PREFIX = "<%";
    final String REPLACE_STRING_POSTFIX = "%>";

    Enumeration names = properties.propertyNames();
    while (names.hasMoreElements()) {
      String name = (String)names.nextElement();
      String value = properties.getProperty(name);

      int start = value.indexOf(REPLACE_STRING_PREFIX);
      int end = value.indexOf(REPLACE_STRING_POSTFIX);

      if (start >= 0 && end >= 0 && end > start + REPLACE_STRING_PREFIX.length()) {
        String placeHolder = value.substring(start + REPLACE_STRING_PREFIX.length(), end).trim();
        if (placeHolder.length() > 0) {
          // trim() may shorten the placeHolder

          String before = value.substring(0, start);

          String after;
          if (end + REPLACE_STRING_POSTFIX.length() < value.length()) {
            after = value.substring(end + REPLACE_STRING_POSTFIX.length());
          }
          else {
            after = "";
          }

          String replaceValue = replaceStrings.getProperty(placeHolder);
          if (replaceValue != null) {
            properties.setProperty(name, before + replaceValue + after);
          }
        }
      }
    }
  }

  private String findConfigurationFilePath(String classpath, String confFile) {
    boolean found = false;
    String path = classpath;

    // first search for the config file at the location of the jar file
    if (path != null) {
      String dir = new File(path).getParent();
      found = new File(dir, confFile).exists();
      if (found) {
        return dir;
      }
    }

    // second search for the config file in the classpath
    StringTokenizer paths =
      new StringTokenizer(
      System.getProperty("java.class.path"),
      System.getProperty("path.separator"));
    confFile = (String)System.getProperty("file.separator") + confFile;

    while (!found && paths.hasMoreTokens()) {
      path = paths.nextToken();
      File cFile = new File(path);
      if (cFile.isFile()) {
        File nFile = new File(cFile.getParent(), confFile);
        if (nFile.exists()) {
          return nFile.getParent();
        }
      }
      else {
        File nFile = new File(cFile.getAbsolutePath(), confFile);
        if (nFile.exists()) {
          return nFile.getParent();
        }
      }
    }

    return null;
  }

  private String getClassPath(Class aClass) {
    final String JAR_FILE_PREFIX = "jar:file:";
    final String FILE_PREFIX = "file:";

    try {
      String className = aClass.getName().replace('.', '/') + ".class";
      ClassLoader classLoader = aClass.getClassLoader();
      String url = classLoader.getResource(className).toString();

      if (url.startsWith(JAR_FILE_PREFIX)) {
        return url.substring(JAR_FILE_PREFIX.length(), url.length() - className.length() - 2);
      }
      // ???
      if (url.startsWith(FILE_PREFIX)) {
        return url.substring(FILE_PREFIX.length());
      }

      return null;
    }
    catch (Exception e) {
            //$JL-EXC$      
      return null;
    }
  }

  private Properties importProperties(
    File configurationFile,
    Properties replaceStrings,
    Properties properties)
    throws Exception {
    String includeFiles = (String)properties.get(CONFIG_INCLUDE);
    if (includeFiles == null || includeFiles.length() <= 0) {
      return properties;
    }

    StringTokenizer tok = new StringTokenizer(includeFiles, ",");
    while (tok.hasMoreTokens()) {
      String includeFilename = tok.nextToken().trim();

      File includeFile = new File(getParentDirectory(configurationFile), includeFilename);
      if (!includeFile.exists()) {
        continue;
      }

      Properties includeProperties = readConfigurationFile(includeFile, replaceStrings);

      Enumeration includePropertiesEnum = includeProperties.keys();
      while (includePropertiesEnum.hasMoreElements()) {
        String include = (String)includePropertiesEnum.nextElement();
        if (include.startsWith("+")) {
          // add values
          String includeKey = include.substring(1);
          StringBuffer newValue;
          if (properties.containsKey(includeKey)) {
            // append included values to existing values
            newValue = new StringBuffer((String)properties.get(includeKey));
            newValue.append(",");
            newValue.append(includeProperties.get(include));
          }
          else {
            // add included values as new property
            newValue = new StringBuffer((String)includeProperties.get(include));
          }
          properties.setProperty(includeKey, newValue.toString());
        }
        else {
          if (include.startsWith("-")) {
            // remove a value
            // ?? what about multiple import values (removing a,b,c from a,b,c,d,e,f)
            String includeKey = include.substring(1);
            if (properties.containsKey(includeKey)) {
              String includeValue = (String)includeProperties.get(include);
              if (includeValue == null || includeValue.trim().length() <= 0) {
                // remove included property
                properties.remove(includeKey);
              }
              else {
                // remove imported value from existing values
                StringBuffer newValue = new StringBuffer();
                StringTokenizer includeTokenizer = new StringTokenizer(includeValue, ",");
                HashSet includeTokens = new HashSet();
                while (includeTokenizer.hasMoreElements()) {
                  String token = (String)includeTokenizer.nextElement();
                  includeTokens.add(token.trim());
                }
                String oldValue = (String)properties.get(includeKey);
                StringTokenizer tokenizer = new StringTokenizer(oldValue, ",");
                boolean isFirstValue = true;
                while (tokenizer.hasMoreElements()) {
                  String token = (String)tokenizer.nextElement();
                  if (!includeTokens.contains(token.trim())) {
                    if (!isFirstValue) {
                      newValue.append(',');
                    }
                    newValue.append(token);
                    isFirstValue = false;
                  }
                }
                properties.setProperty(includeKey, newValue.toString());
              }
            }
          }
          else {
            // simply add included property to properties
            if (include.startsWith("=")) {
              // allow '=' as prefix...
              include = include.substring(1);
            }
            properties.setProperty(include, (String)includeProperties.get(include));
          }
        }
      }
    }
    return properties;
  }

  /**
   * Include a sub configuration (specified by a property) and merge it to the
   * given original properties.
   *
   * @param properties the <code>Properties</code> with the original
   *      configuration.
   * @param configurationFile
   * @param replaceStrings
   * @param caller TBD: Description of the incoming method parameter
   * @return <code>Properties</code> the orignal properties merged with the
   *      included properties.
   */
  private Properties importProperties(
    Object caller,
    String configurationFile,
    Properties replaceStrings,
    Properties properties)
    throws Exception {
    String includeFiles = (String)properties.get(CONFIG_INCLUDE);
    if (includeFiles == null || includeFiles.length() <= 0) {
      return properties;
    }

    StringTokenizer tok = new StringTokenizer(includeFiles, ",");
    while (tok.hasMoreTokens()) {
      String includeFilename = tok.nextToken().trim();

      try {
        Properties includeProperties =
          readConfigurationFile(caller, includeFilename, replaceStrings);
        Enumeration includePropertiesEnum = includeProperties.keys();
        while (includePropertiesEnum.hasMoreElements()) {
          String include = (String)includePropertiesEnum.nextElement();
          if (include.startsWith("+")) {
            // add values
            String includeKey = include.substring(1);
            StringBuffer newValue;
            if (properties.containsKey(includeKey)) {
              // append included values to existing values
              newValue = new StringBuffer((String)properties.get(includeKey));
              newValue.append(",");
              newValue.append(includeProperties.get(include));
            }
            else {
              // add included values as new property
              newValue = new StringBuffer((String)includeProperties.get(include));
            }
            properties.setProperty(includeKey, newValue.toString());
          }
          else {
            if (include.startsWith("-")) {
              // remove a value
              // ?? what about multiple import values (removing a,b,c from a,b,c,d,e,f)
              String includeKey = include.substring(1);
              if (properties.containsKey(includeKey)) {
                String includeValue = (String)includeProperties.get(include);
                if (includeValue == null || includeValue.trim().length() <= 0) {
                  // remove included property
                  properties.remove(includeKey);
                }
                else {
                  // remove imported value from existing values
                  StringBuffer newValue = new StringBuffer();
                  StringTokenizer includeTokenizer = new StringTokenizer(includeValue, ",");
                  HashSet includeTokens = new HashSet();
                  while (includeTokenizer.hasMoreElements()) {
                    String token = (String)includeTokenizer.nextElement();
                    includeTokens.add(token.trim());
                  }
                  String oldValue = (String)properties.get(includeKey);
                  StringTokenizer tokenizer = new StringTokenizer(oldValue, ",");
                  boolean isFirstValue = true;
                  while (tokenizer.hasMoreElements()) {
                    String token = (String)tokenizer.nextElement();
                    if (!includeTokens.contains(token.trim())) {
                      if (!isFirstValue) {
                        newValue.append(',');
                      }
                      newValue.append(token);
                      isFirstValue = false;
                    }
                  }
                  properties.setProperty(includeKey, newValue.toString());
                }
              }
            }
            else {
              // simply add included property to properties
              if (include.startsWith("=")) {
                // allow '=' as prefix...
                include = include.substring(1);
              }
              properties.setProperty(include, (String)includeProperties.get(include));
            }
          }
        }
      }
      catch (FileNotFoundException x) {
            //$JL-EXC$        
        com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(x.getMessage());
      }
    }
    return properties;
  }

  public static synchronized ConfigurationManager getInstance() {
    if (s_configurationManager == null) {
      s_configurationManager = new ConfigurationManager();
    }
    return s_configurationManager;
  }

  public static Properties getSubProperties(Properties properties, String base) {
    if (properties == null) {
      return null;
    }

    Properties result = new Properties();
    Properties propertiesClone = propertiesClone = (Properties)properties.clone();

    String prefix = base + ".";
    Enumeration keys = propertiesClone.propertyNames();
    while (keys.hasMoreElements()) {
      String key = (String)keys.nextElement();
      if (key.equals(base)) {
        result.put("", propertiesClone.get(key));
      }
      else {
        if (key.startsWith(prefix)) {
          result.put(key.substring(prefix.length()), propertiesClone.get(key));
        }
      }
    }

    return result;
  }

}
