/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.config;

import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.util.resource.ResourceBundles;
import com.sapportals.wcm.util.uri.URI;
import java.io.*;
import java.net.URL;
import java.text.*;
import java.util.*;

//=============================================================================
/**
 * ConfigPropertyDef is a definition of a configuration property. Copyright (c)
 * SAP AG 2001-2002
 *
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/config/ConfigPropertyDef.java#7
 *      $
 */
//=============================================================================
public class ConfigPropertyDef {
  public final static int STRING = 0;
  public final static int INTEGER = 1;
  public final static int EXISTINGPATH = 2;
  public final static int EXISTINGFILE = 3;
  public final static int JAVACLASS = 4;
  public final static int URL = 5;
  public final static int URI = 6;
  public final static int BOOLEAN = 7;

  // Resource bundle access key constants
  final static String ERR_NOT_BOOLEAN = "err_not_boolean";
  final static String ERR_NOT_INTEGER = "err_not_integer";
  final static String ERR_CLASS_NOT_FOUND = "err_class_not_found";
  final static String ERR_FILE_NOT_EXISTS = "err_file_not_exists";
  final static String ERR_FILE_NOT_READABLE = "err_file_not_readable";
  final static String ERR_NOT_FILE = "err_not_file";
  final static String ERR_NOT_DIRECTORY = "err_not_directory";
  final static String ERR_REQUIRED = "err_required";
  final static String ERR_MALFORMED_URI = "err_malformed_uri";
  final static String ERR_MALFORMED_URL = "err_malformed_url";

  static ResourceBundles resbund = ResourceBundles.getBundle("com.sapportals.wcm.util.config.ConfigPropertyDefRes");

  String m_name;
  String m_displayName;
  int m_type;
  boolean m_required;
  String m_defaultValue;

  //---------------------------------------------------------------------------
  /**
   * Create a ConfigPropertyDef.
   *
   * @param name The name (relativ path) of the property
   * @param displayName The display name of the property
   * @param type The type of the property. Valid Types are:
   *      <ul>
   *        <li> STRING: A string </li>
   *        <li> INTEGER: An integer </li>
   *        <li> EXISTINGPATH: A path on the (local) filesystem </li>
   *        <li> EXISTINGFILE: A file on the (local) filesystem </li>
   *        <li> JAVACLASS: A java class in the (local) classpath </li>
   *        <li> URL: A fully qualified URL </li>
   *        <li> URI: An URI </li>
   *        <li> BOOLEAN: A boolean value (true/false) </li>
   * @param required true if the property is required
   * @param defaultValue The default value of the property (null if none)
   */
  //---------------------------------------------------------------------------
  public ConfigPropertyDef(String name, String displayName, int type,
    boolean required, String defaultValue) {//---------------------------------------------------------------------------

    m_name = name;
    m_displayName = displayName;
    m_type = type;
    m_required = required;
    m_defaultValue = defaultValue;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the name of the configuration property.
   *
   * @return The name of the configuration property
   */
  //---------------------------------------------------------------------------
  public String getName() {//---------------------------------------------------------------------------

    return m_name;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the display name of the configuration property.
   *
   * @return The display name of the configuration property
   */
  //---------------------------------------------------------------------------
  public String getDisplayName() {//---------------------------------------------------------------------------

    return m_displayName;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the type of the configuration property.
   *
   * @return The type of the configuration property
   */
  //---------------------------------------------------------------------------
  public int getType() {//---------------------------------------------------------------------------

    return m_type;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Chreck if the configuration property is required.
   *
   * @return true if the configuration property is required
   */
  //---------------------------------------------------------------------------
  public boolean isRequired() {//---------------------------------------------------------------------------

    return m_required;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the default value of the property.
   *
   * @return The default value of the property
   */
  //---------------------------------------------------------------------------
  public String getDefaultValue() {//---------------------------------------------------------------------------

    return m_defaultValue;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Check if a property matches its definition.
   *
   * @param collection The configuration collection to be checked
   * @return a problem report or null if the check passed
   */
  //---------------------------------------------------------------------------
  public String check(ConfigCollection collection) {//---------------------------------------------------------------------------

    String value = collection.getProperty(m_name);

    if (value == null || value.length() == 0) {
      if (m_required) {
        return MessageFormat.format(resbund.getString(ERR_REQUIRED),
          new Object[]{m_displayName});
      }
      else {
        return null;
      }
    }

    if (value != null) {
      File file;

      switch (m_type) {
        case EXISTINGFILE:
          file = new File(value);

          if (!file.exists()) {
            return MessageFormat.format(resbund.getString(ERR_FILE_NOT_EXISTS),
              new Object[]{m_displayName, value});
          }
          else if (!file.isFile()) {
            return MessageFormat.format(resbund.getString(ERR_NOT_FILE),
              new Object[]{m_displayName, value});
          }
          else if (!file.canRead()) {
            return MessageFormat.format(resbund.getString(ERR_FILE_NOT_READABLE),
              new Object[]{m_displayName, value});
          }
          return null;
        case EXISTINGPATH:
          file = new File(value);

          if (!file.exists()) {
            return MessageFormat.format(resbund.getString(ERR_FILE_NOT_EXISTS),
              new Object[]{m_displayName, value});
          }
          else if (!file.isDirectory()) {
            return MessageFormat.format(resbund.getString(ERR_NOT_DIRECTORY),
              new Object[]{m_displayName, value});
          }
          else if (!file.canRead()) {
            return MessageFormat.format(resbund.getString(ERR_FILE_NOT_READABLE),
              new Object[]{m_displayName, value});
          }
          return null;
        case JAVACLASS:
          try {
            if (CrtClassLoaderRegistry.forName(value) != null) {
              return null;
            }
          }
          catch (Exception e) {
            com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
          }
          return MessageFormat.format(resbund.getString(ERR_CLASS_NOT_FOUND),
            new Object[]{m_displayName, value});
        case BOOLEAN:
          if (value.equals("true") || value.equals("false")) {
            return null;
          }
          return MessageFormat.format(resbund.getString(ERR_NOT_BOOLEAN),
            new Object[]{m_displayName, value});
        case INTEGER:
          try {
            Integer.parseInt(value);
            return null;
          }
          catch (Exception e) {
            com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
          }
          return MessageFormat.format(resbund.getString(ERR_NOT_INTEGER),
            new Object[]{m_displayName, value});
        case URL:
          try {
            URL url = new URL(value);
            return null;
          }
          catch (Exception e) {
            com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
          }
          return MessageFormat.format(resbund.getString(ERR_MALFORMED_URL),
            new Object[]{m_displayName, value});
        case URI:
          try {
            URI uri = new URI(value);
            return null;
          }
          catch (Exception e) {
            com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(e.getMessage());
          }
          return MessageFormat.format(resbund.getString(ERR_MALFORMED_URI),
            new Object[]{m_displayName, value});
      }
    }
    return null;
  }
  //---------------------------------------------------------------------------

}
