/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.config;

import java.util.*;

//=============================================================================
/**
 * A ConfigCollection is a hirarchical representation of a configuration file.
 * Copyright (c) SAP AG 2001-2002
 *
 * @version $Id: //javabas/com.sapportals.wcm/dev/src/java/util/api/com/sapportals/wcm/util/config/ConfigCollection.java#5
 *      $
 */
//=============================================================================
public class ConfigCollection {
  String m_Name;
  Map m_Children;
  Map m_Properties;

  //---------------------------------------------------------------------------
  /**
   * Create a new ConfigCollection.
   *
   * @param name the name of the ConfigCollection (the root-node has name = null
   *      )
   */
  //---------------------------------------------------------------------------
  public ConfigCollection(String name) {//---------------------------------------------------------------------------

    m_Name = name;
    m_Children = new TreeMap();
    m_Properties = new TreeMap();
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get the name of the ConfigCollection. The root-node has name = null.
   *
   * @return The name of the ConfigCollection
   */
  //---------------------------------------------------------------------------
  public String getName() {//---------------------------------------------------------------------------

    return m_Name;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Add a child ConfigCollection.
   *
   * @param configCollection The child to be added
   */
  //---------------------------------------------------------------------------
  public void addChild(ConfigCollection configCollection) {//---------------------------------------------------------------------------

    m_Children.put(configCollection.getName(), configCollection);
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Remove a child ConfigCollection.
   *
   * @param name TBD: Description of the incoming method parameter
   */
  //---------------------------------------------------------------------------
  public void removeChild(String name) {//---------------------------------------------------------------------------

    m_Children.remove(name);
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get a child ConfigCollection by its name.
   *
   * @param name The name of the child
   * @return The child ConfigCollection or null if none is found
   */
  //---------------------------------------------------------------------------
  public ConfigCollection getChild(String name) {//---------------------------------------------------------------------------

    return (ConfigCollection)m_Children.get(name);
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get a ConfigCollection by its path in dot-notation.
   *
   * @param path The path in dot-notation
   * @return null if none is found
   */
  //---------------------------------------------------------------------------
  public ConfigCollection getByPath(String path) {//---------------------------------------------------------------------------

    if (path == null) {
      return this;
    }

    StringTokenizer t = new StringTokenizer(path, "./");
    ConfigCollection collection = this;

    while (t.hasMoreTokens()) {
      String token = t.nextToken();

      collection = collection.getChild(token);
      if (collection == null) {
        collection = new ConfigCollection(token);
      }
    }

    return collection;
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Set/Replace a child ConfigCollection by its relative path in dot-notation.
   *
   * @param path The path in dot-notation
   * @param configCollection The child ConfigCollection to be set
   */
  //---------------------------------------------------------------------------
  public void setByPath(String path, ConfigCollection configCollection) {//---------------------------------------------------------------------------

    StringTokenizer t = new StringTokenizer(path, "./");
    ConfigCollection collection = this;
    ConfigCollection next;

    while (t.hasMoreTokens()) {
      String token = t.nextToken();

      if (t.hasMoreTokens()) {
        next = collection.getChild(token);
        if (next == null) {
          next = new ConfigCollection(token);
          collection.addChild(next);
        }
        collection = next;
      }
      else {
        collection.addChild(configCollection);
      }
    }
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get all children of the ConfigCollection.
   *
   * @return A Collection of ConfigCollections
   */
  //---------------------------------------------------------------------------
  public Collection getChildren() {//---------------------------------------------------------------------------

    return m_Children.values();
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Set/add a property to the ConfigCollection.
   *
   * @param name The name of the property (dot-notation will be resolved)
   * @param value The value of the property
   */
  //---------------------------------------------------------------------------
  public void putProperty(String name, String value) {//---------------------------------------------------------------------------

    if (name.indexOf(".") < 0) {
      m_Properties.put(name, value);
    }
    else {
      StringTokenizer t = new StringTokenizer(name, ".");
      ConfigCollection collection = this;
      ConfigCollection next;

      while (collection != null && t.hasMoreTokens()) {
        String token = t.nextToken();
        if (t.hasMoreTokens()) {
          next = collection.getChild(token);
          if (next == null) {
            next = new ConfigCollection(token);
            collection.addChild(next);
          }
          collection = next;
        }
        else {
          collection.m_Properties.put(token, value);
        }
      }
    }
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Remove a property from the ConfigCollection
   *
   * @param name The name of the property to by removed (dot-notation will be
   *      resoved )
   */
  //---------------------------------------------------------------------------
  public void removeProperty(String name) {//---------------------------------------------------------------------------

    if (name.indexOf(".") < 0) {
      m_Properties.remove(name);
    }
    else {
      StringTokenizer t = new StringTokenizer(name, ".");
      ConfigCollection collection = this;
      ConfigCollection next;

      while (collection != null && t.hasMoreTokens()) {
        String token = t.nextToken();
        if (t.hasMoreTokens()) {
          next = collection.getChild(token);
          if (next == null) {
            next = new ConfigCollection(token);
            collection.addChild(next);
          }
          collection = next;
        }
        else {
          collection.m_Properties.remove(token);
        }
      }
    }
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get all property names in this ConfigCollection.
   *
   * @return A Collection of property names
   */
  //---------------------------------------------------------------------------
  public Collection getPropertyNames() {//---------------------------------------------------------------------------

    return m_Properties.keySet();
  }
  //---------------------------------------------------------------------------

  //---------------------------------------------------------------------------
  /**
   * Get a property by its name.
   *
   * @param name The name of the property (dot-notation will by resolved)
   * @return The value of the property or null if none is found
   */
  //---------------------------------------------------------------------------
  public String getProperty(String name) {//---------------------------------------------------------------------------

    if (name.indexOf(".") < 0) {
      return (String)m_Properties.get(name);
    }
    else {
      StringTokenizer t = new StringTokenizer(name, ".");
      ConfigCollection collection = this;

      while (collection != null && t.hasMoreTokens()) {
        String token = t.nextToken();
        if (t.hasMoreTokens()) {
          collection = collection.getChild(token);
        }
        else {
          return (String)collection.m_Properties.get(token);
        }
      }
    }
    return null;
  }
  //---------------------------------------------------------------------------
}
