/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels;
import com.sap.tc.logging.Location;
import com.sapportals.portal.security.usermanagement.IGroup;
import com.sapportals.portal.security.usermanagement.IGroupFactory;
import com.sapportals.portal.security.usermanagement.IRole;
import com.sapportals.portal.security.usermanagement.IRoleFactory;

import com.sapportals.portal.security.usermanagement.IUMPrincipal;
import com.sapportals.portal.security.usermanagement.IUser;
import com.sapportals.portal.security.usermanagement.IUserFactory;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.util.factories.AbstractObjectFactory;
import com.sapportals.wcm.util.factories.ReflectionUtils;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.usermanagement.WPUMFactory;

import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;

import java.util.Properties;
import java.util.StringTokenizer;

/**
 * A factory to retrieve <code>IRecipient</code> s. <br>
 *
 *
 * @see IRecipient
 */
public class RecipientFactory extends AbstractObjectFactory {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * the unique id for the factory.
   */
  private final static String ID = "recipients";

  /**
   * the unique id for the factory.
   */
  private final static String RECIPIENT_ID = "Recipient";

  /**
   * the unique id for the factory.
   */
  private final static String RECIPIENTLIST_ID = "RecipientList";

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.channels.RecipientFactory.class);

  private Constructor m_RecipientUserConstructor = null;
  private Constructor m_RecipientGroupConstructor = null;
  private Constructor m_RecipientRoleConstructor = null;

  private Constructor m_RecipientListConstructor = null;

  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  // ------------------------------------------------------------------------
  /**
   * Create a recipient from a given id and user.
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      user-id).
   * @param user an <code>IUser</code> with the user object for the recipient.
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private IRecipient newRecipient(String id, IUser user) throws WcmException {

    Object[] parameters = new Object[] { id, user };

    return (IRecipient)ReflectionUtils.newObject(RECIPIENT_ID, m_RecipientUserConstructor, parameters);
  }

  // ------------------------------------------------------------------------
  /**
   * Create a recipient from a given id and group.
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      group-id).
   * @param group an <code>IGroup</code> with the group object for the
   *      recipient.
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private IRecipient newRecipient(String id, IGroup group) throws WcmException {

    Object[] parameters = new Object[] { id, group };

    return (IRecipient)ReflectionUtils.newObject(RECIPIENT_ID, m_RecipientGroupConstructor, parameters);
  }

  // ------------------------------------------------------------------------
  /**
   * Create a recipient from a given id and role.
   *
   * @param id a <code>String</code> with the system id of the recipient (the
   *      role-id).
   * @param role an <code>IRole</code> with the role object for the recipient.
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private IRecipient newRecipient(String id, IRole role) throws WcmException {

    Object[] parameters = new Object[] { id, role };

    return (IRecipient)ReflectionUtils.newObject(RECIPIENT_ID, m_RecipientRoleConstructor, parameters);
  }

  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  /**
   * Create a <code>RecipientFactory</code> with a given id.<br>
   * <i>Note</i> : This constructor has to be public (although it should be
   * protected), to allow dynamic creation via <code>AbstractClassFactory()
   * </code>.
   *
   * @param id a <code>String</code> with the unique id for the factory.
   * @exception WcmException Exception raised in failure situation
   */
  public RecipientFactory(String id) throws WcmException {

    super("RecipientFactory", id);

    initialize();

  }

  // --------------------
  // AbstractClassFactory -----------------------------------------------------
  // --------------------

  private void initialize() throws WcmException {

    try {
      Class recipientClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.util.channels.wcm.Recipient");

      m_RecipientUserConstructor = ReflectionUtils.loadConstructor(RECIPIENT_ID, recipientClass, new Class[] { String.class, IUser.class });
      m_RecipientGroupConstructor = ReflectionUtils.loadConstructor(RECIPIENT_ID, recipientClass, new Class[] { String.class, IGroup.class });
      m_RecipientRoleConstructor = ReflectionUtils.loadConstructor(RECIPIENT_ID, recipientClass, new Class[] { String.class, IRole.class });

      Class recipientListClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.util.channels.wcm.RecipientList");

      m_RecipientListConstructor = ReflectionUtils.loadConstructor(RECIPIENT_ID, recipientListClass, new Class[] {
      });
    } catch (ClassNotFoundException x) {
      throw new WcmException(x);
    }
  }

  // -------------- x
  /**
   * Get an instance of the <code>RecipientFactory</code> .
   *
   * @return an instance of the <code>RecipientFactory</code> .
   * @throws WcmException if no factory instance is available.
   */
  public static RecipientFactory getInstance() throws WcmException {

    return (RecipientFactory)getFactoryInstance(ID, RecipientFactory.class, SINGLETON);
  }

  /**
   * Get an <code>IRecipient</code> for the given user-management object.
   *
   * @param principal a <code>IUMPrincipal</code> with user-management object
   *      (an <code>IUser</code> , <code>IGroup</code> or <code>IRole</code> ).
   * @return an <code>IRecipient</code> for the given user-management object,
   *      <code>null</code> if the recipient does not exist.
   * @throws WcmException if the recipient could not be retrieved.
   */
  public IRecipient getRecipient(IUMPrincipal principal) throws WcmException {

    if (principal == null) {
      return null;
    }

    return getRecipient(principal.getId(), principal.getType());
  }

  /**
   * Get a <code>IRecipient</code> for the given id and type.
   *
   * @param id a <code>String</code> with the <code>IRecipient</code> 's system
   *      id.
   * @param type an <code>int</code> with the type of the recipient (<code>IUSER
   *      </code>, <code>IGROUP</code> or <code>IROLE</code> , see <i>
   *      com.sap.wp.security.IUMPrincipal</i> ).
   * @return an <code>IRecipient</code> for the given id and type, <code>null
   *      </code> if the recipient does not exist.
   * @throws WcmException if the recipient could not be retrieved.
   */
  public IRecipient getRecipient(String id, int type) throws WcmException {

    try {

      switch (type) {

        case IUMPrincipal.IUSER :
          {
            IUserFactory userFactory = WPUMFactory.getUserFactory();
            IUser user = userFactory.getUser(id);
            if (user != null) {
              return newRecipient(id, user);
            }
          }
          break;
        case IUMPrincipal.IGROUP :
          {
            IGroupFactory groupFactory = WPUMFactory.getGroupFactory();
            IGroup group = groupFactory.getGroup(id);
            if (group != null) {
              return newRecipient(id, group);
            }
          }
          break;
        case IUMPrincipal.IROLE :
          {
            IRoleFactory roleFactory = WPUMFactory.getRoleFactory();
            IRole role = roleFactory.getRole(id);
            if (role != null) {
              return newRecipient(id, role);
            }
          }
          break;
        default :
          {
          }

      }

      g_Log.warningT("getRecipient(287)", "RecipientFactory: unable to create recipient id=" + id + " type=" + type);

      return null;
    } catch (Exception e) {
      String message = "RecipientFactory: caught exception while creating recipient id=" + id + " type=" + type + ": " + e.getMessage();
      g_Log.debugT("getRecipient(295)", message);
      throw new WcmException(message, e);
    }

  }

  /**
   * Create an empty <code>IRecipientList</code> .
   *
   * @return an empty <code>IRecipientList</code> .
   * @throws WcmException if the list could not be created.
   */
  public IRecipientList createRecipientList() throws WcmException {

    return (IRecipientList)ReflectionUtils.newObject(RECIPIENTLIST_ID, m_RecipientListConstructor, new Object[] {
    });
  }

}
