/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels;
import com.sapportals.portal.security.usermanagement.IUMPrincipal;

import com.sapportals.wcm.WcmException;

import java.lang.String;
import java.util.Locale;
import javax.mail.internet.MimeMessage;

/**
 * Recipients are objects, where messages can be sent to via a channel. <br>
 * A recipient represents an (addressable) person or a distribution list. A
 * recipient has adresses, one for each of the several channels it can receive
 * messages from. For example, recipient A has an email address for the email
 * channel and a mobile phone number for the sms-channel. <br>
 * For the moment, a recipient is a simple mapping of users, groups and roles to
 * recipients and distrubtion lists.
 *
 * @see RecipientFactory
 * @see IChannel
 */
public interface IRecipient {

  /**
   * Get the recipient's type.
   *
   * @return a <code>int</code> with the recipient's type (for example <code>
   *      IUMPrincipal.IUSER</code> ).
   * @see RecipientFactory#getRecipient(String, int)
   */
  public int getType();


  /**
   * Get the recipient's unique system id.
   *
   * @return a <code>String</code> with the recipient's unique system id.
   */
  public String getID();


  /**
   * Get the recipient's displayable name.
   *
   * @return a <code>String</code> with a displayable name.
   */
  public String getDescription();


  /**
   * Get the recipient's displayable name for a specific locale.
   *
   * @param locale a <code>Locale</code> defining the language to get the name
   *      for.
   * @return a <code>String</code> with a displayable name for the given locale.
   */
  public String getDescription(Locale locale);


  /**
   * Get the recipient's locale.
   *
   * @return a <code>Locale</code> with the recipient's locale.
   * @throws WcmException if the locale data cannot be retrieved.
   */
  public Locale getLocale()
    throws WcmException;


  /**
   * Get the associated user object (<code>IUser</code> , <code>IGroup</code> or
   * <code>IRole</code> ).
   *
   * @return an <code>IUMPrincipal</code> as the user-/group-/role-object
   *      associated to this recipient, or <code>null</code> if this recipient
   *      has no user/group/role associated.
   * @throws WcmException if the operation failed (if there was an
   *      error while obtaining the recipient's userobject).
   */
  public IUMPrincipal getUserManagementObject()
    throws WcmException;


  /**
   * Test if the recipient is a group and contains other recipients.
   *
   * @return a <code>boolean</code> <code>true</code> , if the recipient is a
   *      distribution list (user group or user role), <code>false</code> if
   *      it's a single person (user).
   * @throws WcmException if the type for this recipient can not
   *      be resolved.
   */
  public boolean isGroup()
    throws WcmException;


  /**
   * Get all the members of a group or role.
   *
   * @return a <code>IRecipientList</code> with the members for this group,
   *      <code>null</code> if this recipient is not a group or role.
   * @throws WcmException if the members can not be retrieved.
   */
  public IRecipientList getGroupMembers()
    throws WcmException;


  /**
   * Test if a given recipient is a member of a group.
   *
   * @param recipient TBD: Description of the incoming method parameter
   * @return a <code>boolean</code> <code>true</code> if the given recipient is
   *      a member of this group or role, <code>false</code> if this recipient
   *      is not a group, the given recipient is a role or the given recipient
   *      is not a member of this group or role.
   * @throws WcmException if the members can not be checked.
   */
  public boolean contains(IRecipient recipient)
    throws WcmException;


  /**
   * Get the recipient's channel for a given channel id.
   *
   * @param id a <code>String</code> with the desired channel's system id.
   * @return an <code>IChannel</code> for the recipient with the given id,
   *      <code>null</code> if the channel is not available for this recipient.
   * @throws WcmException if the operation failed (if there was an
   *      error while resolving the recipients address).
   */
  public IChannel getChannel(String id)
    throws WcmException;


  /**
   * Get all available channels.
   *
   * @return a <code>IChannelList</code> with all available channels (may be
   *      empty) for this recipient.
   * @throws WcmException if the list cannot be retrieved.
   */
  public IChannelList getChannels()
    throws WcmException;


  /**
   * Get the default channel.
   *
   * @return an <code>IChannel</code> with the default channel, <code>null
   *      </code> if there is no default channel defined for this recipient.
   * @throws WcmException if the operation failed (an error
   *      occured while obtaining the recipients data).
   */
  public IChannel getDefaultChannel()
    throws WcmException;


  /**
   * Send a message text via the specified channel.
   *
   * @param channel an <code>IChannel</code> with the channel to use.
   * @param from a <code>String</code> with the sender's address in the proper
   *      format for the channel.
   * @param title a <code>String</code> with the title for the message.
   * @param messagetext a <code>String</code> with the messagetext to send.
   * @param type a <code>String</code> with the type of message, for example
   *      <code>text/plain</code> .
   * @throws WcmException if the message cannot be sent.
   */
  public void sendTo(IChannel channel,
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException;


  /**
   * Send a mime message via the specified channel.
   *
   * @param channel an <code>IChannel</code> with the channel to use.
   * @param from a <code>String</code> with the sender's address in the
   *      appropiate format for the channel.
   * @param mimemessage a <code>MimeMessage</code> with the mimemessage to send.
   * @throws WcmException if the message cannot be sent.
   */
  public void sendTo(IChannel channel,
    String from,
    MimeMessage mimemessage)
    throws WcmException;

}
