/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels;

import com.sapportals.wcm.WcmException;

import java.lang.String;
import java.util.Locale;
import javax.mail.internet.MimeMessage;

/**
 * A channel is the abstraction of an transport mechanism for sending messages
 * to recipients. <br>
 * A message can be a simple <code>String</code> or an internet <code>
 * MimeMessage</code> . A recipient can be something like an user or a
 * distribution list.
 *
 * @see ChannelFactory
 * @see IRecipient
 * @see javax.mail.internet.MimeMessage
 */
public interface IChannel {

  /**
   * Get the channel's type.
   *
   * @return a <code>String</code> with the channel's type, for example <code>
   *      EMAIL</code> .
   * @see ChannelFactory
   */
  public String getType();


  /**
   * Get the channel's unique system id.
   *
   * @return a <code>String</code> with the channel's system id.
   */
  public String getID();


  /**
   * Get the channel's displayable name.
   *
   * @return a <code>String</code> with the channel's displayable name.
   */
  public String getDescription();


  /**
   * Get the channel's displayable name for a specific locale.
   *
   * @param locale a <code>Locale</code> defining the language to get the name
   *      for.
   * @return a <code>String</code> with the channel's displayable name in the
   *      given locale.
   */
  public String getDescription(Locale locale);


  /**
   * Get the given recipient's channel address in the channel's specific format.
   *
   * @param recipient an <code>IRecipient</code> to get the channel's address
   *      for.
   * @return a <code>String</code> with the address of the given recipient in
   *      the channel's specific format; <code>null</code> if the given
   *      recipient has no address for this channel.
   * @throws <code> WcmException</code> if there was an error while resolving
   *      the recipient's address for this channel.
   */
  public String getRecipientAddress(IRecipient recipient)
    throws WcmException;


  /**
   * Get an empty internet mime message.
   *
   * @return an empty <code>MimeMessage</code> (see <i>
   *      javax.mail.internet.MimeMessage</i> ).
   * @throws <code> WcmException</code> if there was an error while creating the
   *      mime message.
   */
  public MimeMessage getEmptyMessage()
    throws WcmException;


  /**
   * Send a message text via this channel to the given address.
   *
   * @param to a <code>String</code> with the address of the receiver in the
   *      channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param title a <code>String</code> with the title for the message.
   * @param messagetext a <code>String</code> with the content of the message to
   *      send.
   * @param type a <code>String</code> with the mimetype of message, for example
   *      <code>text/plain</code> .
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to,
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException;


  /**
   * Send a message text via this channel to the given address.
   *
   * @param to a <code>String</code> with the address of the receiver in the
   *      channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param title a <code>String</code> with the title for the message.
   * @param messagetext a <code>String</code> with the content of the message to
   *      send.
   * @param type a <code>String</code> with the mimetype of message, for example
   *      <code>text/plain</code> .
   * @param encoding a <code>String</code> with the encoding for the message
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to,
    String from,
    String title,
    String messagetext,
    String type,
    String encoding)
    throws WcmException;

  /**
   * Send a message text via this channel to the given addresses.
   *
   * @param to an array of <code>String<code>s with the addresses of the
   *      receivers in the channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param title a <code>String</code> with the title for the message.
   * @param messagetext a <code>String</code> with the content of the message to
   *      send.
   * @param type a <code>String</code> with the mimetype of message, for example
   *      <code>text/plain</code> .
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to[],
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException;

  /**
   * Send a message text via this channel to the given addresses.
   *
   * @param to an array of <code>String<code>s with the addresses of the
   *      receivers in the channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param title a <code>String</code> with the title for the message.
   * @param messagetext a <code>String</code> with the content of the message to
   *      send.
   * @param type a <code>String</code> with the mimetype of message, for example
   *      <code>text/plain</code> .
   * @param encoding a <code>String</code> with the encoding for this message
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to[],
    String from,
    String title,
    String messagetext,
    String type,
    String encoding)
    throws WcmException;

  /**
   * Send a mime message via this channel to the given address.
   *
   * @param to a <code>String</code> with the address of the receiver in the
   *      channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param mimemessage an internet <code>MimeMessage</code> with the content of
   *      the message to send (see <i>javax.mail.internet.MimeMessage</i> ).
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to,
    String from,
    MimeMessage mimemessage)
    throws WcmException;


  /**
   * Send a mime message via this channel to the given addresses.
   *
   * @param to an array of <code>String</code> s with the addresses of the
   *      receivers in the channel's specific format.
   * @param from a <code>String</code> with the address of the sender in the
   *      channel's specific format.
   * @param mimemessage an internet <code>MimeMessage</code> with the content of
   *      the message to send (see <i>javax.mail.internet.MimeMessage</i> ).
   * @throws <code> WcmException</code> if there was an error while sending the
   *      message.
   * @throws <code>PartialSendException</code> if the message was sent only
   *        parially (to some recipients).
   */
  public void sendTo(String to[],
    String from,
    MimeMessage mimemessage)
    throws WcmException;

}
