/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels;

import java.lang.reflect.Constructor;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;

import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.config.fwk.IConfigurable;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.crt.CrtClassLoaderRegistry;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.factories.AbstractObjectFactory;
import com.sapportals.wcm.util.factories.ReflectionUtils;

/**
 * A factory to retrieve <code>IChannel</code> s. <br>
 *
 *
 * @see IChannel
 */
public class ChannelFactory
   extends AbstractObjectFactory {

  private final static String CFG_PLUGIN_CM_UTILITIES_CHANNELS = "/cm/utilities/channels";
     
  // ---------
  // Constants ----------------------------------------------------------------
  // ---------

  /**
   * channel type EMAIL.
   */
  public final static String EMAIL = "EMAIL";

  /**
   * channel type SMS.
   */
  public final static String SMS = "SMS";

  /**
   * channel type FAX.
   */
  public final static String FAX = "FAX";

  /**
   * channel type STORE.
   */
  public final static String STORE = "STORE";

  /**
   * channel type NULL. Messages send to this channel are not send at all.
   */
  public final static String NULL_CHANNEL = "NULL";

  // ------------------------------------------------------------------------

  /**
   * the unique id for the factory.
   */
  private final static String ID = "channel";
  
  /**
   * the name of the channel list
   */
  private final static String CHANNELLIST_ID = "ChannelList";

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  private static com.sap.tc.logging.Location g_Log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.channels.ChannelFactory.class);

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  private Constructor m_ChannelListConstructor = null;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  // ------------------------------------------------------------------------
  /**
   * Create a ChannelFactory with a given id.<br>
   * <i>Note</i> : This constructor has to be public (although it should be
   * protected), to allow dynamic creation via the <code>AbstractClassFactory()
   * </code>.
   *
   * @param id a <code>String</code> with the unique id for the factory.
   * @exception WcmException Exception raised in failure situation
   */
  public ChannelFactory(String id)
    throws WcmException {

    super("ChannelFactory", id);

    initialize();

  }


  // ---------------
  // Private Methods ----------------------------------------------------------
  // ---------------

  // ------------------------------------------------------------------------
  private void initialize()
    throws WcmException {

    try {
      g_Log.debugT("initialize(137)", "ChannelFactory starting");
      IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
      IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfg.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_CHANNELS);
      if (plugin == null) {
        throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_UTILITIES_CHANNELS);
      }
      
      IConfigurable[] configurables = plugin.getConfigurables();
  
      loadObjectMap(configurables);

      Class channelListClass = CrtClassLoaderRegistry.forName("com.sapportals.wcm.util.channels.wcm.ChannelList");
      m_ChannelListConstructor = ReflectionUtils.loadConstructor(CHANNELLIST_ID, channelListClass, new Class[]{});
    }
    catch (WcmException e) {
      throw e;
    }
    catch (Exception e) {
      throw new WcmException(e);
    }
  }


  // --------------
  // Public Methods -----------------------------------------------------------
  // --------------

  // ------------------------------------------------------------------------
  /**
   * Get a new instance of the factory.
   *
   * @return a new instance of the <code>ChannelFactory</code> .
   * @throws WcmException if the factory can not be created.
   */
  public static ChannelFactory getInstance()
    throws WcmException {

    return (ChannelFactory)getFactoryInstance(ID, ChannelFactory.class, SINGLETON);
  }


  // --------------------
  // AbstractClassFactory -----------------------------------------------------
  // --------------------

  protected ObjectClassEntry newObjectListEntry(String id,
    Constructor constructor,
    Properties properties,
    boolean singletonFlag) {

    return super.newObjectListEntry(id, constructor, properties, true);
  }


  // ------------------------------------------------------------------------
  /**
   * Get an <code>IChannel</code> instance for a specific id.
   *
   * @param id TBD: Description of the incoming method parameter
   * @return an <code>IChannel</code> with the channel for the given id, <code>
   *      null</code> if no channel exists for the given id.
   * @throws WcmException if there an error occured while creating
   *      the channel.
   */
  public IChannel getChannel(String id)
    throws WcmException {

    return (IChannel)getObjectInstance(id);
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list of <code>IChannel</code> s for a given type.
   *
   * @param type TBD: Description of the incoming method parameter
   * @return a <code>IChannelList</code> with channels for the given type, may
   *      be empty if no channels exists for that type.
   * @throws WcmException if there an error occured while creating
   *      the channels.
   */
  public IChannelList getChannelForType(String type)
    throws WcmException {

    IChannelList channels = createChannelList();

    if (type == null) {
      return channels;
    }

    Collection allChannels = m_ObjectMap.values();
    Iterator iterator = allChannels.iterator();
    while (iterator.hasNext()) {
      IChannel channel = (IChannel)((ObjectClassEntry)iterator.next()).getObjectInstance();
      if (type.equals(channel.getType())) {
        channels.add(channel);
      }
    }

    return channels;
  }


  // ------------------------------------------------------------------------
  /**
   * Get the first instance of an <code>IChannel</code> for a given type.
   *
   * @param type TBD: Description of the incoming method parameter
   * @return an <code>IChannel</code> for the given type, <code>null</code> if
   *      there is no channel with the given type.
   * @throws WcmException if an error occured while creating the
   *      channel.
   */
  public IChannel getFirstChannelForType(String type)
    throws WcmException {

    if (type == null) {
      return null;
    }

    IChannel channel = null;

    Collection allChannels = m_ObjectMap.values();
    Iterator iterator = allChannels.iterator();
    while ((channel == null)
       && (iterator.hasNext())
      ) {
      channel = (IChannel)((ObjectClassEntry)iterator.next()).getObjectInstance();
      if (!type.equals(channel.getType())) {
        channel = null;
      }
    }

    return channel;
  }


  // ------------------------------------------------------------------------
  /**
   * Get a list with all available <code>IChannel</code> s.
   *
   * @return an <code>IChannelList</code> which may be empty if no channels
   *      exist.
   * @throws WcmException if there an error occured while creating
   *      the channel.
   */
  public IChannelList getChannels()
    throws WcmException {

    IChannelList channels = createChannelList();

    Collection allChannels = m_ObjectMap.values();
    Iterator iterator = allChannels.iterator();
    while (iterator.hasNext()) {
      IChannel channel = (IChannel)((ObjectClassEntry)iterator.next()).getObjectInstance();
      channels.add(channel);
    }

    return channels;
  }


  // ------------------------------------------------------------------------
  /**
   * Utility method to convert an array of recipients to an array of addresses
   * in a specific channel format.
   *
   * @param channel an <code>IChannel</code> which defines the address format.
   * @param recipients an array of <code>IRecipient</code> s to get the
   *      addresses for.
   * @return an array of <code>String<code>s with the addresses.
   * @throws WcmException if an error occured while resolving the
   *      recipients addresses.
   */
  public static String[] getRecipientAdresses(IChannel channel,
    IRecipientList recipients)
    throws WcmException {

    if (channel == null) {
      return null;
    }
    if (recipients == null) {
      return null;
    }

    IRecipientListIterator recipientIterator = recipients.listIterator();

    int i = 0;
    int count = recipients.size();
    String[] addresses = new String[count];

    while (recipientIterator.hasNext()) {
      IRecipient recipient = recipientIterator.next();
      addresses[i++] = channel.getRecipientAddress(recipient);
    }

    return addresses;
  }


  // ------------------------------------------------------------------------
  /**
   * Create an empty <code>IChannelList</code> .
   *
   * @return an empty <code>IChannelList</code> .
   * @throws WcmException if the list could not be created.
   */
  public IChannelList createChannelList()
    throws WcmException {

    return (IChannelList)ReflectionUtils.newObject(CHANNELLIST_ID, m_ChannelListConstructor, new Object[]{});
  }

}
