/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.channels;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.util.resource.ResourceBundles;

import java.lang.String;
import java.util.Locale;
import java.util.Properties;

import javax.mail.Session;
import javax.mail.internet.MimeMessage;

/**
 * Base class for <code>IChannel</code> -implementations. <br>
 * The <code>AbstractChannel</code> implements some generic behavior for loading
 * configuration data and retrieving a standard channel description from the
 * channel's type.
 *
 * @see IChannel
 */
public abstract class AbstractChannel
   implements IChannel {

  // ---------
  // Constants ----------------------------------------------------------------
  // ---------


  /**
   * encoding to use as default for messages. The default value given can be
   * overridden by the caller
   */
  public final static String ENCODING = "UTF-8";

  // ----------------
  // Static Variables ---------------------------------------------------------
  // ----------------

  /**
   * log.
   */
  protected static com.sap.tc.logging.Location _log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.channels.AbstractChannel.class);

  /**
   * resource bundles.
   */
  protected static ResourceBundles _res = ResourceBundles.getBundle("com.sapportals.wcm.util.channels.wcm.Res");

  // ------------------
  // Instance Variables -------------------------------------------------------
  // ------------------

  /**
   * The system id for this channel.
   */
  protected String m_ID = null;

  /**
   * The properties for this channel.
   */
  protected Properties m_Properties = null;


  // ------------
  // Constructors -------------------------------------------------------------
  // ------------

  /**
   * Create a channel with a specific id from the given properties.
   *
   * @param id TBD: Description of the incoming method parameter
   * @param properties TBD: Description of the incoming method parameter
   * @throws WcmException if the channel could not be created.
   */
  protected AbstractChannel(String id,
    Properties properties)
    throws WcmException {
    //
    m_ID = id;
    m_Properties = properties;
  }


  // --------
  // IChannel -----------------------------------------------------------------
  // --------

  /**
   * Get the type for this channel.
   *
   * @return <code>String</code> is the channel's type (see ChannelFactory).
   */
  public abstract String getType();

  /**
   * Checks if this channel is equal to a given channel.
   *
   * @param object TBD: Description of the incoming method parameter
   * @return <code>true</code> if both channels are equal; <code>false</code> if
   *      not.
   */
  public boolean equals(Object object) {

    if (object == null) {
      return false;
    }

    if (object instanceof IChannel) {
      IChannel other = (IChannel)object;
      return m_ID.equals(other.getID());
    }

    return false;
  }

  /**
   * Get the system id for this channel.
   *
   * @return <code>String</code> is the channel's id.
   */
  public String getID() {
    return m_ID;
  }


  /**
   * Get the displayable name.
   *
   * @return <code>String</code> is the channel's name.
   */
  public String getDescription() {
    return _res.getString("Channel." + getType());
  }


  /**
   * Get the displayable name for the given locale.
   *
   * @param locale TBD: Description of the incoming method parameter
   * @return <code>String</code> is the channel's name for the given locale.
   */
  public String getDescription(Locale locale) {
    return _res.getString("Channel." + getType(), locale);
  }


  /**
   * Get the address for the specified recipient.
   *
   * @param recipient TBD: Description of the incoming method parameter
   * @return <code>String</code> is the address of the given recipient for this
   *      channel.
   * @throws WcmException if the recipient's address for this
   *      channel can't be resolved.
   */
  public abstract String getRecipientAddress(IRecipient recipient)
    throws WcmException;


  /**
   * Get an empty mime message for this channel.
   *
   * @return an empty <code>MimeMessage</code> for this channel.
   * @throws WcmException if there was an error while creating the
   *      mime message.
   */
  public MimeMessage getEmptyMessage()
    throws WcmException {

    String step = null;
    try {
      step = "getting system properites";
      Properties properties = System.getProperties();
      step = "getting session instance";
      Session session = Session.getInstance(properties, null);
      step = "creating mime message";
      return new MimeMessage(session);
    }
    catch (Exception e) {
      String message = "error while " + step
         + " for " + m_ID + ": " + e.getMessage();
      _log.warningT("getEmptyMessage(207)", message + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      throw new WcmException(message, e);
    }

  }

  /**
   * Send a message via this channel to the given address.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public abstract void sendTo(String address,
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException;


  /**
   * Send a message via this channel to the given addresses.
   *
   * @param addresses TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String addresses[],
    String from,
    String title,
    String messagetext,
    String type)
    throws WcmException {

    this.sendTo(addresses, from, title, messagetext, type, ENCODING);
  }

  /**
   * Send a message via this channel to the given addresses.
   *
   * @param addresses TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param title TBD: Description of the incoming method parameter
   * @param messagetext TBD: Description of the incoming method parameter
   * @param type TBD: Description of the incoming method parameter
   * @param encoding TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String addresses[],
    String from,
    String title,
    String messagetext,
    String type,
    String encoding)
    throws WcmException {
    //

    if (addresses == null) {
      throw new WcmException("sendTo: missing parameter for addresses");
    }

    for (int i = 0; i < addresses.length; i++) {
      sendTo(addresses[i], from, title, messagetext, type);
    }

  }


  /**
   * Send a mime message via this channel to the given recipient.
   *
   * @param address TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param mimemessage TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public abstract void sendTo(String address,
    String from,
    MimeMessage mimemessage)
    throws WcmException;


  /**
   * Send a mime message via this channel.
   *
   * @param addresses TBD: Description of the incoming method parameter
   * @param from TBD: Description of the incoming method parameter
   * @param mimemessage TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void sendTo(String addresses[],
    String from,
    MimeMessage mimemessage)
    throws WcmException {
    //

    if (addresses == null) {
      throw new WcmException("sendTo: missing parameter for addresses");
    }
    if (from == null) {
      throw new WcmException("sendTo: missing parameter for from");
    }

    try {
      String title = mimemessage.getSubject();
      for (int i = 0; i < addresses.length; i++) {
        MimeMessage message = new MimeMessage(mimemessage);
        message.setSubject(title);
        sendTo(addresses[i], from, message);
      }
    }
    catch (Exception e) {
      String message = "caught mail exception: " + e.getMessage();
      _log.warningT("sendTo(327)", message + " - " + com.sapportals.wcm.util.logging.LoggingFormatter.extractCallstack(e));
      throw new WcmException(message, e);
    }

  }

}
