/**
 * Copyright (c) SAP AG 2001-2002
 * @author Paul Goetz
 * @version $Id: //kmgmt/bc.util/60NW_SP_COR/src/_public/java/api/com/sapportals/wcm/util/cache/IClusterCacheArea.java#1 $
 */
package com.sapportals.wcm.util.cache;


import com.sapportals.wcm.WcmException;


/**
 * Represents a cacheable area.
 * A cacheable area is identified by its unique ID. An area is either in state
 * <i>valid</i> or <i>invalid</i>. These states are toggled by the following
 * events:
 * <table>
 *  <thead>
 *   <tr>
 *    <th>from<th>
 *    <th>event</th>
 *    <th>to</th>
 *    <th>side-effect</th>
 *   </tr>
 *  </thead>
 *  <tbody>
 *   <tr>
 *     <td><code>VALID</code><td>
 *     <td><code>validate()</code></td>
 *     <td><code>VALID</code></td>
 *     <td>initial state, calling <code>validate()</code> won't do anything.</td>
 *   </tr>
 *   <tr>
 *     <td><code>VALID</code><td>
 *     <td><code>invalidate()</code></td>
 *     <td><code>VALID</code></td>
 *     <td>calling <code>invalidate()</code> will send an <i>INVALIDATE</i> event and update the <code>getLastNodeInvalidation()</code> timestamp.</td>
 *   </tr>
 *   <tr>
 *     <td><b><code>VALID</code></b><td>
 *     <td><i>INVALIDATE</i></td>
 *     <td><b><code>INVALID</code></b></td>
 *     <td>an <i>INVALIDATE</i> event from other cluster nodes invalidates the cache area and updates the <code>getLastClusterInvalidation()</code> timestamp.</td>
 *   </tr>
 *   <tr>
 *     <td><code>INVALID</code><td>
 *     <td><i>INVALIDATE</i></td>
 *     <td><code>INVALID</code></td>
 *     <td><i>INVALIDATE</i> event in <code>INVALID</code> state just update the <code>getLastClusterInvalidation()</code> timestamp.</td>
 *   </tr>
 *   <tr>
 *     <td><code>INVALID</code><td>
 *     <td><code>invalidate()</code></td>
 *     <td><code>INVALID</code></td>
 *     <td>calling <code>invalidate()</code> will send an <i>INVALIDATE</i> event and update the <code>getLastNodeInvalidation()</code> timestamp.</td>
 *   </tr>
 *   <tr>
 *     <td><b><code>INVALID</code></b><td>
 *     <td><code>validate()</code></td>
 *     <td><b><code>VALID</code></b></td>
 *     <td>calling <code>validate()</code> toogles the state back to <code>VALID</code>.</td>
 *   </tr>
 *  <tbody>
 * </table>
 * For example used by the <code>IDMapper</code>.
 */
public interface IClusterCacheArea {

  /**
   * Get the area's ID.
   * @return a <code>String</code>, which holds the area's ID.
   */
  public String getAreaID();


  /**
   * Get the cluster's cache flag.
   * @return <code>true</code> if the system is running in cluster mode.
   */
  public boolean isCluster();


  /**
   * Get the cluster cache timeout.
   * @return a <code>long</code> with the system's cluster cache timeout in
   *        msec.
   */
  public long getClusterTimeout();


  /**
   * Mark this cache area as valid.
   * Changes the state from <code>INVALID</code> to <code>VALID</code>.
   */
  public void validate();


  /**
   * Remove all expired entries from the given cache (does a
   * <code>cache.removeEntriesOlderThan(getLastClusterInvalidation())</code>).
   * @param cache the <code>ICache</code> to remove expired entries from.
   */
  public void refresh(ICache cache)
               throws CacheException;


  /**
   * Mark this cache area as invalid.
   * Sends an <i>INVALIDATE</i> event to the other cluster nodes but doesn't
   * change the state.
   */
  public void invalidate();


  /**
   * Check, if this cache area is marked as invalid.
   * @return <code>true</code>, if state is <code>VALID</code>.
   */
  public boolean isValid();


  /**
   * Get the timestamp of the last invalidation (internally or externally, the
   * later one).
   * @return the timestamp of the last invalidation, the later timestamp from
   *        <code>getLastExternalInvalidation()</code> and
   *        <code>getLastInternalInvalidation()</code>.
   */
  public long getLastInvalidation();


  /**
   * Get the timestamp of the last (external) invalidation from other cluster
   * nodes.
   * @return a <code>long</code> with the timestamp of the last incoming
   *        <i>INVALIDATE</i> event.
   */
  public long getLastClusterInvalidation();


  /**
   * Get the timestamp of the last (internal) invalidation from this node.
   * @return a <code>long</code> with the timestamp of the last call to
   *        <code>invalidate()</code>.
   */
  public long getLastNodeInvalidation();


}
