/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache;

import java.util.*;

/**
 * Generic cache interface.
 */
public interface ICache {
  public final String DEFAULT_CACHE_TYPE = "memory";// cache type
  public final String DEFAULT_CACHE_CLASS = "com.sapportals.wcm.util.cache.memory.MemoryLRUCache";// cache class
  public final int DEFAULT_CAPACITY = 1000;// max. no. of entries in the cache
  public final long DEFAULT_MAX_CACHE_SIZE = 0;// max. size of the cache (in bytes) (=: no limit)
  public final long DEFAULT_MAX_ENTRY_SIZE = 0;// max. size of a single cache entry (in bytes)
  public final long DEFAULT_AVG_ENTRY_SIZE = 0;// average size of a single cache entry (in bytes) (MEMORY)
  public final int DEFAULT_TIME_TO_LIVE = 300;// (in seconds) (0: no expiration)
  public final String DEFAULT_STORAGE_CLASS = "com.sapportals.wcm.util.cache.persistent.filesystem.FilesystemMap";// class for persisting maps
  public final String DEFAULT_FOLDER = "/tmp";// folder (PERSISTENT)
  public final String DEFAULT_FILE_PREFIX = "cache_";// filename prefix (PERSISTENT)
  public final boolean DEFAULT_SECURE = false;// encryption? (PERSISTENT)
  public final boolean DEFAULT_CLEAR_CACHE_ON_INIT = true;// clear cache on init? (PERSISTENT)
  public final boolean DEFAULT_SINGLETON = true;// singleton?
  public final boolean DEFAULT_AUTO_DELAY_EXPIRATION = true;// auto delay expiration?

  /**
   * add a new entry to the cache
   *
   * @param entry new entry
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(ICacheEntry entry)
    throws CacheException;

  /**
   * create a new entry and add it to the cache
   *
   * @param key key of the new entry
   * @param object object of the new entry
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object)
    throws CacheException;

  /**
   * create a new entry and add it to the cache
   *
   * @param key key of the new entry
   * @param object object of the new entry
   * @param timeToLive after this time (in seconds) the entry is removed from
   *      the cache
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object, int timeToLive)
    throws CacheException;

  /**
   * create a new entry and add it to the cache - on access the expiration is
   * delayed by timeToLive
   *
   * @param key key of the new entry
   * @param object object of the new entry
   * @param timeToLive after this time (in seconds) the entry is removed from
   *      the cache
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive)
    throws CacheException;

  /**
   * create a new entry and add it to the cache
   *
   * @param key key of the new entry
   * @param object object of the new entry
   * @param timeToLive after this time (in seconds) the entry is removed from
   *      the cache
   * @param size supposed size of the object (in bytes) (may be ignored in
   *      persistent caches)
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntry(String key, Object object, int timeToLive, long size)
    throws CacheException;

  /**
   * create a new entry and add it to the cache - on access the expiration is
   * delayed by timeToLive
   *
   * @param key key of the new entry
   * @param object object of the new entry
   * @param timeToLive after this time (in seconds) the entry is removed from
   *      the cache
   * @param size supposed size of the object (in bytes) (may be ignored in
   *      persistent caches)
   * @exception CacheException Exception raised in failure situation
   */
  public void addEntryAutoDelay(String key, Object object, int timeToLive, long size)
    throws CacheException;

  /**
   * get the cache ID
   *
   * @return the cache ID
   */
  public String getID();

  /**
   * get an entry from the cache
   *
   * @param key key of the entry
   * @return entry or null if no entry with the key is stored in the cache
   * @exception CacheException Exception raised in failure situation
   */
  public ICacheEntry getEntry(String key)
    throws CacheException;

  /**
   * remove an entry from the cache
   *
   * @param entry entry to be removed
   * @return true iff the entry was found and removed
   * @exception CacheException Exception raised in failure situation
   */
  public boolean removeEntry(ICacheEntry entry)
    throws CacheException;

  /**
   * remove an entry from the cache
   *
   * @param key key of the entry
   * @return true iff an entry was found and removed
   * @exception CacheException Exception raised in failure situation
   */
  public boolean removeEntry(String key)
    throws CacheException;

  /**
   * remove all entries from the cache which's keys start with a specific prefix
   *
   * @param prefix TBD: Description of the incoming method parameter
   * @return true iff at least one entry was found and removed
   * @exception CacheException Exception raised in failure situation
   */
  public boolean removeEntriesStartingWith(String prefix)
    throws CacheException;

  /**
   * remove all entries from the cache with modification date older than the
   * given timestamp.
   * @param timestamp the compare the modification dates with
   * @return true if at least one entry was found and removed
   */
  public boolean removeEntriesOlderThan(long timestamp) throws CacheException;


  /**
   * check, whether an entry with the key is stored in the cache
   *
   * @param key key of the entry
   * @return TBD: Description of the outgoing return value
   * @exception CacheException Exception raised in failure situation
   */
  public boolean containsEntry(String key)
    throws CacheException;

  /**
   * @return enumeration of the keys of all entries in the cache
   * @exception CacheException Exception raised in failure situation
   * @deprecated as of NW04. Use keySet() instead
   */
  public Enumeration keys()
    throws CacheException;

  /**
   * Returns a setview on the current set of keys of all entries. For safe
   * iteration over the set, either synchronize the cache or make a copy of the
   * set.
   *
   * @return set view on keys of all entries in the cache
   * @exception CacheException Exception raised in failure situation
   */
  public Set keySet()
    throws CacheException;

  /**
   * @return list of all cache entries
   * @exception CacheException Exception raised in failure situation
   */
  public CacheEntryList elements()
    throws CacheException;

  /**
   * remove all entries from the cache
   *
   * @exception CacheException Exception raised in failure situation
   */
  public void clearCache()
    throws CacheException;

  /**
   * remove all expired entries
   *
   * @exception CacheException Exception raised in failure situation
   */
  public void refresh()
    throws CacheException;

  /**
   * @return the capacity of the cache
   */
  public int getCapacity();

  /**
   * @return the maximum entry size of the cache in bytes
   */
  public long getMaxEntrySize();

}
