/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.cache;

import com.sapportals.config.fwk.*;

import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.util.config.ConfigCrutch;
import com.sapportals.wcm.util.uuid.UUID;
import java.lang.reflect.*;
import java.util.*;

/**
 * Create caches (singleton)
 */
public class CacheFactory {
	public final static String CFG_CACHE_CLASS_KEY = "class"; // cache classname
	public final static String CFG_CACHE_TYPE_KEY = "type"; // cache type (memory, persistent)
	public final static String CFG_CAPACITY_KEY = "capacity"; // cache capacity (max. no. of entries)
	public final static String CFG_MAX_CACHE_SIZE_KEY = "maxsize"; // max. cache size in bytes
	public final static String CFG_AVERAGE_ENTRY_SIZE_KEY = "averageentrysize"; // average entry size in bytes
	public final static String CFG_MAX_ENTRY_SIZE_KEY = "maxentrysize"; // max. entry size in bytes
	public final static String CFG_STORAGE_CLASS_KEY = "storageclass"; // class for persisting maps
	public final static String CFG_FOLDER_KEY = "folder"; // folder for persistent cache
	public final static String CFG_FILE_PREFIX_KEY = "fileprefix"; // prefix for cache file names
	public final static String CFG_SECURE_KEY = "secure"; // secure cache?
	public final static String CFG_CLEAR_CACHE_ON_INIT_KEY = "clearcacheoninit"; // clear cache on init?
	public final static String CFG_PERSISTENT_KEY = "persistent"; // persistent?
  public final static String CFG_MEMORY_KEY = "memory"; // in memory?
  public final static String CFG_LONGKEYMEMORY_KEY = "longkeymemory"; // in memory (optimized for long)?
  public final static String CFG_STRINGKEYMEMORY_KEY = "stringkeymemory"; // in memory (optimized for string)?
	public final static String CFG_DEFAULT_TIME_TO_LIVE_KEY = "defaulttimetolive"; // default entry expiration time in seconds
	public final static String CFG_AUTO_DELAY_EXPIRATION_KEY = "autodelayexpiration"; // auto delay expiration
	public final static String CFG_SINGLETON_KEY = "singleton"; // cache is singleton (MEMORY)

	private final static String CFG_PLUGIN_CM_UTILITIES_CACHES = "/cm/utilities/caches";
	private final static String CFG_MEMORY_CACHE_CLASS_NAME = "MemoryCache";
  private final static String CFG_LONGKEYMEMORY_CACHE_CLASS_NAME = "LongKeyMemoryCache";
  private final static String CFG_STRINGKEYMEMORY_CACHE_CLASS_NAME = "StringKeyMemoryCache";
	private final static String CFG_PERSISTENT_CACHE_CLASS_NAME = "PersistentCache";

	private static com.sap.tc.logging.Location s_log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.util.cache.CacheFactory.class);
	private static CacheFactory s_cacheFactory = null;

	private Set m_configuredCaches = null;
	private Hashtable m_singletonCaches = null; // cache map by cacheID; contains all singleton caches
	private Hashtable m_allCaches = null; // cache map by cacheID; contains all caches
	private RefreshThread m_refreshThread = null;

	/**
	 * @return an instance of the cache factory. When called the first time, the
	 *      configurations of all preconfigured caches are checked for
	 *      completeness and consistency.
	 * @exception CacheException Exception raised in failure situation
	 */
	public static synchronized CacheFactory getInstance() throws CacheException {

		if (s_cacheFactory == null) {
			s_cacheFactory = new CacheFactory();
		}
		return s_cacheFactory;
	}

	/**
	 * @return a new, unique cache ID that can be used in a call of
	 *      getCache(cacheID,properties).
	 */
	public static String getUniqueCacheID() {

		return new UUID().toString();
	}

	/**
	 * Construct
	 *
	 * @exception CacheException Exception raised in failure situation
	 */
	private CacheFactory() throws CacheException {

		try {
			m_configuredCaches = new HashSet(10);
			IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
			IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
			IConfigPlugin plugin = cfg.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_CACHES);
			if (plugin == null) {
				throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_UTILITIES_CACHES);
			}

			IConfigurable[] configurables = plugin.getConfigurables();
			for (int i = 0; i < configurables.length; i++) {
				checkCacheProperties(configurables[i].getIdValue(), getCacheProperties(configurables[i]));
				m_configuredCaches.add(configurables[i].getIdValue());
			}

			m_singletonCaches = new Hashtable();
			m_allCaches = new Hashtable();
			m_refreshThread = new RefreshThread();

			startRefreshThread();
		} catch (CacheException e) {
			throw e;
		} catch (Exception e) {
			throw new CacheException(e);
		}
	}

  /**
   * @param cacheID TBD: Description of the incoming method parameter
   * @return a preconfigured cache.
   * @exception CacheException Exception raised in failure situation
   */
  public ICache getCache(String cacheID) throws CacheException {
    return getCache(cacheID, cacheID, true);
  }

  /**
   * @param configID TBD: Description of the incoming method parameter
   * @param cacheID TBD: Description of the incoming method parameter
   * @return a preconfigured cache.
   * @exception CacheException Exception raised in failure situation
   */
  public ICache getCache(String configID, String cacheID) throws CacheException {
    return getCache(configID, cacheID, true);
  }

	/**
	 * @param cacheID TBD: Description of the incoming method parameter
	 * @return a preconfigured cache.
	 * @exception CacheException Exception raised in failure situation
	 */
	public ICache getCache(String cacheID, boolean useDefault) throws CacheException {
		return getCache(cacheID, cacheID, useDefault);
	}

	/**
	 * @param configID TBD: Description of the incoming method parameter
	 * @param cacheID TBD: Description of the incoming method parameter
	 * @return a preconfigured cache.
	 * @exception CacheException Exception raised in failure situation
	 */
	public synchronized ICache getCache(String configID, String cacheID, boolean useDefault) throws CacheException {

		if (cacheID == null || configID == null) {
			throw new IllegalArgumentException();
		}

		try {
			configID = configID.trim();
			cacheID = cacheID.trim();
			Object cachedCache = m_singletonCaches.get(cacheID);
			if (cachedCache != null) {
				return (ICache) cachedCache;
			}

			Properties cacheProperties = null;
			IConfigClientContext context = IConfigClientContext.createContext(ConfigCrutch.getConfigServiceUser());
			IConfigManager cfg = Configuration.getInstance().getConfigManager(context);
			IConfigPlugin plugin = cfg.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_CACHES);
			if (plugin == null) {
				throw new RuntimeException("missing plugin " + CFG_PLUGIN_CM_UTILITIES_CACHES);
			}

			IConfigurable configurable = plugin.getConfigurable(configID);
			if (configurable == null) {
				if (useDefault)
        {
          s_log.warningT("getCache(152)", "cache " + configID + " is not configured and will use default properties");
          cacheProperties = getDefaultProperties();
        }
        else
        {
          return null;
        }
			} else {
				cacheProperties = getCacheProperties(configurable);
			}
			String className = cacheProperties.getProperty(CFG_CACHE_CLASS_KEY).trim();

			Class cacheClass = CrtClassLoaderRegistry.forName(className);
			Class[] paramTypes = { String.class, Properties.class };
			Constructor cacheConstructor = cacheClass.getConstructor(paramTypes);
			Object[] initArgs = { cacheID, cacheProperties };
			ICache cache = (ICache) cacheConstructor.newInstance(initArgs);

			if (isPersistent(cacheProperties) || (isMemory(cacheProperties) && isSingleton(cacheProperties))) {
				m_singletonCaches.put(cacheID, cache);
				m_allCaches.put(cacheID, cache);
			} else {
				m_allCaches.put(cacheID + "_" + new UUID().toString(), cache);
			}

			return cache;
		} catch (Exception e) {
			throw new CacheException(e);
		}
	}

	/**
	 * @param cacheID TBD: Description of the incoming method parameter
	 * @param properties TBD: Description of the incoming method parameter
	 * @return a cache with specific properties. If no cache with the given ID
	 *      exists, it is created - otherwise the existing cache is returned and
	 *      the properties parameter is ignored. Use the getUniqueCacheID() method
	 *      to get a new, unique cache ID for creating a new cache.
	 * @exception CacheException Exception raised in failure situation
	 */
	public synchronized ICache getCache(String cacheID, Properties properties) throws CacheException {

		if (cacheID == null) {
			throw new IllegalArgumentException();
		}

		try {
			cacheID = cacheID.trim();
			Object cachedCache = m_singletonCaches.get(cacheID);
			if (cachedCache != null) {
				return (ICache) cachedCache;
			}

			if (m_configuredCaches.contains(cacheID)) {
				s_log.warningT("getCache(205)", "passed properties override preconfigured properties for cache " + cacheID);
			}

			String className = properties.getProperty(CFG_CACHE_CLASS_KEY).trim();
			Class cacheClass = CrtClassLoaderRegistry.forName(className);
			Class[] paramTypes = { String.class, Properties.class };
			Constructor cacheConstructor = cacheClass.getConstructor(paramTypes);
			Object[] initArgs = { cacheID, properties };
			ICache cache = (ICache) cacheConstructor.newInstance(initArgs);

			if (isPersistent(properties) || (isMemory(properties) && isSingleton(properties))) {
				m_singletonCaches.put(cacheID, cache);
				m_allCaches.put(cacheID, cache);
			} else {
				m_allCaches.put(cacheID + "_" + new UUID().toString(), cache);
			}

			return cache;
		} catch (Exception e) {
			throw new CacheException(e);
		}
	}

	/**
	 * @return a Hashtable that contains all caches. Key is cacheID, elements
	 *      implement ICache.
	 */
	public synchronized Hashtable getAllCaches() {

		return (Hashtable) m_allCaches.clone();
	}

  /**
   * Gets the cache with the given id out of all caches. This method is for use
   * in the jmx instrumentation of the cache service only.
   * @return Cache with given id.
   */
  public synchronized ICache getCacheRaw( String cacheId ) {

    return ( ICache ) m_allCaches.get( cacheId );
  }

	/**
	 * Release all cache factory resources.
	 */
	public void finalize() {

		stopRefreshThread();
	}

	/**
	 * Only for testing purposes.
	 */
	public static synchronized void resetFactory() {
    if ( s_cacheFactory != null ) {
      s_cacheFactory.stopRefreshThread();
  		s_cacheFactory = null;
    }
	}

	/**
	 * @return the DefaultProperties attribute of the CacheFactory object.
	 */
	private Properties getDefaultProperties() {

		Properties properties = new Properties();

		properties.setProperty(CacheFactory.CFG_CACHE_CLASS_KEY, ICache.DEFAULT_CACHE_CLASS);
		properties.setProperty(CacheFactory.CFG_CACHE_TYPE_KEY, ICache.DEFAULT_CACHE_TYPE);
		properties.setProperty(CacheFactory.CFG_CAPACITY_KEY, new Integer(ICache.DEFAULT_CAPACITY).toString());
		properties.setProperty(CacheFactory.CFG_MAX_CACHE_SIZE_KEY, new Long(ICache.DEFAULT_MAX_CACHE_SIZE).toString());
		properties.setProperty(CacheFactory.CFG_MAX_ENTRY_SIZE_KEY, new Long(ICache.DEFAULT_MAX_ENTRY_SIZE).toString());
		properties.setProperty(CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY, new Integer(ICache.DEFAULT_TIME_TO_LIVE).toString());
		properties.setProperty(CacheFactory.CFG_AUTO_DELAY_EXPIRATION_KEY, new Boolean(ICache.DEFAULT_SINGLETON).toString());
		properties.setProperty(CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY, new Long(ICache.DEFAULT_AVG_ENTRY_SIZE).toString());
		properties.setProperty(CacheFactory.CFG_SINGLETON_KEY, new Boolean(ICache.DEFAULT_SINGLETON).toString());

		return properties;
	}

	/**
	 * @param configurable TBD: Description of the incoming method parameter
	 * @return the CacheProperties attribute of the CacheFactory object.
	 */
	private Properties getCacheProperties(IConfigurable configurable) {

		Properties cacheProperties = getConfigurableProperties(configurable);

		if (configurable.getConfigClass().getName().equals(CFG_MEMORY_CACHE_CLASS_NAME)) {
			cacheProperties.setProperty(CFG_CACHE_TYPE_KEY, CFG_MEMORY_KEY);
		} else {
			if (configurable.getConfigClass().getName().equals(CFG_PERSISTENT_CACHE_CLASS_NAME)) {
				cacheProperties.setProperty(CFG_CACHE_TYPE_KEY, CFG_PERSISTENT_KEY);
			} else {
        if (configurable.getConfigClass().getName().equals(CFG_LONGKEYMEMORY_CACHE_CLASS_NAME)) {
          cacheProperties.setProperty(CFG_CACHE_TYPE_KEY, CFG_LONGKEYMEMORY_KEY);
        } else {
          if (configurable.getConfigClass().getName().equals(CFG_STRINGKEYMEMORY_CACHE_CLASS_NAME)) {
            cacheProperties.setProperty(CFG_CACHE_TYPE_KEY, CFG_STRINGKEYMEMORY_KEY);
          }
        }
      }
		}

		return cacheProperties;
	}

	/**
	 * @param configurable Description of the Parameter
	 * @return the name/value pairs of the attributes of a configurable in a
	 *      property bag
	 */
	private static Properties getConfigurableProperties(IConfigurable configurable) {
		Properties properties = new Properties();
		Map map = configurable.getProperties(true);
		Iterator i = map.keySet().iterator();
		while (i.hasNext()) {
			Object key = i.next();
			Object value = map.get(key);
			if (value != null) {
				properties.setProperty(key.toString(), value.toString());
			}
		}
		return properties;
	}

	/**
	 * @param properties TBD: Description of the incoming method parameter
	 * @return the Singleton attribute of the CacheFactory object.
	 */
	private boolean isSingleton(Properties properties) {

		Object singletonObject = properties.get(CacheFactory.CFG_SINGLETON_KEY);
		if (singletonObject != null) {
			return new Boolean(singletonObject.toString()).booleanValue();
		}
		return ICache.DEFAULT_SINGLETON;
	}

	/**
	 * @param properties TBD: Description of the incoming method parameter
	 * @return the CacheType attribute of the CacheFactory object.
	 */
	private String getCacheType(Properties properties) {

		return properties.getProperty(CFG_CACHE_TYPE_KEY);
	}

	/**
	 * @param properties TBD: Description of the incoming method parameter
	 * @return the Persistent attribute of the CacheFactory object.
	 */
	private boolean isPersistent(Properties properties) {

		return getCacheType(properties).equalsIgnoreCase(CacheFactory.CFG_PERSISTENT_KEY);
	}

  /**
   * @param properties TBD: Description of the incoming method parameter
   * @return the Memory attribute of the CacheFactory object.
   */
  private boolean isMemory(Properties properties) {

    return getCacheType(properties).equalsIgnoreCase(CacheFactory.CFG_MEMORY_KEY);
  }

  /**
   * @param properties TBD: Description of the incoming method parameter
   * @return the Memory attribute of the CacheFactory object.
   */
  private boolean isLongKeyMemory(Properties properties) {

    return getCacheType(properties).equalsIgnoreCase(CacheFactory.CFG_LONGKEYMEMORY_KEY);
  }

  /**
   * @param properties TBD: Description of the incoming method parameter
   * @return the Memory attribute of the CacheFactory object.
   */
  private boolean isStringKeyMemory(Properties properties) {

    return getCacheType(properties).equalsIgnoreCase(CacheFactory.CFG_STRINGKEYMEMORY_KEY);
  }

	/**
	 * Check the properties of a cache; throw an exception in case they are
	 * invalid
	 *
	 * @param cacheID TBD: Description of the incoming method parameter
	 * @param properties TBD: Description of the incoming method parameter
	 * @exception CacheException Exception raised in failure situation
	 */
	private void checkCacheProperties(String cacheID, Properties properties) throws CacheException {

		// common
		try {
			properties.get(CacheFactory.CFG_CACHE_CLASS_KEY).toString();
		} catch (Exception e) {
            //$JL-EXC$      
			throw new CacheException(
				"property " + CacheFactory.CFG_CAPACITY_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
		}
		try {
			new Integer(properties.get(CacheFactory.CFG_CAPACITY_KEY).toString()).intValue();
		} catch (Exception e) {
            //$JL-EXC$      
			throw new CacheException(
				"property " + CacheFactory.CFG_CAPACITY_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
		}
		try {
			new Long(properties.get(CacheFactory.CFG_MAX_CACHE_SIZE_KEY).toString()).longValue();
		} catch (Exception e) {
            //$JL-EXC$      
			throw new CacheException(
				"property " + CacheFactory.CFG_MAX_CACHE_SIZE_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
		}
		try {
			new Long(properties.get(CacheFactory.CFG_MAX_ENTRY_SIZE_KEY).toString()).longValue();
		} catch (Exception e) {
            //$JL-EXC$      
			throw new CacheException(
				"property " + CacheFactory.CFG_MAX_ENTRY_SIZE_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
		}
		try {
			new Integer(properties.get(CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY).toString()).intValue();
		} catch (Exception e) {
            //$JL-EXC$      
			throw new CacheException(
				"property " + CacheFactory.CFG_DEFAULT_TIME_TO_LIVE_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
		}

		if (isPersistent(properties)) {
			// persistent
			try {
				properties.get(CacheFactory.CFG_STORAGE_CLASS_KEY).toString();
			} catch (Exception e) {
            //$JL-EXC$        
				throw new CacheException(
					"property " + CacheFactory.CFG_STORAGE_CLASS_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
			}
			try {
				properties.get(CacheFactory.CFG_FOLDER_KEY).toString();
			} catch (Exception e) {
            //$JL-EXC$        
				throw new CacheException(
					"property " + CacheFactory.CFG_FOLDER_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
			}
			try {
				properties.get(CacheFactory.CFG_FILE_PREFIX_KEY).toString();
			} catch (Exception e) {
            //$JL-EXC$        
				throw new CacheException(
					"property " + CacheFactory.CFG_FILE_PREFIX_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
			}
			try {
				new Boolean(properties.get(CacheFactory.CFG_SECURE_KEY).toString()).booleanValue();
			} catch (Exception e) {
            //$JL-EXC$        
				throw new CacheException(
					"property " + CacheFactory.CFG_SECURE_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
			}
			try {
				new Boolean(properties.get(CacheFactory.CFG_CLEAR_CACHE_ON_INIT_KEY).toString()).booleanValue();
			} catch (Exception e) {
            //$JL-EXC$        
				throw new CacheException(
					"property "
						+ CacheFactory.CFG_CLEAR_CACHE_ON_INIT_KEY
						+ " for cache "
						+ cacheID
						+ " is invalid or missing in cache configuration");
			}
		} else {
			if (isMemory(properties)) {
				// memory
				try {
					new Long(properties.get(CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY).toString()).longValue();
				} catch (Exception e) {
            //$JL-EXC$          
					throw new CacheException(
						"property "
							+ CacheFactory.CFG_AVERAGE_ENTRY_SIZE_KEY
							+ " for cache "
							+ cacheID
							+ " is invalid or missing in cache configuration");
				}
				try {
					new Boolean(properties.get(CacheFactory.CFG_SINGLETON_KEY).toString()).booleanValue();
				} catch (Exception e) {
            //$JL-EXC$          
					throw new CacheException(
						"property " + CacheFactory.CFG_SINGLETON_KEY + " for cache " + cacheID + " is invalid or missing in cache configuration");
				}
			} else {
        if (isLongKeyMemory(properties) || isStringKeyMemory(properties)) {
        } else {
  				// undefined
  				throw new CacheException(
  					"invalid " + CacheFactory.CFG_CACHE_TYPE_KEY + " property for cache " + cacheID + " - check cache configuration");
        }
			}
		}
	}

	/**
	 * Call the refresh() method of all caches to remove expired entries
	 */
	private void privateRefreshCaches() {

		List caches = Collections.EMPTY_LIST;
		synchronized (this) {
			caches = new ArrayList(m_allCaches.values());
		}

		for (int i = 0, n = caches.size(); i < n; i++) {
			ICache cache = (ICache) caches.get(i);
			if (cache != null) {
				try {
					cache.refresh();
				} catch (Exception e) {
                      //$JL-EXC$
					s_log.debugT(e.getMessage());
				}
			}
		}

	}

	/**
	 * Start a thread which refreshes the caches (remove expired entries)
	 */
	private void startRefreshThread() {

		m_refreshThread.start();
	}

	/**
	 * Stop the thread which refreshes the caches
	 */
	private void stopRefreshThread() {

		m_refreshThread.terminate();
    try {
      m_refreshThread.join() ;
    }
    catch ( InterruptedException e ) {
            //$JL-EXC$      
      s_log.debugT(e.getMessage());
    }
	}

	/**
	 * Tell the CacheFactory that the caches have to be refreshed
	 */
	protected static void refreshCaches() {

		if (s_cacheFactory != null) {
			s_cacheFactory.privateRefreshCaches();
		}
	}

	/**
	 * Thread which refreshes the caches
	 */
	class RefreshThread extends Thread {
		private final static long REFRESH_INTERVAL = 300; // seconds
		private final static long SLEEP = 300; // seconds

		private long m_nextTime = 0;
		private boolean m_terminate = false;

		/**
		 * Main processing method
		 */
		public void run() {

			while (!m_terminate) {
				try {
					Thread.sleep(SLEEP * 1000);
				} catch (InterruptedException e) {
            //$JL-EXC$          
					s_log.debugT(e.getMessage());
					return;
				}
				if (new Date().getTime() >= m_nextTime) {
					CacheFactory.refreshCaches();
					m_nextTime = new Date().getTime() + REFRESH_INTERVAL * 1000;
				}
			}
		}

		/**
		 * Terminate the thread
		 */
		public void terminate() {
			m_terminate = true;
			this.interrupt();
		}
	}
}
