/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl;

import com.sapportals.portal.security.usermanagement.*;

/**
 * An <code>IOwner</code> holds a list of principals that are the owners of an
 * ACL. <br>
 *
 */
public interface IOwner {
  /**
   * add a new owner to the list
   *
   * @param caller the caller must be an owner of the ACL
   * @param principal the new owner
   * @return true iff the owner was added successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner
   */
  public boolean addOwner(IUMPrincipal caller, IUMPrincipal principal)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * remove an owner from the list
   *
   * @param caller the caller must be an owner of the ACL
   * @param owner the owner
   * @return true iff the owner was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner
   * @exception LastOwnerException attempt to remove the last owner of the ACL
   *      (which is forbidden)
   */
  public boolean removeOwner(IUMPrincipal caller, IUMPrincipal owner)
    throws AclPersistenceException, NotAuthorizedException, LastOwnerException;

  /**
   * check whether a principal is part of the owner list of the ACL
   *
   * @param principal the principal
   * @return true iff the principal is part of the owner list of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isOwner(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * check whether a principal is an owner or a member of some owner group/role
   * of the ACL
   *
   * @param principal the principal
   * @return true iff the principal an owner or a member of some owner group of
   *      the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isAuthorized(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * get a list of the owners of the ACL
   *
   * @return the list of the owners of the ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IUMPrincipalList getOwners()
    throws AclPersistenceException;

  /**
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IUMPrincipalList lnkIUMPrincipalList;
   */
}
