/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl;

import com.sapportals.portal.security.usermanagement.*;
import com.sapportals.wcm.util.cache.*;

import java.util.*;


/**
 * An <code>IAclManager</code> administers and persists <i>Access Control Lists
 * </i> (ACLs). <br>
 * Abbreviations:
 * <ul>
 *   <li> ACL: Access Control List</li>
 *   <li> ACE: Access Control List Entry</li>
 * </ul>
 *
 */
public interface IAclManager {
  /**
   * create a new ACE to grant or deny a permission to a principal
   *
   * @param principal the principal
   * @param negative true if the ACE denies a permission, false if it grants
   * @param permission the permission
   * @param sortIndex the position of the ACE in the ACL
   * @param propagate TBD: Description of the incoming method parameter
   * @return the newly created ACE
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the permission was created by a foreign
   *      resource acl manager
   */
  public IAclEntry createAclEntry(IUMPrincipal principal, boolean negative, IAclPermission permission, int sortIndex, boolean propagate)
    throws AclPersistenceException, InvalidClassException;

  /**
   * create a new ACL
   *
   * @param owner the initial owner of the ACL
   * @param aclID the ID of the new ACL
   * @param objectType the type of object the ACL is for
   * @return the newly created ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception AclExistsException an ACL with the same ID already exists
   */
  public IAcl createAcl(IUMPrincipal owner, String aclID, IObjectType objectType)
    throws AclPersistenceException, AclExistsException;

  /**
   * get a specific ACL
   *
   * @param aclID ID of the ACL
   * @param contextDBVersion TBD: Description of the incoming method parameter
   * @return the ACL with the given ID or null
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAcl getAcl(String aclID, long contextDBVersion)
    throws AclPersistenceException;

  /**
   * get multiple ACLs; set the ACLs as values of the HashMap; if no ACL is
   * assigned to an ID the value will be null
   *
   * @param aclIDs IDs of the ACLs
   * @return acls
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAcl[] getAcls(String aclIDs[])
    throws AclPersistenceException;

  /**
   * check multiple ACL permissions
   *
   * @param acls ACLs to check
   * @param principal the user or group
   * @param permissions requested permissions
   * @return array elements are true for grant on corresponding acl, otherwise
   *      false
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean[] checkPermissions(IAcl[] acls, IUMPrincipal principal, IAclPermission[] permissions)
    throws AclPersistenceException;

  /**
   * remove a specific ACL
   *
   * @param caller a principal which must be an owner of the ACL
   * @param aclID ID of the ACL
   * @return true iff the ACL did exist and was removed
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner of the ACL
   */
  public boolean removeAcl(IUMPrincipal caller, String aclID)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * remove a specific ACL
   *
   * @param caller a principal which must be an owner of the ACL
   * @param acl the subject of removal
   * @return true iff the ACL did exist and was removed
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner of the ACL
   * @exception InvalidClassException Exception raised in failure situation
   */
  public boolean removeAcl(IUMPrincipal caller, IAcl acl)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException;

  /**
   * delete a principal - delete all ACEs which are subject to this principal -
   * remove the principal from all ACL owner lists - if the principal is the
   * last owner of an ACL, remove the ACL
   *
   * @param principalID the principal's ID
   * @param principalType type of the principal (from IUMPrincipal)
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public void deletePrincipal(String principalID, int principalType)
    throws AclPersistenceException;

  /**
   * update a principal - remove related entries from caches
   *
   * @param principalID the principal's ID
   * @param principalType type of the principal (from IUMPrincipal)
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public void updatePrincipal(String principalID, int principalType)
    throws AclPersistenceException;

  /**
   * get a list of supported object types
   *
   * @return the supported object types
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IObjectTypeList getSupportedObjectTypes()
    throws AclPersistenceException;

  /**
   * get a list of permissions which are supported for a specific object type
   *
   * @param objectType the object type
   * @return the permissions
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAclPermissionList getSupportedPermissions(IObjectType objectType)
    throws AclPersistenceException;

  /**
   * get the IAclPermission object of the (supported) permission with a given
   * name
   *
   * @param name the name of the permission
   * @return the IAclPermission object
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAclPermission getPermission(String name)
    throws AclPersistenceException;

  /**
   * assign a permission to an object type
   *
   * @param objectType the object type
   * @param permission the permission
   * @return true iff the permission was assigned successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   */
  public boolean addSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException;

  /**
   * remove the assignment of a permission to an object type
   *
   * @param objectType the object type
   * @param permission the permission
   * @return true iff the assignment was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   * @exception PermissionUsedException Exception raised in failure situation
   */
  public boolean removeSupportedPermission(IObjectType objectType, IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException;

  /**
   * create a new permission (the permission can not be used until it is
   * assigned to an object type by calling the addSupportedPermission() method)
   *
   * @param name the permission name
   * @return the newly created permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PermissionExistsException a permission with that name exists
   *      already
   */
  public IAclPermission createPermission(String name)
    throws AclPersistenceException, PermissionExistsException;

  /**
   * remove a permission - a permission can only be removed if it is not
   * predefined and not used in an ACL
   *
   * @param permission the permission
   * @return true iff the permission was removed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception PredefinedPermissionException the permission is predefined and
   *      thus not allowed to be changed
   * @exception PermissionUsedException the permission is used in some ACL and
   *      thus not allowed to be changed
   */
  public boolean removePermission(IAclPermission permission)
    throws AclPersistenceException, PredefinedPermissionException, PermissionUsedException;

  /**
   * check whether a permission is used in an ACL
   *
   * @param permission the permission
   * @return true iff the permission is used in an ACL
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isPermissionUsedInAcl(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * get a list of all IDs of ACLs which use a permission
   *
   * @param permission the permission
   * @return the list of accected ACL IDs (Strings)
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public List getAffectedAclIDs(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * change the ID of an ACL ("rename")
   *
   * @param caller a principal which must be an owner of the ACL
   * @param acl the ACL
   * @param newID the new ID of the ACL
   * @return true iff the ID has been changed successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException Exception raised in failure situation
   * @exception AclExistsException Exception raised in failure situation
   */
  public boolean changeAclID(IUMPrincipal caller, IAcl acl, String newID)
    throws AclPersistenceException, NotAuthorizedException, AclExistsException;

  /**
   * change the ID of multiple ACLs ("recursive rename") - does not change the
   * root ACL (call changeAclID() for the root node previously to change the
   * root ACL and to check for authorization)
   *
   * @param oldAclIdPrefix prefix of the old ACL IDs
   * @param newAclIdPrefix prefix of the new ACL IDs
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception AclExistsException Exception raised in failure situation
   */
  public void changeMultipleAclIDs(String oldAclIdPrefix, String newAclIdPrefix)
    throws AclPersistenceException, AclExistsException;

  /**
   * remove multiple ACLs ("recursive deletion") - does not remove the root ACL
   * (call removeAcl() for the root node previously to remove the root ACL and
   * to check for authorization)
   *
   * @param aclIdPrefix prefix of the ACL IDs to be deleted
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public void removeMultipleAcls(String aclIdPrefix)
    throws AclPersistenceException;

  /**
   * check whether IAclManager is read only (no creation or modification of ACLs
   * is allowed)
   *
   * @return true iff the IAclManager is read only
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;

  /**
   * check whether the ACLs are up to date (compared to the database)
   *
   * @param acls the ACLs to check
   * @return a boolean array containing true for the ACLs which are up to date
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean[] areAclsUpToDate(IAcl[] acls)
    throws AclPersistenceException;

  /**
   * Returns a value representing the state (timestamp or modifycounter) of all
   * persisted acl entries. If a negative number is returned this method is not
   * supported or an error occured.
   *
   * @return dBVersion
   */
  public long getDBVersion();

  public IClusterCacheArea getCacheArea();


}
