/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl;

import com.sapportals.portal.security.usermanagement.*;

import java.util.*;

/**
 * An <code>IAclEntry</code> is an entry in an <code>IAcl</code> which grants or
 * denies a permission to a specific principal. <br>
 * Abbreviations:
 * <ul>
 *   <li> ACL: Access Control List</li>
 *   <li> ACE: Access Control List Entry</li>
 * </ul>
 *
 */
public interface IAclEntry {
  /**
   * get the permission of the ACE
   *
   * @return the permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IAclPermission getPermission()
    throws AclPersistenceException;

  /**
   * check whether some permission is covered by the permission of the ACE
   *
   * @param permission the permission
   * @return true iff all actions of the permission are contained in the
   *      permission of the ACE
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean checkPermission(IAclPermission permission)
    throws AclPersistenceException;

  /**
   * check whether the ACE denies its permission
   *
   * @return true iff the ACE denies its permission
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isNegative()
    throws AclPersistenceException;

  /**
   * get the principal of the ACE
   *
   * @return the principal of the ACE
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IUMPrincipal getPrincipal()
    throws AclPersistenceException;

  /**
   * get the sort index of the ACE - the sequence of the ACEs in the ACL is
   * controlled by the sort indices
   *
   * @return the sort index
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public int getSortIndex()
    throws AclPersistenceException;

  /**
   * set the sort index of the ACE
   *
   * @param caller the caller which must be an owner of the ACL the ACE is
   *      assigned to (may be null in case it is not assigned yet)
   * @param sortIndex the new sort index
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the user in the resource context of the
   *      belonging ACL is not an owner of the ACL
   */
  public void setSortIndex(IUMPrincipal caller, int sortIndex)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * get the propagation flag of the ACE - propagation is no task of the basic
   * acl package - only the flag is persisted
   *
   * @return the propagation flag of the ACE
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isPropagated()
    throws AclPersistenceException;

  /**
   * set the propagation flag of the ACE - propagation is no task of the basic
   * acl package - only the flag is persisted
   *
   * @param caller the caller which must be an owner of the ACL the ACE is
   *      assigned to (may be null in case it is not assigned yet)
   * @param propagate the propagation flag
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException Exception raised in failure situation
   */
  public void setPropagation(IUMPrincipal caller, boolean propagate)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * check wether the ACE is read only
   *
   * @return true iff the ACE is read only
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;

  /**
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IAclPermission lnkIAclPermission;
   */
}
