/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl;

import com.sapportals.portal.security.usermanagement.*;

import java.util.*;

/**
 * An <code>IAcl</code> is a <i>Access Control List</i> which defines the access
 * permissions to an object. <br>
 * Abbreviations:
 * <ul>
 *   <li> ACL: Access Control List</li>
 *   <li> ACE: Access Control List Entry</li>
 * </ul>
 *
 */
public interface IAcl extends IOwner {
  /**
   * add an ACE to the ACL
   *
   * @param caller a principal which must be an owner of the ACL
   * @param aclEntry the ACE
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the ACE was created by a foreign resource
   *      acl manager
   * @exception NotAuthorizedException the caller in not an owner
   * @exception AlreadyAssignedToAclException the ACE is already assigned to an
   *      ACL
   * @exception PermissionNotSupportedException the permission of the ACE is not
   *      supported for the type of the ACL's object
   * @returns true iff the ACE has been successfully added
   */
  public boolean addEntry(IUMPrincipal caller, IAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException, AlreadyAssignedToAclException, PermissionNotSupportedException;

  /**
   * remove an ACE from the ACL
   *
   * @param caller a principal which must be an owner of the ACL
   * @param aclEntry the ACE
   * @return TBD: Description of the outgoing return value
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception InvalidClassException the ACE was created by a foreign resource
   *      acl manager
   * @exception NotAuthorizedException the caller in not an owner
   * @returns true iff the ACE has been successfully removed
   */
  public boolean removeEntry(IUMPrincipal caller, IAclEntry aclEntry)
    throws AclPersistenceException, InvalidClassException, NotAuthorizedException;

  /**
   * get the ACEs of the ACL
   *
   * @return entries
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @returns the ACEs
   */
  public IAclEntryList getEntries()
    throws AclPersistenceException;

  /**
   * get the ACEs of the ACL concerning a specific principal
   *
   * @param principal the principal
   * @return entries
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @returns the ACEs
   */
  public IAclEntryList getEntries(IUMPrincipal principal)
    throws AclPersistenceException;

  /**
   * check whether the ACL grants a specific permission to a user
   *
   * @param permission the permission
   * @param principal TBD: Description of the incoming method parameter
   * @return true iff the permission is granted
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean checkPermission(IUMPrincipal principal, IAclPermission permission)
    throws AclPersistenceException;

  /**
   * lock the ACL (the lock is cooperative and affects only the result of
   * subsequent calls to lock() or isLocked())
   *
   * @param caller TBD: Description of the incoming method parameter
   * @return true iff the ACL was locked successfully
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not an owner
   */
  public boolean lock(IUMPrincipal caller)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * check whether the ACL is locked
   *
   * @return true iff the ACL is locked
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isLocked()
    throws AclPersistenceException;

  /**
   * get the user which locked the ACL
   *
   * @return the user which locked the ACL or null in case the ACL is not locked
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public IUMPrincipal getLockingUser()
    throws AclPersistenceException;

  /**
   * unlock the ACL
   *
   * @param caller TBD: Description of the incoming method parameter
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   * @exception NotAuthorizedException the caller is not the user which locked
   *      the ACL
   */
  public void unlock(IUMPrincipal caller)
    throws AclPersistenceException, NotAuthorizedException;

  /**
   * check whether the ACl has descendants that have an ACL too
   *
   * @param postFix is appended to the ACL ID before the descendants are
   *      identified as all ACLs whose IDs start with this ID
   * @return true iff the ACL has descendants that have an ACL too
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean hasDescendantsWithAcl(String postFix)
    throws AclPersistenceException;

  /**
   * get a list of all IDs of the descendants of the ACL that have an ACL too
   *
   * @param postFix is appended to the ACL ID before the descendants are
   *      identified as all ACLs whose IDs start with this ID
   * @return the IDs of the descendants
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public List getDescendantsWithAcl(String postFix)
    throws AclPersistenceException;// return list of Strings

  /**
   * check whether the ACL is read only
   *
   * @return true iff the ACL is read only
   * @exception AclPersistenceException a problem with the storage where the
   *      ACLs are persisted occurred
   */
  public boolean isReadOnly()
    throws AclPersistenceException;

  /**
   * check whether the ACL is up to date (check change level of persisted ACL)
   * (used to invalidate ACL caches)
   *
   * @return true iff the persisted ACL has the same change level
   * @exception AclPersistenceException Exception raised in failure situation
   */
  public boolean isUpToDate()
    throws AclPersistenceException;

  /**
   * @supplierCardinality 1
   * @directed
   */
  /*
   * #IAclEntryList lnkIAclEntryList;
   */
}
