/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.util.acl;

import com.sapportals.config.fwk.Configuration;
import com.sapportals.config.fwk.IConfigClientContext;
import com.sapportals.config.fwk.IConfigManager;
import com.sapportals.config.fwk.IConfigPlugin;
import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.crt.*;
import com.sapportals.wcm.util.config.*;
import com.sapportals.wcm.util.cache.*;

import java.util.*;


/**
 * The factory which creates <code>IAclManager</code> s. <br>
 *
 */
public final class AclManagerFactory {
  private final static String CFG_PLUGIN_CM_UTILITIES_ACL_FACTORIES = "/cm/utilities/acl_factories";  
  private final static String CFG_MANAGER_CLASS_KEY = "class";
  private final static String CFG_MANAGER_CONFIG_KEY = "cfg";

  private static AclManagerFactory s_instance = null;
  private static Hashtable s_managers = null ;

  /**
   * Construct
   */
  private AclManagerFactory() { }

  /**
   * @param factoryConfig TBD: Description of the incoming method parameter
   * @return an ACL manager for the given configuration
   * @exception AclException Exception raised in failure situation
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidConfigException Exception raised in failure situation
   */
  public synchronized IAclManager getAclManager(String factoryConfig)
    throws AclException, AclLoadClassException, AclPersistenceException, InvalidConfigException {

    return getAclManager(factoryConfig, null);

  }

  /**
   * @param factoryConfig TBD: Description of the incoming method parameter
   * @return an ACL manager for the given configuration
   * @exception AclException Exception raised in failure situation
   * @exception AclLoadClassException Exception raised in failure situation
   * @exception AclPersistenceException Exception raised in failure situation
   * @exception InvalidConfigException Exception raised in failure situation
   */
  public synchronized IAclManager getAclManager(String factoryConfig, IClusterCacheArea cacheArea)
    throws AclException, AclLoadClassException, AclPersistenceException, InvalidConfigException {

    if (factoryConfig == null) {
      throw new java.lang.IllegalArgumentException();
    }

    String managerClassName = null;
    String managerConfig = null;
    try {
      IConfigClientContext context = IConfigClientContext.createContext(
        ConfigCrutch.getConfigServiceUser());
      IConfigManager cfgManager = Configuration.getInstance().getConfigManager(context);
      IConfigPlugin plugin = cfgManager.getConfigPlugin(CFG_PLUGIN_CM_UTILITIES_ACL_FACTORIES);
      
      Properties properties = ConfigCrutch.getConfigurableProperties( plugin.getConfigurable(factoryConfig));
      managerClassName = properties.getProperty(CFG_MANAGER_CLASS_KEY);
      managerConfig = properties.getProperty(CFG_MANAGER_CONFIG_KEY);
    }
    catch (Exception e) {
            //$JL-EXC$      
      throw new InvalidConfigException(e.getMessage());
    }

    // manager already instantiated (managers are singletons for a configuration)?
    String managerKey = managerClassName + '#' + managerConfig;
    IAclManager manager = (IAclManager)s_managers.get(managerKey);
    if (manager != null) {
      return manager;
    }

    // instantiate new manager
    Class managerClass = null;
    try {
      managerClass = CrtClassLoaderRegistry.forName(managerClassName);
      Class[] classes = new Class[]{String.class, String.class, IClusterCacheArea.class};
      Object[] params = new Object[]{IWcmConst.SAP_WCM_NAMESPACE, managerConfig, cacheArea};
      manager = (IAclManager)(managerClass.getConstructor(classes).newInstance(params));
    }
    catch (Exception e) {
            //$JL-EXC$      
      throw new AclLoadClassException(managerClassName);
    }

    // store manager
    s_managers.put(managerKey, manager);
    return manager;
  }

  /**
   * @return an AclManagerFactory instance (singleton)
   */
  public static synchronized AclManagerFactory getInstance() {

    if (s_instance == null) {
      s_instance = new AclManagerFactory();
      s_managers = new Hashtable() ;
    }
    return s_instance;
  }
  
  public static synchronized void resetFactory() {
    if ( s_managers != null ) {
      Enumeration e = s_managers.elements() ;
      while ( e.hasMoreElements()) {
        Object o = e.nextElement() ;
        java.lang.reflect.Method m ;
        try {
          m = o.getClass().getMethod("shutdown", new Class[0]) ;
          m.invoke(o,new Object[0]);
        }
        catch ( Exception ex ) {
          //$JL-EXC$
          // if we can't call shutdown then that's sad but we can live with it
        }
      }
      s_managers.clear();
      s_managers = null ;
    }
    s_instance = null ;
  }
}
