package com.sap.tc.loggingStandard;

/**
 * Title:        LoggingStandard
 * Description:
 * Copyright:    Copyright (c) 2001
 * Company:      SAP Markets, Inc
 * @author
 * @version $Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/Level.java#1 $
 */

import java.lang.*;
import java.util.*;

import com.sap.tc.logging.*;

/**
 * <p>
 *   This class is equivalent to <code>java.util.logging.Level</code>.
 * </p>
 * <p>
 *   It provides the standard severity levels defined by the SUN API,
 *   and internally maps them to the SAP severities.
 * </p>
 */
public class Level implements java.io.Serializable{

  static String version = "$Id: //sapmarkets/Logging/630_VAL_REL/src/_StdLogging1.3/java/com/sap/tc/loggingStandard/Level.java#1 $";

  private static ArrayList regLevels = new ArrayList();
  public static final Level OFF = new Level("OFF", Integer.MAX_VALUE);
  //NEW
   static final Level SAP_MAX = new Level("SAP_MAX", 1001);
  public static final Level SEVERE = new Level("SEVERE", 1000);
  //NEW
   static final Level SAP_ERROR = new Level("SAP_ERROR", 999);
  public static final Level WARNING = new Level("WARNING", 900);
  public static final Level INFO = new Level("INFO", 800);
  public static final Level CONFIG = new Level("CONFIG", 700);
  public static final Level FINE = new Level("FINE", 500);
  public static final Level FINER = new Level("FINER", 400);
  public static final Level FINEST = new Level("FINEST", 300);
  public static final Level ALL = new Level("ALL",Integer.MIN_VALUE);


  final static int MAPPEDCONFIG = Severity.INFO - 10;
  final static int MAPPEDFINE   = Severity.INFO - 20;

  protected Level(String name, int value) {
    levelName = name;
    levelValue = value;
    //synchronized this class?
    regLevels.add(this);
  }

  static Level toLevel(int severity){
    if (severity <= Severity.ALL)
      return Level.ALL;
    if (severity <= Severity.DEBUG)
      return Level.FINEST;
    if (severity <= Severity.PATH)
      return Level.FINER;
    if (severity <= MAPPEDFINE)
      return Level.FINE;
    if (severity <= MAPPEDCONFIG)
      return Level.CONFIG;
    if (severity <= Severity.INFO)
      return Level.INFO;
    if (severity <= Severity.WARNING)
      return Level.WARNING;
//NEW:::: more precise
  if (severity <= Severity.ERROR)
    return Level.SAP_ERROR;

    if (severity <= Severity.FATAL)
      return Level.SEVERE;
//NEW:::: more precise
  if (severity <= Severity.MAX)
    return Level.SAP_MAX;
    if (severity <= Severity.NONE)
      return Level.OFF;
    return Level.OFF;    //unknown
  }

  static int mapLevel(Level inLevel){
    if(inLevel == Level.OFF)
      return Severity.NONE;     //701
//NEW::::more precise
    else if (inLevel == Level.SAP_MAX)
      return Severity.MAX;      //700
    else if (inLevel == Level.SEVERE)
      return Severity.FATAL;    //600
//NEW::::more precise
    else if (inLevel == Level.SAP_ERROR)
      return Severity.ERROR;    //500
    else if (inLevel == Level.WARNING)
      return Severity.WARNING;  //400
    else if (inLevel == Level.INFO)
      return Severity.INFO;     //300
    else if (inLevel == Level.CONFIG)
      return MAPPEDCONFIG;     //290
    else if (inLevel == Level.FINE)
      return MAPPEDFINE;      //280
    else if (inLevel == Level.FINER)
      return Severity.PATH;      //200
    else if (inLevel == Level.FINEST)
      return Severity.DEBUG;      //100
    else if (inLevel == Level.ALL)
      return Severity.ALL;      //0
    else
      return Severity.NONE;  //no mapping, treat as NONE
  }


  /**
   * Parse a level name string into a Level.
   * <p>
   * The argument string may consist of either a level name or an integer value.
   * For example: <br>
   * "SEVERE" <br>
   * "1000"
   * <p>
   * In this wrapper:
   * If the pass-in name does not exist, no further new Level will be created,
   * IllegalArgumentException thrown.
   * <p>
   * @param level   String to be pased
   * @return   parsed value
   * @exception  NullPointerException  if the name is null
   * @exception  IllegalArgumentException if the value is neither one of the
   *             known names nor an integer.
   */
  public static Level parse(String name)
                              throws IllegalArgumentException {
    if (name == null)
      throw new NullPointerException();

    //search if the pass-in arg is a string
    for (int i = 0; i < regLevels.size(); i++) {
      Level _lev = (Level)regLevels.get(i);
      if (name.equals(_lev.levelName)) {
        return _lev;
      }
    }

    //not string, probably an Integer. Otherwise, null.
    for (int i = 0; i < regLevels.size(); i++) {
      Level _lev = (Level)regLevels.get(i);
      try {
        int x = Integer.parseInt(name);
        if (_lev.levelValue == x) {
          return _lev;
        }
      }
      catch (NumberFormatException ex) {
        //LogManager.getIntErrorCat().errorT
        throw new IllegalArgumentException("Invalid level : " + name);
      }
    }

    return null;
  }


  /**
   * Return the literal name of the severity level.
   * @return the string name of the Level, for example "INFO".
   */
  public final String toString() {
    return levelName;
  }

//intValue


  private final String levelName;
  private final int levelValue;

}