<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:UML="UML-NS" xmlns:SG="http://www.sap.com/xapps/xslt_generation/">
	<xsl:import href="XMIToDictionary-s.xslt"/>
	
	<xsl:output method="xml" version="1.0" encoding="UTF-8" indent="yes"/>
	
	<xsl:param name="app-provider"/>
	<xsl:param name="app-id"/>
	
	<xsl:template match="XMI.Simplified">
		<MetaData>
<!--			<xsl:copy-of select="."/>-->
			<!-- [=============== Process all classes or diagram classes if valid diagram name specified ==============] -->
			<xsl:apply-templates select="XMI.content/UML:Model/Classes/UML:Class[contains($class-names,concat('#',@xmi.id,'#')) and ($diagram-name!='' or not(UML:TaggedValue[@tag='externalservice']))]" mode="s"/>
			<!-- [=============== Process external services ==============] -->
			<xsl:apply-templates select="XMI.content/UML:Model/Classes/UML:Class[$diagram-name='' and UML:TaggedValue[@tag='externalservice']]" mode="s-externalservice"/>
			<!-- [=============== TypeAccess service ==============] -->
			<xsl:if test="$diagram-name=''">
				<xsl:call-template name="s-service-type-access"/>
			</xsl:if>
			<SG:dictionary>
				<xsl:apply-templates select="XMI.content/UML:Model/Classes/UML:Class[@stereotype='type' and not(UML:TaggedValue[@tag='predefined'])]" mode="s-simple-type"/>
			</SG:dictionary>
		</MetaData>
	</xsl:template>

	<xsl:template match="UML:Class" mode="s">
		<xsl:variable name="class-id" select="@xmi.id"/>
		<xsl:if test="@stereotype='BO Service' or @stereotype='Composite Service'">
			<xsl:variable name="class-name" select="@name"/>
			<coolservice name="{@name}" SG:stereotype="{@stereotype}">
				<xsl:if test="UML:TaggedValue[@tag='optimisticlocking']">
					<xsl:attribute name="SG:olock"/>
				</xsl:if>
				<aspect name="{@name}" primary="true" id="{@xmi.id}">
					<xsl:call-template name="s-aspect-content"/>
				</aspect>
				<!-- [=============== Generate key aspect, for composite services extended from BO services this is the second aspect ==============] -->
				<xsl:if test="@stereotype='Composite Service' and @parent">
					<xsl:apply-templates select="." mode="s-key-aspect"/>
				</xsl:if>
				<xsl:if test="@stereotype='BO Service'">
					<!-- [=============== Process extended classes (as non-primary aspects) ==============] -->
					<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@parent=$class-id and @stereotype='Aspect']">
						<aspect name="{@name}" id="{@xmi.id}">
							<xsl:call-template name="s-aspect-content"/>
						</aspect>
						<xsl:apply-templates select="UML:Operation[@visibility!='private' and (@stereotype='Query' or @stereotype='Find-by query')]" mode="s-query"/>
						<xsl:apply-templates select="UML:Operation[@visibility!='private']" mode="s-method"/>
					</xsl:for-each>
				</xsl:if>
				<!-- [=============== Process non-primary aspects ==============] -->
				<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id or @target=$class-id]" mode="s-aspect"/>
				<!-- [=============== Generate key aspect, for BO services this is the last aspect ==============] -->
				<xsl:if test="@stereotype='BO Service' or not(@parent)">
					<xsl:apply-templates select="." mode="s-key-aspect"/>
				</xsl:if>
				<!-- [=============== Generate method descriptors ==============] -->
				<xsl:apply-templates select="UML:Operation[@visibility!='private']" mode="s-method"/>
				<!-- [=============== Generate relation specific method descriptiors (relation adders and removers) ==============] -->
				<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id and @visibility!='private']" mode="s-method"/>
				<!-- [=============== Generate descriptors of methods from related aspect ==============] -->
				<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id or @target=$class-id]" mode="s-aspect-method"/>
				<!-- [=============== Generate query declarations ==============] -->
				<xsl:apply-templates select="UML:Operation[@visibility!='private' and (@stereotype='Query' or @stereotype='Find-by query')]" mode="s-query"/>
				<!-- [=============== Generate query declarations found in non-primary aspects ==============] -->
				<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id or @target=$class-id]" mode="s-aspect-query"/>
				<!-- [=============== Generate predefined CAF specific queries (and methods) ==============] -->
				<xsl:if test="@stereotype='BO Service'">
					<xsl:call-template name="s-predefined-queries"/>
				</xsl:if>
				<application id="{$app-id}" provider="{$app-provider}"/>
			</coolservice>
		</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Class" mode="s-key-aspect">
		<xsl:variable name="class-id" select="@xmi.id"/>
		<aspect name="KEY_{@name}" key="true" id="{@xmi.id}key">
			<xsl:variable name="allprops">
				<xsl:for-each select="UML:Attribute">
					<xsl:variable name="associates"><xsl:call-template name="s-associates-field"/></xsl:variable>
					<xsl:if test="$associates!=''">
						<xsl:value-of select="concat('%',@name,'%#',$associates,'#')"/>
					</xsl:if>
				</xsl:for-each>
			</xsl:variable>
			<xsl:choose>
				<xsl:when test="UML:Attribute[@stereotype='Primary key']">
					<xsl:attribute name="SG:custom-key">true</xsl:attribute>
					<xsl:for-each select="UML:Attribute[@stereotype='Primary key']">
						<property name="{@name}">
							<xsl:apply-templates select="." mode="s-property-attributes"/>
						</property>
					</xsl:for-each>
				</xsl:when>
				<xsl:when test="$allprops!=''">
					<xsl:attribute name="SG:custom-key">true</xsl:attribute>
					<xsl:for-each select="UML:Attribute">
						<xsl:variable name="associates"><xsl:call-template name="s-associates-field"/></xsl:variable>
						<xsl:choose>
							<xsl:when test="substring-after($associates,'.')='guid'">
								<xsl:apply-templates select="." mode="s-property"/>
							</xsl:when>
							<xsl:when test="$associates!='' and not(contains(substring-before($allprops,concat('%',@name,'%')),concat('#',substring-before($associates,'.'),'.')))
									and not(contains($allprops,concat('#',substring-before($associates,'.'),'.guid#')))">
								<property SG:visibility="public" type="String" text="">
									<xsl:attribute name="name">
										<xsl:call-template name="java-id">
											<xsl:with-param name="id" select="substring-before($associates,'.')"/>
										</xsl:call-template>
										<xsl:text>Key</xsl:text>
									</xsl:attribute>
								</property>
							</xsl:when>
						</xsl:choose>
					</xsl:for-each>
				</xsl:when>
				<xsl:when test="@stereotype='BO Service' or @parent">
					<property name="guid" type="String" text="ID"/>
				</xsl:when>
			</xsl:choose>
			<xsl:if test="@stereotype='BO Service'">
				<!-- generate aspect relations from parents to children -->
				<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@parent=$class-id]" mode="s-relation-from-parent"/>
			</xsl:if>
			<!-- [=============== Process usual model relations ==============] -->
			<xsl:for-each select="UML:Attribute[@visibility!='private']">
				<xsl:variable name="target" select="@type"/>
				<xsl:if test="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@full-name=$target and (@stereotype='Composite Service' or @stereotype='Aspect' or @stereotype='BO Service')]">
					<xsl:apply-templates select="." mode="s-aspectrelation"/>
				</xsl:if>
			</xsl:for-each>
			<xsl:variable name="full-name" select="@full-name"/>
			<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class">
				<xsl:apply-templates select="UML:Attribute[@visibility!='private' and @type=$full-name and UML:TaggedValue[@tag='bidirectional']]" mode="s-aspectrelation-back"/>
			</xsl:for-each>
			<!--xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id and @visibility!='private']" mode="s-aspectrelation"/-->
			<!-- [=============== Process bidirectional usual relations ==============] -->
			<!--xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@target=$class-id and @visibility!='private']" mode="s-aspectrelation-back"/-->
			<!-- [=============== Process private relations in the same way ==============] -->
			<xsl:if test="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@target=$class-id and @visibility='private']">
				<SG:privaterelations>
					<xsl:for-each select="UML:Attribute[@visibility='private']">
						<xsl:variable name="target" select="@type"/>
						<xsl:if test="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@visibility='private' and @full-name=$target and (@stereotype='Composite Service' or @stereotype='Aspect' or @stereotype='BO Service')]">
							<xsl:apply-templates select="." mode="s-aspectrelation"/>
						</xsl:if>
					</xsl:for-each>
					<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class">
						<xsl:apply-templates select="UML:Attribute[@visibility='private' and @type=$full-name and UML:TaggedValue[@tag='bidirectional']]" mode="s-aspectrelation-back"/>
					</xsl:for-each>
					<!--xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id and @visibility='private']" mode="s-aspectrelation"/>
					<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@target=$class-id and @visibility='private']" mode="s-aspectrelation-back"/-->
				</SG:privaterelations>
			</xsl:if>
		</aspect>
	</xsl:template>
	
	<xsl:template name="s-service-type-access">
		<coolservice name="TypeAccess" SG:stereotype="Composite Service" SG:implemented-service="type-access">
			<aspect name="TypeAccessPrimary" primary="true" id="TypeAccessPrimary"/>
			<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@stereotype='type' and not(UML:TaggedValue[@tag='predefined'])]">
				<xsl:variable name="class-id" select="@xmi.id"/>
				<xsl:variable name="t-type" select="concat(substring(@name,1,1),translate(substring(@name,2,string-length(@name)-1),$upper-case,$lower-case))"/>
				<xsl:variable name="base-type" select="UML:TaggedValue[@tag='originaltype']"/>
				<xsl:variable name="is-enum" select="UML:TaggedValue[@tag='enumaration']"/>
				<xsl:if test="UML:TaggedValue[@tag='option']">
					<aspect name="{$t-type}Value" id="{$t-type}.id" SG:stereotype="Composite Service" SG:type-name="{@name}" SG:type-package="{@package}">
						<property name="backendValue" SG:visibility="public" type="{$base-type}" text="Backend value"/>
						<property name="frontendValue" SG:visibility="public" text="Frontend value">
							<xsl:attribute name="type">
								<xsl:choose>
									<xsl:when test="$is-enum">String</xsl:when>
									<xsl:otherwise><xsl:value-of select="$base-type"/></xsl:otherwise>
								</xsl:choose>
							</xsl:attribute>
						</property>
					</aspect>
					<query name="provide{$t-type}Values" SG:visibility="public" aspect="{$t-type}Value" id="{$t-type}.query"/>
					<method name="provide{$t-type}Values" SG:visibility="public" methodType="0" returnType="java.util.Collection"/>
				</xsl:if>
			</xsl:for-each>
			<aspect name="TypeMetaDataProvider" id="TypeMetaDataProvider" SG:visibility="private">
				<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@stereotype='type']">
					<xsl:variable name="class-id" select="@xmi.id"/>
					<xsl:variable name="t-type" select="concat(substring(@name,1,1),translate(substring(@name,2,string-length(@name)-1),$upper-case,$lower-case))"/>
					<xsl:variable name="base-type" select="UML:TaggedValue[@tag='originaltype']"/>
					<xsl:variable name="is-enum" select="UML:TaggedValue[@tag='enumaration']"/>
					<xsl:if test="UML:TaggedValue[@tag='option']">
						<property name="{@full-name}" custom_type="{@full-name}" SG:visibility="public" text="{@full-name} type">
							<xsl:attribute name="type">
								<xsl:choose>
									<xsl:when test="$is-enum">String</xsl:when>
									<xsl:otherwise><xsl:value-of select="$base-type"/></xsl:otherwise>
								</xsl:choose>
							</xsl:attribute>
						</property>
					</xsl:if>
				</xsl:for-each>
				<property name="com.sap.dictionary.date" SG:visibility="public" type="Date" text="date type"/>
				<property name="com.sap.dictionary.integer" SG:visibility="public" type="Integer" text="int type"/>
				<property name="com.sap.dictionary.string" SG:visibility="public" type="String" text="string type"/>
				<property name="com.sap.dictionary.double" SG:visibility="public" type="Double" text="double type"/>
				<property name="com.sap.dictionary.boolean" SG:visibility="public" type="Boolean" text="boolean type"/>
				<property name="com.sap.dictionary.decimal" SG:visibility="public" type="DECIMAL" text="decimal type"/>
			</aspect>
			<aspect name="TypeAccessPK" key="true" id="TypeAccessPK"/>
			<application id="{$app-id}" provider="{$app-provider}"/>
		</coolservice>
	</xsl:template>

	<xsl:template match="UML:Class" mode="s-externalservice">
		<SG:externalservice name="{@full-name}" wrapperjndi="{UML:TaggedValue[@tag='wrapperjndi']}" wrapperlocalhome="{UML:TaggedValue[@tag='wrapperlocalhome']}" servicejndi="{UML:TaggedValue[@tag='servicejndi']}" servicelocalhome="{UML:TaggedValue[@tag='servicelocalhome']}" servicelocal="{UML:TaggedValue[@tag='servicelocal']}"/>
	</xsl:template>
	
	<xsl:template name="s-aspect-content">
		<xsl:param name="add-guid"/>
		<xsl:apply-templates select="." mode="s-aspect-attributes"/>
		<xsl:if test="$add-guid">
			<property SG:visibility="public" name="guid" type="String" text="GUID"/>
		</xsl:if>
		<xsl:apply-templates select="." mode="s-aspect-children"/>
	</xsl:template>
	
	<xsl:template match="UML:Class" mode="s-aspect-attributes">
		<xsl:attribute name="SG:stereotype"><xsl:value-of select="@stereotype"/></xsl:attribute>
		<xsl:if test="UML:TaggedValue[@tag='objectID']">
			<xsl:attribute name="SG:objectID"><xsl:value-of select="UML:TaggedValue[@tag='objectID']"/></xsl:attribute>
		</xsl:if>
		<xsl:if test="UML:TaggedValue[@tag='instancePermission']">
			<xsl:attribute name="SG:instancePermission">true</xsl:attribute>
		</xsl:if>
		<xsl:if test="UML:TaggedValue[@tag='tableName']">
			<xsl:attribute name="SG:tableName"><xsl:value-of select="UML:TaggedValue[@tag='tableName']"/></xsl:attribute>
		</xsl:if>
		<xsl:if test="UML:TaggedValue[@tag='interfaceID']">
			<xsl:attribute name="SG:interfaceID"><xsl:value-of select="UML:TaggedValue[@tag='interfaceID']"/></xsl:attribute>
		</xsl:if>
		<xsl:if test="@parent">
			<xsl:variable name="parent-id" select="@parent"/>
			<xsl:attribute name="SG:parent"><xsl:value-of select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$parent-id]/@name"/></xsl:attribute>
		</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Class" mode="s-aspect-children">
		<xsl:variable name="class-id" select="@xmi.id"/>
		<!-- Parent actions and properties are inherited -->
		<xsl:if test="@parent">
			<xsl:variable name="parent-id" select="@parent"/>
			<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$parent-id]" mode="s-aspect-children"/>
		</xsl:if>
		<!-- [=============== Generate property declarations ==============] -->
		<xsl:apply-templates select="UML:Attribute[@visibility!='private']" mode="s-property"/>
		<!-- [=============== For composite services: add keys for services that are 'building blocks'  for this composite service==============] -->
		<xsl:call-template name="s-detail-services"/>
		<!-- [=============== Generate private property declarations ==============] -->
		<xsl:if test="UML:Attribute[@visibility='private']">
			<SG:privateproperties>
				<xsl:apply-templates select="UML:Attribute[@visibility='private']" mode="s-property"/>
			</SG:privateproperties>
		</xsl:if>
		<!-- [=============== Generate action declarations ==============] -->
		<xsl:apply-templates select="UML:Operation[@visibility!='private' and @stereotype='Action']" mode="s-action"/>
		<!-- [=============== Generate relation specific action declarations (relation adders and removers) ==============] -->
		<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Associations/UML:Association[@source=$class-id and @visibility!='private']" mode="s-action"/>
	</xsl:template>
	
	<xsl:template name="s-detail-services">
		<xsl:variable name="allprops">
			<xsl:for-each select="UML:Attribute[UML:TaggedValue[@tag='associates']]">
				<xsl:value-of select="concat('%',@name,'%#')"/>
				<xsl:call-template name="s-associates-field"/>
				<xsl:text>#</xsl:text>
			</xsl:for-each>
		</xsl:variable>
		<xsl:for-each select="UML:Attribute[UML:TaggedValue[@tag='associates']]">
			<xsl:variable name="associates"><xsl:call-template name="s-associates-field"/></xsl:variable>
			<xsl:if test="not(contains(substring-before($allprops,concat('%',@name,'%')),concat('#',substring-before($associates,'.'),'.')))
					and not(contains($allprops,concat('#',substring-before($associates,'.'),'.guid#')))">
				<property SG:visibility="public" type="String" text="">
					<xsl:attribute name="name">
						<xsl:call-template name="java-id">
							<xsl:with-param name="id" select="substring-before($associates,'.')"/>
						</xsl:call-template>
						<xsl:text>Key</xsl:text>
					</xsl:attribute>
				</property>
			</xsl:if>
		</xsl:for-each>
		<xsl:if test="$allprops!=''">
			<SG:detail-services>
				<xsl:for-each select="UML:Attribute[UML:TaggedValue[@tag='associates']]">
					<xsl:variable name="associates"><xsl:call-template name="s-associates-field"/></xsl:variable>
					<xsl:if test="substring-after($associates,'.')='guid'">
						<detail-service name="{substring-before($associates,'.')}" foreign-key="{@name}"/>
					</xsl:if>
					<xsl:if test="not(contains(substring-before($allprops,concat('%',@name,'%')),concat('#',substring-before($associates,'.'),'.')))
							and not(contains($allprops,concat('#',substring-before($associates,'.'),'.guid#')))">
						<detail-service name="{substring-before($associates,'.')}">
							<xsl:attribute name="foreign-key">
								<xsl:call-template name="java-id">
									<xsl:with-param name="id" select="substring-before($associates,'.')"/>
								</xsl:call-template>
								<xsl:text>Key</xsl:text>
							</xsl:attribute>
						</detail-service>
					</xsl:if>
				</xsl:for-each>
			</SG:detail-services>
		</xsl:if>
	</xsl:template>
	
	<xsl:template name="s-associates-field">
		<xsl:variable name="associates-tag" select="UML:TaggedValue[@tag='associates']"/>
		<xsl:if test="string-length($associates-tag)&gt;4">
			<xsl:call-template name="s-trail">
				<xsl:with-param name="string" select="substring($associates-tag,3,string-length($associates-tag)-4)"/>
			</xsl:call-template>
		</xsl:if>
	</xsl:template>
	
	<xsl:template name="s-trail">
		<xsl:param name="string"/>
		<xsl:param name="trail"/>
		<xsl:param name="dots" select="0"/>
		<xsl:choose>
			<xsl:when test="string-length($string)&gt;0">
				<xsl:variable name="lastchar" select="substring($string,string-length($string),1)"/>
				<xsl:choose>
					<xsl:when test="$lastchar!='.' or $dots=0">
						<xsl:call-template name="s-trail">
							<xsl:with-param name="string" select="substring($string,1,string-length($string)-1)"/>
							<xsl:with-param name="trail" select="concat($lastchar,$trail)"/>
							<xsl:with-param name="dots">
								<xsl:choose>
									<xsl:when test="$dots=1 or $lastchar='.'"><xsl:value-of select="1"/></xsl:when>
									<xsl:otherwise><xsl:value-of select="0"/></xsl:otherwise>
								</xsl:choose>
							</xsl:with-param>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise><xsl:value-of select="$trail"/></xsl:otherwise>
				</xsl:choose>
			</xsl:when>
			<xsl:otherwise><xsl:value-of select="$trail"/></xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<!-- a parent->child relation -->
	<xsl:template match="UML:Class" mode="s-relation-from-parent">
		<xsl:if test="contains($class-names,concat('#',@xmi.id,'#'))">
			<aspectrelation name="{@name}" cardinality="1..1" targetaspect="{@name}" SG:visibility="{@visibility}" SG:clientcardinality="1..1">
				<xsl:attribute name="targetservice">
					<xsl:choose>
						<!-- target (child) service for aspect is the same as parentservice -->
						<xsl:when test="@stereotype='Aspect'">
							<xsl:variable name="parent-id" select="@parent"/>
							<xsl:value-of select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$parent-id]/@name"/>
						</xsl:when>
						<xsl:otherwise>
							<!-- UML:Generalization tag is a located under the child UML:Class -->
							<xsl:value-of select="@name"/>
						</xsl:otherwise>
					</xsl:choose>
				</xsl:attribute>
			</aspectrelation>
		</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Attribute" mode="s-property">
		<xsl:variable name="element-id" select="@xmi.id"/>
		<xsl:variable name="target" select="@type"/>
		<!-- check the property type against model classes -->
		<xsl:if test="not(/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@full-name=$target and (@stereotype='Composite Service' or @stereotype='Aspect' or @stereotype='BO Service')])">
			<!-- Eliminate Together special properties associated with relations by checking some specific tagged values -->
			<xsl:if test="not(UML:TaggedValue[@tag='link' or @tag='bidirectional' or @tag='directed' or @tag='undirected' or @tag='supplierCardinality' or @tag='supplierRole'])">
				<property SG:visibility="{@visibility}">
					<xsl:attribute name="name">
						<xsl:choose>
							<!-- For UI typed model we replace property 'key' by 'guid' -->
							<xsl:when test="$diagram-name!='' and @name='key'">guid</xsl:when>
							<xsl:otherwise><xsl:value-of select="@name"/></xsl:otherwise>
						</xsl:choose>
					</xsl:attribute>
					<xsl:apply-templates select="." mode="s-property-attributes"/>
				</property>
				<xsl:if test="@name='createdBy' or @name='lastChangedBy' or @type=concat($package-prefix,'dictionary.UMEUSERID')">
					<property SG:base-property="{@name}" SG:visibility="protected" type="String" text="">
						<xsl:attribute name="name">
							<xsl:text>displayed</xsl:text>
							<xsl:call-template name="title-case">
								<xsl:with-param name="string" select="@name"/>
							</xsl:call-template>
						</xsl:attribute>
					</property>
				</xsl:if>
			</xsl:if>
		</xsl:if>
	</xsl:template>

	<xsl:template name="s-type-attributes">
		<!--xsl:param name="type">
			<xsl:choose>
				<xsl:when test="name()='UML:Attribute' and (@name='createdBy' or @name='lastChangedBy')">TIMESTAMP</xsl:when>
				<xsl:otherwise><xsl:value-of select="@type"/></xsl:otherwise>
			</xsl:choose>
		</xsl:param-->
		<xsl:param name="type" select="@type"/>
		<xsl:param name="upper-cased" select="'false'"/>
			<!-- model class corresponding to data type -->
			<xsl:variable name="data-class" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[(@name=$type or @full-name=$type) and @stereotype='type']"/>
			<xsl:variable name="is-enum">
				<xsl:if test="$data-class/UML:TaggedValue[@tag='enumaration']">true</xsl:if>
			</xsl:variable>
			<xsl:variable name="custom-type">
				<xsl:if test="$data-class">
					<xsl:value-of select="$data-class/@full-name"/>
				</xsl:if>
			</xsl:variable>
			<xsl:variable name="original-type">
				<xsl:call-template name="s-data-type">
					<xsl:with-param name="data-type-name">
						<xsl:choose>
							<xsl:when test="$data-class">
								<xsl:value-of select="$data-class/UML:TaggedValue[@tag='originaltype']"/>
							</xsl:when>
							<xsl:otherwise>
								<xsl:value-of select="$type"/>
							</xsl:otherwise>
						</xsl:choose>
					</xsl:with-param>
				</xsl:call-template>
			</xsl:variable>
			<xsl:variable name="type">
				<xsl:choose>
					<xsl:when test="$is-enum='true'">String</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="$original-type"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:variable>
			<xsl:attribute name="type">
				<xsl:choose>
					<xsl:when test="$upper-cased='true' and contains($primitive-type-wrappers,concat('#',$type,'#'))">
						<xsl:value-of select="translate($type,$lower-case,$upper-case)"/>
					</xsl:when>
					<xsl:otherwise><xsl:value-of select="$type"/></xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
			<xsl:if test="$custom-type!=''">
				<xsl:attribute name="custom_type">
					<xsl:value-of select="$custom-type"/>
				</xsl:attribute>
			</xsl:if>
			<xsl:if test="$is-enum='true'">
				<xsl:attribute name="SG:enumeration">
					<xsl:value-of select="$original-type"/>
				</xsl:attribute>
			</xsl:if>
			<xsl:if test="$data-class/UML:TaggedValue[@tag='notNull']">
				<xsl:attribute name="SG:not-null"/>
			</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Attribute" mode="s-property-attributes">
		<xsl:if test="UML:TaggedValue[@tag='objectID']">
			<xsl:attribute name="SG:objectID"><xsl:value-of select="UML:TaggedValue[@tag='objectID']"/></xsl:attribute>
		</xsl:if>
		<xsl:if test="UML:TaggedValue[@tag='fieldName']">
			<xsl:attribute name="SG:fieldName"><xsl:value-of select="UML:TaggedValue[@tag='fieldName']"/></xsl:attribute>
		</xsl:if>
		<xsl:call-template name="s-type-attributes"/>
		<xsl:variable name="calculable" select="UML:TaggedValue[@tag='calculable']"/>
		<xsl:if test="$calculable!=''">
			<xsl:attribute name="calculable">
				<xsl:value-of select="$calculable"/>
			</xsl:attribute>
		</xsl:if>
		<xsl:attribute name="text"><xsl:value-of select="UML:TaggedValue[@tag='documentation']"/></xsl:attribute>
		<xsl:variable name="attr-id" select="@xmi.id"/>
		<xsl:variable name="type" select="@type"/>
		<xsl:variable name="load_on_demand" select="UML:TaggedValue[@tag='load_on_demand']"/>
		<xsl:variable name="default-value" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@full-name=$type]/UML:TaggedValue[@tag='dbDefault']"/>
		<xsl:if test="$default-value!=''">
			<xsl:attribute name="SG:default"><xsl:value-of select="$default-value"/></xsl:attribute>
		</xsl:if>
		<xsl:if test="$load_on_demand">
			<xsl:attribute name="load_on_demand">true</xsl:attribute>
			<xsl:attribute name="SG:custom-load-attribute"/>
		</xsl:if>
		<xsl:variable name="associates"><xsl:call-template name="s-associates-field"/></xsl:variable>
		<xsl:if test="$associates!=''">
			<xsl:variable name="parent-prop" select="substring-after($associates,'.')"/>
			<xsl:attribute name="SG:parent-service"><xsl:value-of select="substring-before($associates,'.')"/></xsl:attribute>
			<xsl:attribute name="SG:parent-property">
				<xsl:choose>
					<xsl:when test="$parent-prop!=''"><xsl:value-of select="$parent-prop"/></xsl:when>
					<xsl:otherwise><xsl:value-of select="@name"/></xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
		</xsl:if>
	</xsl:template>

	<xsl:template match="UML:Operation" mode="s-method">
		<xsl:if test="@stereotype='Query' or @stereotype='Find-by query' or @stereotype='Action'">
			<xsl:variable name="semantics" select="UML:TaggedValue[@tag='semantics']"/>
			<method name="{@name}" SG:visibility="{@visibility}" SG:stereotype="{@stereotype}">
				<xsl:if test="UML:TaggedValue[@tag='objectID']">
					<xsl:attribute name="SG:objectID"><xsl:value-of select="UML:TaggedValue[@tag='objectID']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='inputMessageID']">
					<xsl:attribute name="SG:inputMessageID"><xsl:value-of select="UML:TaggedValue[@tag='inputMessageID']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='outputMessageID']">
					<xsl:attribute name="SG:outputMessageID"><xsl:value-of select="UML:TaggedValue[@tag='outputMessageID']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='permissionID']">
					<xsl:attribute name="SG:permissionID"><xsl:value-of select="UML:TaggedValue[@tag='permissionID']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='implicitCheck']">
					<xsl:attribute name="SG:implicitCheck">true</xsl:attribute>
				</xsl:if>
				<!-- 0=Query, 1=Action -->
				<xsl:attribute name="methodType">
					<xsl:choose>
						<xsl:when test="@stereotype='Query' or @stereotype='Find-by query'">0</xsl:when>
						<xsl:otherwise>1</xsl:otherwise>
					</xsl:choose>
				</xsl:attribute>
				<xsl:if test="@stereotype='Query' or @stereotype='Find-by query'">
					<xsl:attribute name="returnType">java.util.Collection</xsl:attribute>
				</xsl:if>
				<xsl:if test="$semantics!=''">
					<xsl:attribute name="SG:description">
						<xsl:value-of select="$semantics"/>
					</xsl:attribute>
				</xsl:if>
				<xsl:variable name="element-id" select="@xmi.id"/>
				<xsl:if test="UML:TaggedValue[@tag='useMultipleParams']">
					<xsl:attribute name="SG:use-multipleParams"/>
				</xsl:if>
				<xsl:attribute name="SG:bo-invoker">
					<xsl:variable name="bo-invoker" select="UML:TaggedValue[@tag='boinvoker']"/>
					<xsl:choose>
						<xsl:when test="$bo-invoker!=''"><xsl:value-of select="$bo-invoker"/></xsl:when>
						<xsl:otherwise><xsl:value-of select="@name"/></xsl:otherwise>
					</xsl:choose>
				</xsl:attribute>
				<xsl:if test="UML:TaggedValue[@tag='internalCoding']">
					<SG:internalCoding><xsl:value-of select="UML:TaggedValue[@tag='internalCoding']"/></SG:internalCoding>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='importsCoding']">
					<SG:importsCoding><xsl:value-of select="UML:TaggedValue[@tag='importsCoding']"/></SG:importsCoding>
				</xsl:if>
				<xsl:for-each select="UML:Parameter">
					<parameter name="{@name}">
						<xsl:call-template name="s-type-attributes">
							<xsl:with-param name="upper-cased" select="'true'"/>
						</xsl:call-template>
						<xsl:variable name="p-nameid" select="concat('parameterID_',position(),'_',@name)"/>
						<xsl:if test="../UML:TaggedValue[@tag=$p-nameid]">
							<xsl:attribute name="SG:objectID"><xsl:value-of select="../UML:TaggedValue[@tag=$p-nameid]"/></xsl:attribute>
						</xsl:if>
					</parameter>
				</xsl:for-each>
				<xsl:for-each select="UML:TaggedValue[@tag='param' or @tag='deprecated' or @tag='see' or @tag='exception' or @tag='since' or @tag='return']">
					<SG:javadoc tag="{@tag}" value="{.}"/>
				</xsl:for-each>
			</method>
		</xsl:if>
	</xsl:template>

	<xsl:template match="UML:Operation" mode="s-comments">
		<xsl:variable name="semantics" select="UML:TaggedValue[@tag='semantics']"/>
		<xsl:variable name="description" select="UML:TaggedValue[@tag='documentation']"/>
		<xsl:if test="$semantics!=''">
			<xsl:attribute name="longText">
				<xsl:value-of select="$semantics"/>
			</xsl:attribute>
		</xsl:if>
		<xsl:if test="$description!=''">
			<xsl:attribute name="shortText">
				<xsl:value-of select="$description"/>
			</xsl:attribute>
		</xsl:if>
	</xsl:template>

	<xsl:template match="UML:Operation[@stereotype='Action']" mode="s-action">
		<action name="{@name}" id="{@xmi.id}" SG:visibility="{@visibility}">
			<xsl:apply-templates select="." mode="s-comments"/>
		</action>
	</xsl:template>

	<xsl:template match="UML:Operation[@stereotype='Query' or @stereotype='Find-by query']" mode="s-query">
		<xsl:variable name="qname">
			<xsl:call-template name="title-case">
				<xsl:with-param name="string" select="@name"/>
			</xsl:call-template>
		</xsl:variable>
		<query name="{@name}" id="{@xmi.id}" SG:visibility="{@visibility}">
			<xsl:attribute name="aspect">
				<xsl:choose>
					<xsl:when test="UML:TaggedValue[@tag='target']">
						<xsl:call-template name="substring-after-last">
							<xsl:with-param name="string" select="UML:TaggedValue[@tag='target']"/>
							<xsl:with-param name="substring" select="'.'"/>
						</xsl:call-template>
					</xsl:when>
					<xsl:otherwise><xsl:value-of select="../@name"/></xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
			<xsl:apply-templates select="." mode="s-comments"/>
		</query>
	</xsl:template>
	
	<xsl:template name="s-predefined-queries">
		<query name="_provideByPks" aspect="{@name}" id="{@xmi.id}_provideByPks" shortText="Returns the set of data  beans provided their pks" SG:CAF-specific="true"/>
		<method methodType="0" name="_provideByPks" returnType="java.util.Collection" SG:CAF-specific="true">
			<parameter name="pks" type="STRING"/>
		</method>
	</xsl:template>
	
	<xsl:template match="UML:Attribute" mode="s-aspectrelation">
		<xsl:variable name="source-class-id" select="../@xmi.id"/>
		<xsl:variable name="target-class-full-name" select="@type"/>
		<xsl:variable name="target-class" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@full-name=$target-class-full-name]"/>
		<xsl:variable name="target-class-stereo-type" select="$target-class/@stereotype"/>
		<!--xsl:variable name="element-id" select="@xmi.id"/-->
		<xsl:if test="contains($class-names,concat('#',$target-class/@xmi.id,'#'))">
		<!-- undirected class relations do not cause aspect relations -->
		<xsl:if test="not(UML:TaggedValue[@tag='undirected'])">
		<xsl:if test="$target-class-stereo-type='BO Service' or $target-class-stereo-type='Aspect' or $target-class-stereo-type='ValueSet' or $target-class-stereo-type='Composite Service'">
			<xsl:variable name="cardinality">
				<xsl:variable name="tcardinality" select="UML:TaggedValue[@tag='supplierCardinality']"/>
				<xsl:choose>
					<xsl:when test="$tcardinality='1'">1..1</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="$tcardinality"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:variable>
			<xsl:variable name="ccardinality">
				<xsl:variable name="tcardinality" select="UML:TaggedValue[@tag='clientCardinality']"/>
				<xsl:choose>
					<xsl:when test="$tcardinality='1'">1..1</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="$tcardinality"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:variable>
			<xsl:variable name="link" select="UML:TaggedValue[@tag='link']"/>
			<xsl:variable name="target-class-name" select="$target-class/@name"/>
			<xsl:variable name="element-id" select="@xmi.id"/>
			<xsl:variable name="external-relation" select="$target-class/UML:TaggedValue[@tag='externalservice']"/>
			<aspectrelation cardinality="{translate($cardinality,'*','n')}" targetaspect="{$target-class-name}" SG:visibility="{@visibility}" SG:clientcardinality="{translate($ccardinality,'*','n')}" >
				<xsl:attribute name="name"><xsl:value-of select="UML:TaggedValue[@tag='supplierRole']"/></xsl:attribute>
				<xsl:if test="UML:TaggedValue[@tag='objectID']">
					<xsl:attribute name="SG:objectID"><xsl:value-of select="UML:TaggedValue[@tag='objectID']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='fieldName']">
					<xsl:attribute name="SG:fieldName"><xsl:value-of select="UML:TaggedValue[@tag='fieldName']"/></xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='tableName']">
					<xsl:attribute name="SG:tableName"><xsl:value-of select="UML:TaggedValue[@tag='tableName']"/></xsl:attribute>
				</xsl:if>
				<xsl:attribute name="targetservice">
					<xsl:choose>
						<!-- target service in external relations is defined by the fully-qualified name -->
						<xsl:when test="$external-relation">
							<xsl:value-of select="$target-class/@full-name"/>
						</xsl:when>
						<!-- Find service containing ralated aspect if it is not the same as source service -->
						<xsl:when test="UML:TaggedValue[@tag='targetnotaspect']">
							<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[UML:Attribute[@type=$target-class-full-name and not(UML:TaggedValue[@tag='targetnotaspect'])]]">
								<xsl:value-of select="@name"/>
							</xsl:for-each>
						</xsl:when>
						<!-- target service in realtions to aspects is the same as source service -->
						<xsl:when test="$target-class-stereo-type='Aspect'">
							<xsl:value-of select="../@name"/>
						</xsl:when>
						<xsl:otherwise>	<xsl:value-of select="$target-class-name"/></xsl:otherwise>
					</xsl:choose>
				</xsl:attribute>
				<xsl:if test="$external-relation">
					<xsl:attribute name="SG:externalrelation">true</xsl:attribute>
				</xsl:if>
				<xsl:if test="$link!=''">
					<xsl:attribute name="SG:relationtype">composition</xsl:attribute>
				</xsl:if>
				<xsl:attribute name="SG:clientRole">
					<xsl:value-of select="UML:TaggedValue[@tag='clientRole']"/>
				</xsl:attribute>
				<xsl:variable name="foreignkeys" select="UML:TaggedValue[@tag='foreignkeys']"/>
				<xsl:if test="UML:TaggedValue[@tag='genfinder']">
					<xsl:attribute name="SG:gen-finder">true</xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='useMultipleParams']">
					<xsl:attribute name="SG:use-multipleParams"/>
				</xsl:if>
				<xsl:call-template name="s-foreign-key-attributes">
					<xsl:with-param name="clientforeignkey-default" select="$target-class-name"/>
					<xsl:with-param name="foreignkey-default" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$source-class-id]/@name"/>
					<xsl:with-param name="clientforeignkey-exists" select="$foreignkeys='forward' or $foreignkeys='both'"/>
					<xsl:with-param name="foreignkey-exists" select="$foreignkeys='back' or $foreignkeys='both'"/>
					<xsl:with-param name="clientforeignkey-tag" select="'bforeignkey'"/>
					<xsl:with-param name="foreignkey-tag" select="'foreignkey'"/>
					<xsl:with-param name="bofinder-tag" select="'bofinder'"/>
				</xsl:call-template>
			</aspectrelation>
		</xsl:if>
		</xsl:if>
		</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Attribute" mode="s-aspectrelation-back">
		<!--xsl:variable name="element-id" select="@xmi.id"/-->
		<xsl:if test="UML:TaggedValue[@tag='bidirectional']!=''">
			<xsl:variable name="source-class-id" select="../@xmi.id"/>
			<xsl:variable name="target-class-full-name" select="@full-name"/>
			<xsl:variable name="cardinality">
				<xsl:variable name="tcardinality" select="UML:TaggedValue[@tag='clientCardinality']"/>
				<xsl:choose>
					<xsl:when test="$tcardinality='1'">1..1</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="$tcardinality"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:variable>
			<xsl:variable name="source-class-name" select="../@name"/>
			<aspectrelation cardinality="{translate($cardinality,'*','n')}" targetaspect="{$source-class-name}" targetservice="{$source-class-name}" SG:indirect="true" SG:visibility="{@visibility}">
				<xsl:attribute name="name"><xsl:value-of select="UML:TaggedValue[@tag='clientRole']"/></xsl:attribute>
				<xsl:variable name="foreignkeys" select="UML:TaggedValue[@tag='foreignkeys']"/>
				<xsl:if test="UML:TaggedValue[@tag='bgenfinder']">
					<xsl:attribute name="SG:gen-finder">true</xsl:attribute>
				</xsl:if>
				<xsl:if test="UML:TaggedValue[@tag='buseMultipleParams']">
					<xsl:attribute name="SG:use-multipleParams"/>
				</xsl:if>
				<xsl:call-template name="s-foreign-key-attributes">
					<xsl:with-param name="clientforeignkey-default" select="$source-class-name"/>
					<xsl:with-param name="foreignkey-default" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@full-name=$target-class-full-name]/@name"/>
					<xsl:with-param name="clientforeignkey-exists" select="$foreignkeys='back' or $foreignkeys='both'"/>
					<xsl:with-param name="foreignkey-exists" select="$foreignkeys='forward' or $foreignkeys='both'"/>
					<xsl:with-param name="clientforeignkey-tag" select="'foreignkey'"/>
					<xsl:with-param name="foreignkey-tag" select="'bforeignkey'"/>
					<xsl:with-param name="bofinder-tag" select="'bbofinder'"/>
				</xsl:call-template>
			</aspectrelation>
		</xsl:if>
	</xsl:template>

	<xsl:template name="s-foreign-key-attributes">
		<xsl:param name="clientforeignkey-default"/>
		<xsl:param name="foreignkey-default"/>
		<xsl:param name="clientforeignkey-exists"/>
		<xsl:param name="foreignkey-exists"/>
		<xsl:param name="clientforeignkey-tag"/>
		<xsl:param name="foreignkey-tag"/>
		<xsl:param name="bofinder-tag"/>
		<xsl:variable name="clientforeignkey" select="UML:TaggedValue[@tag=$clientforeignkey-tag]"/>
		<xsl:variable name="foreignkey" select="UML:TaggedValue[@tag=$foreignkey-tag]"/>
		<xsl:variable name="bofinder" select="UML:TaggedValue[@tag=$bofinder-tag]"/>
		<xsl:if test="$clientforeignkey-exists">
			<xsl:attribute name="SG:clientforeignkey">
				<xsl:choose>
					<xsl:when test="$clientforeignkey!=''"><xsl:value-of select="$clientforeignkey"/></xsl:when>
					<xsl:otherwise><xsl:value-of select="$clientforeignkey-default"/></xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
		</xsl:if>
		<xsl:if test="$foreignkey-exists">
			<xsl:attribute name="SG:foreignkey">
				<xsl:choose>
					<xsl:when test="$foreignkey!=''"><xsl:value-of select="$foreignkey"/></xsl:when>
					<xsl:otherwise><xsl:value-of select="$foreignkey-default"/></xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
			<xsl:attribute name="SG:bofinder">
				<xsl:choose>
					<xsl:when test="$bofinder!=''">
						<xsl:value-of select="$bofinder"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>findBy</xsl:text>
						<xsl:call-template name="title-case">
							<xsl:with-param name="string">
								<xsl:choose>
									<xsl:when test="$foreignkey!=''"><xsl:value-of select="$foreignkey"/></xsl:when>
									<xsl:otherwise><xsl:value-of select="$foreignkey-default"/></xsl:otherwise>
								</xsl:choose>
							</xsl:with-param>
						</xsl:call-template>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
		</xsl:if>
	</xsl:template>

	<xsl:template match="UML:Association" mode="s-find-related-aspect-id">
		<xsl:variable name="target-class-id" select="@target"/>
		<xsl:variable name="target-class-stereo-type" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$target-class-id]/@stereotype"/>
		<xsl:choose>
			<xsl:when test="$target-class-stereo-type='Aspect' or $target-class-stereo-type='ValueSet'">
				<xsl:value-of select="$target-class-id"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:variable name="source-class-id" select="@source"/>
				<xsl:variable name="source-class-stereo-type" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$source-class-id]/@stereotype"/>
				<xsl:if test="$source-class-stereo-type='Aspect'">
					<xsl:value-of select="$source-class-id"/>
				</xsl:if>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
	
	<xsl:template match="UML:Association" mode="s-aspect">
		<xsl:variable name="related-aspect">	
			<xsl:apply-templates select="." mode="s-find-related-aspect-id"/>
		</xsl:variable>
		<xsl:variable name="element-id" select="@xmi.id"/>
		<xsl:if test="$related-aspect!='' and not(UML:TaggedValue[@tag='targetnotaspect'])">
			<xsl:variable name="source-id" select="@source"/>
			<xsl:variable name="source-stereo-type" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$source-id]/@stereotype"/>
			<xsl:for-each select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$related-aspect]">
				<aspect name="{@name}" id="{@xmi.id}">
					<xsl:call-template name="s-aspect-content">
						<xsl:with-param name="add-guid" select="$source-stereo-type='BO Service' and not(UML:Attribute[@name='guid'])"/>
					</xsl:call-template>
				</aspect>
			</xsl:for-each>
		</xsl:if>
	</xsl:template>

	<xsl:template match="UML:Association" mode="s-aspect-method">
		<xsl:variable name="related-aspect">	
			<xsl:apply-templates select="." mode="s-find-related-aspect-id"/>
		</xsl:variable>
		<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$related-aspect]/UML:Operation[@visibility!='private']" mode="s-method"/>
	</xsl:template>

	<xsl:template match="UML:Association" mode="s-aspect-query">
		<xsl:variable name="related-aspect">	
			<xsl:apply-templates select="." mode="s-find-related-aspect-id"/>
		</xsl:variable>
		<xsl:apply-templates select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$related-aspect]/UML:Operation[@visibility!='private' and (@stereotype='Query' or @stereotype='Find-by query')]" mode="s-query"/>
	</xsl:template>

	<xsl:template match="UML:Association" mode="s-method">
		<xsl:variable name="target-class-id" select="@target"/>
		<xsl:variable name="target-class" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$target-class-id]"/>
		<xsl:if test="$target-class/@stereotype='BO Service' or $target-class/@stereotype='Aspect'">
			<xsl:variable name="target-class-name" select="$target-class/@name"/>
			<xsl:variable name="id-param">
				<xsl:call-template name="java-id">
					<xsl:with-param name="id" select="$target-class-name"/>
				</xsl:call-template>
				<xsl:text>Id</xsl:text>
			</xsl:variable>
			<method name="relate{$target-class-name}" methodType="1" SG:wrapper-specific="true" SG:visibility="{@visibility}">
				<parameter name="{$id-param}" type="STRING" text="ID of the {$target-class-name}  to relate"/>
			</method>
			<method name="remove{$target-class-name}Relation" methodType="1" SG:wrapper-specific="true" SG:visibility="{@visibility}">
				<parameter name="{$id-param}" type="STRING" text="ID of the {$target-class-name}  to remove"/>
			</method>
		</xsl:if>
	</xsl:template>
	
	<xsl:template match="UML:Association" mode="s-action">
		<xsl:variable name="target-class-id" select="@target"/>
		<xsl:variable name="target-class" select="/XMI.Simplified/XMI.content/UML:Model/Classes/UML:Class[@xmi.id=$target-class-id]"/>
		<xsl:if test="$target-class/@stereotype='BO Service' or $target-class/@stereotype='Aspect'">
			<xsl:variable name="target-class-name" select="$target-class/@name"/>
			<action name="relate{$target-class-name}" id="{@xmi.id}relate" SG:wrapper-specific="true"/>
			<action name="remove{$target-class-name}Relation" id="{@xmi.id}remove" SG:wrapper-specific="true"/>
		</xsl:if>
	</xsl:template>
	
	<xsl:template name="s-data-type">
		<xsl:param name="data-type-name"/>
		<xsl:choose>
			<xsl:when test="$data-type-name='BigDecimal'">DECIMAL</xsl:when>
			<xsl:when test="$data-type-name='byte[]'">BINARY</xsl:when>
			<xsl:when test="$data-type-name='int'">Integer</xsl:when>
			<xsl:when test="$data-type-name='double'">Double</xsl:when>
			<xsl:when test="$data-type-name='boolean'">Boolean</xsl:when>
			<xsl:when test="starts-with($data-type-name,'java.lang.') or starts-with($data-type-name,'java.util.')">
				<xsl:value-of select="substring($data-type-name,11,string-length($data-type-name)-10)"/>
			</xsl:when>
			<xsl:when test="$data-type-name='java.sql.Date'">Date</xsl:when>
			<xsl:when test="$data-type-name='java.sql.Timestamp' or translate($data-type-name,$lower-case,$upper-case)='TIMESTAMP'">TIMESTAMP</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$data-type-name"/>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:template>
</xsl:stylesheet>
