/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.objecttypehandler;

import com.sap.tc.logging.Location;

import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.repository.*;
import com.sapportals.wcm.service.IServiceTypesConst;
import com.sapportals.wcm.util.log.*;
import com.sapportals.wcm.util.logging.LoggingFormatter;
import com.sapportals.wcm.util.regex.PathPatternMatcher;
import com.sapportals.wcm.util.uri.RID;
import java.util.*;

/**
 * TODO: Description of the interface.
 *
 * @author d029266
 */
public class ObjectTypeHandlerUtil {

  private final static int VALUE_PATH = 8;
  private final static int VALUE_RESTYPE = 4;
  private final static int VALUE_MIMETYPE = 2;
  private final static int VALUE_EXTENSION = 1;
  private final static String URL_SEPARATOR = "/";
  private final static String LINE_BREAK = System.getProperty("line.separator");

  private static com.sap.tc.logging.Location log = com.sap.tc.logging.Location.getLocation(com.sapportals.wcm.service.objecttypehandler.ObjectTypeHandlerUtil.class);

  private IResource res;

  /**
   * @param res if <code>null</code> is given, the class cannot be constructed.
   * @exception WcmException Exception raised in failure situation
   * @throws WcmException if the given <code>IResource</code> is <code>null
   *      </code>.
   */
  public ObjectTypeHandlerUtil(IResource res)
    throws WcmException {
    if (res == null) {
      throw new WcmException("No Resource given. The class cannot operate without it.");
    }
    this.res = res;
  }

  /**
   * @param list TBD: Description of the incoming method parameter
   * @return mergedActions
   * @exception WcmException Exception raised in failure situation
   * @Properties a list of properties retrieved from all OTHs in the given list
   *      for the given resource. If more than one OTH specifies a value for a
   *      property, the value of the best-matching oth (the algorithm to specify
   *      that best-match are inside this class as well) will be used
   */
  public IActionList getMergedActions(IObjectTypeHandlerList list)
    throws WcmException {
    if ((list == null) || (list.size() == 0)) {
      return null;
    }
    IActionList result = list.get(0).getActionList().subList(0, 0);
    IObjectTypeHandlerList temp = this.rankHandlers(list);
    IObjectTypeHandlerListIterator iter = temp.listIterator(temp.size());
    IObjectTypeHandler handler = null;
    while (iter.hasPrevious()) {
      handler = iter.previous();
      this.mergeActions(result, handler.getActionList());
    }
    return result;
  }

  private void mergeActions(IActionList master, IActionList added) {
    for (int i = 0; i < added.size(); i++) {
      //check if we have same action in master list -> remove that and add this one
      for (int j = 0; j < master.size(); j++) {
        if (master.get(j).getActionType().equals(added.get(i).getActionType())) {
          master.remove(j);
          break;
        }
      }
      master.add(added.get(i));
    }
  }


  /**
   * @param list TBD: Description of the incoming method parameter
   * @return mergedProperties
   * @exception WcmException Exception raised in failure situation
   * @Properties a list of properties retrieved from all OTHs in the given list
   *      for the given resource. If more than one OTH specifies a value for a
   *      property, the value of the best-matching oth (the algorithm to specify
   *      that best-match are inside this class as well) will be used
   */
  public Properties getMergedProperties(IObjectTypeHandlerList list)
    throws WcmException {
    Properties result = new Properties();
    IObjectTypeHandlerList temp = this.rankHandlers(list);
    IObjectTypeHandlerListIterator iter = temp.listIterator(temp.size());
    IObjectTypeHandler handler = null;
    while (iter.hasPrevious()) {
      handler = iter.previous();
      this.mergeProperties(result, handler.getProperties());
    }
    return result;
  }

  private void mergeProperties(Properties master, Properties added) {
    Iterator iter = added.keySet().iterator();
    Object key = null;
    Object value = null;
    while (iter.hasNext()) {
      key = iter.next();
      if (added.get(key) != null) {
        master.put(key, added.get(key));
      }
    }
  }

  public Collection getMergedMultiValuedProperties(IObjectTypeHandlerList list, String propertykey)
    throws WcmException {
    Collection result = new ArrayList();
    String value = null;
    IObjectTypeHandlerList temp = this.rankHandlers(list);
    for (int i = temp.size(); i > 0; i--) {
      value = temp.get(i - 1).getProperties().getProperty(propertykey);
      if (value != null) {
        result.addAll(readOutValues(value));
      }
    }
    return result;
  }

  private Collection readOutValues(String value) {
    Collection result = new ArrayList();
    StringTokenizer token = new StringTokenizer(value, ",");
    while (token.hasMoreTokens()) {
      result.add(token.nextToken().trim());
    }
    return result;
  }

  /**
   * The ranking mechanism works after the following schema: For every selection
   * criteria given in an <code>IObjectTypeHandler</code> , the algorithms gives
   * points to this handler if a path is given -> 8 points if a resource type is
   * given -> 4 points if a mimetype is given -> 2 points if an extension is
   * given -> 1 point If several handlers have the same score after that
   * procedure, the length of the resource path acts as the tie breaker: The
   * more folders of the path are specified, the more exact the match naturally
   * becomes. If this is still tied, then the OTH-Files are valid for the exact
   * same set of resources. This points to a less-optimal configuration. In this
   * case, the behavior of the method is not determined.
   *
   * @param list a list of matching handlers
   * @return the list of handlers, ranked by matching: The best matching handler
   *      is in first position
   * @exception WcmException Exception raised in failure situation
   */
  public IObjectTypeHandlerList rankHandlers(IObjectTypeHandlerList list)
    throws WcmException {
    try {
      IObjectTypeHandlerList result = this.getService().getEmptyObjectTypeHandlerList();

      //if we only have one or none, it's easy
      if (list.size() == 0) {
        if (this.log.beDebug()) {
          String debuginfo = this.writeDebugInformation(this.res, result);
          this.log.debugT("rankHandlers(183)", debuginfo);
        }
        return result;
      }
      else if (list.size() == 1) {
        result.add(list.get(0));
        if (this.log.beDebug()) {
          String debuginfo = this.writeDebugInformation(this.res, result);
          this.log.debugT("rankHandlers(191)", debuginfo);
        }
        return result;
      }

      //give points to the single handlers
      IObjectTypeHandler handler = null;
      HashMap map = new HashMap();
      int points = 0;
      IObjectTypeHandlerListIterator iter = list.listIterator();
      while (iter.hasNext()) {
        handler = iter.next();
        points = 0;
        if (handler.getPaths().length > 0) {
          points = points + VALUE_PATH;
        }
        if (handler.getResourceTypes().size() > 0) {
          points = points + VALUE_RESTYPE;
        }
        if (handler.getMimeTypes().size() > 0) {
          points = points + VALUE_MIMETYPE;
        }
        if (handler.getExtensions().length > 0) {
          points = points + VALUE_EXTENSION;
        }
        this.getMatchingList(map, points).add(handler);
      }

      //now we have all handlers sorted in lists according to their points.
      //if we have more than one element in a list, they have to be sorted
      //by their resource path lengths
      Iterator mapiter = map.keySet().iterator();
      while (mapiter.hasNext()) {
        this.sortListByResourcePath((IObjectTypeHandlerList)map.get(mapiter.next()));
      }

      //all sublists are now sorted internally by their resourcepathlength
      //now put all the sublists again in one BIG IObjectTypeHandlerList
      result = this.mergeListFromMap(map);

      /**
       * @todo We cannot see here, if two oths have the same quality - we have
       *      to add this information to the log. Idea: ask the
       *      mergeListFromMap() if more than one oth is in the same category...
       *      Tricky
       */
      if (this.log.beDebug()) {
        String debuginfo = this.writeDebugInformation(this.res, result);
        this.log.debugT("rankHandlers(239)", debuginfo);
      }

      return result;
    }
    catch (Exception e) {
      throw new WcmException(e);
    }
  }

  /**
   * @param map TBD: Description of the incoming method parameter
   * @param points TBD: Description of the incoming method parameter
   * @return matchingList
   * @exception WcmException Exception raised in failure situation
   */
  private IObjectTypeHandlerList getMatchingList(HashMap map, int points)
    throws WcmException {
    Iterator iter = map.keySet().iterator();
    Integer val = null;
    while (iter.hasNext()) {
      val = (Integer)iter.next();
      if (val.intValue() == points) {
        return (IObjectTypeHandlerList)map.get(val);
      }
    }
    //could not find matching entry -> create one!
    IObjectTypeHandlerList list = this.getService().getEmptyObjectTypeHandlerList();
    map.put(new Integer(points), list);
    return list;
  }

  private void sortListByResourcePath(IObjectTypeHandlerList list)
    throws WcmException {
    //check if we have more than one at all
    if ((list.size() == 0) || (list.size() == 1)) {
      return;
    }

    String uri = this.res.getRID().getPath();
    //check the path for best match. The longest one must be the best!
    int length = 0;
    int localpathlength = 0;
    HashMap map = new HashMap();
    Iterator iter = null;
    for (int i = 0; i < list.size(); i++) {
      length = this.getBestMatchingPath(list.get(i).getPaths(), uri);
      this.getMatchingList(map, length).add(list.get(i));
    }
    //we are done! all handlers are in different lists depending on their resourcepathlength
    //now merge them all into the given list again
    //iterate first over list to remove all handlers from it
    IObjectTypeHandlerListIterator listiter = list.listIterator();
    while (listiter.hasNext()) {
      listiter.next();
      listiter.remove();
    }
    list.add(this.mergeListFromMap(map));
  }

  /**
   * The given map consists of key-value pairs with key - an Integer,
   * representing the value of the handlers that are contained in the value - an
   * IObjectTypeHandlerList The method loops through the map, first identifying
   * the key with the highest value. Its list is added into the result list. The
   * key-value-pair is then removed from the map. Now we iterate again through
   * the map, identifying the key with the highest value. As we removed the
   * key-value-pair we found before, the next highest pair is found, the list is
   * appended to the result list and the key-value-pair is removed from the map.
   * This continues until all lists have been appended to the result list.
   *
   * @param map TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   * @exception WcmException Exception raised in failure situation
   */
  private IObjectTypeHandlerList mergeListFromMap(HashMap map)
    throws WcmException {
    IObjectTypeHandlerList result = this.getService().getEmptyObjectTypeHandlerList();
    Integer val = null;
    int max = -1;
    Integer key = null;
    while (map.size() > 0) {
      Iterator iter = map.keySet().iterator();
      max = -1;
      while (iter.hasNext()) {
        val = (Integer)iter.next();
        if (val.intValue() > max) {
          max = val.intValue();
          key = val;
        }
      }
      //we have the list with the highest score. add this list AT THE END of our result list
      result.add((IObjectTypeHandlerList)map.get(key));
      map.remove(key);
    }
    return result;
  }

  private int getBestMatchingPath(String[] paths, String uri) {
    String trimmedpath = null;
    int length = 0;
    int maxlength = 0;
    int penalty = 0;
    String path = null;
    for (int i = 0; i < paths.length; i++) {
      //first count number of path-elements that do hold a name instead
      //of just a "*" to see how "well" the path matches
      path = paths[i];
      length = 10;
      for (int j = 0; j < path.length() - 1; j++) {
        if (path.substring(j, j + 1).equals(URL_SEPARATOR)) {
          //we are behind a path separator - now check, if we have no "*"
          if (((j + 2) < path.length()) && (!path.substring(j + 1, j + 2).equals("*"))) {
            length++;
          }
        }
      }
      if (paths[i].endsWith("**")) {
        trimmedpath = paths[i].substring(0, paths[i].length() - 2);
        penalty = -2;
      }
      else if (paths[i].endsWith("*")) {
        trimmedpath = paths[i].substring(0, paths[i].length() - 1);
        penalty = -1;
      }
      else {
        trimmedpath = paths[i];
        penalty = 0;
      }
//      if (uri.startsWith(trimmedpath)) {
      if (this.pathmatches(paths[i], uri)) {
        length = length * 2 + penalty;
        if (length > maxlength) {
          maxlength = length;
        }
      }
    }
    return maxlength;
  }

  private boolean pathmatches(String path, String uri) {
    PathPatternMatcher pm = new PathPatternMatcher(path);
    return pm.matches(uri);
  }

  public String writeDebugInformation(IResource res, IObjectTypeHandlerList result) {
    try {
      StringBuffer buffer = new StringBuffer();
      buffer.append("Resource < " + res.getRID().getPath() + " >");
      buffer.append(LINE_BREAK);
      buffer.append("Found " + result.size() + " matching handlers with this ranking (best first):");
      buffer.append(LINE_BREAK);
      for (int i = 0; i < result.size(); i++) {
        buffer.append(result.get(i).getName() + LINE_BREAK);
      }
      return buffer.toString();
    }
    catch (Throwable th) {
      return "Exception occured, when trying to write debug information:" + th.getMessage();
    }
  }

  private IObjectTypeHandlerService getService()
    throws WcmException {
    return (IObjectTypeHandlerService)ResourceFactory.getInstance().getServiceFactory().getService(IServiceTypesConst.OTH_SERVICE);
  }
}
