/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sapportals.wcm.service.objecttypehandler;
import com.sapportals.wcm.*;
import com.sapportals.wcm.IWcmConst;
import com.sapportals.wcm.WcmException;
import com.sapportals.wcm.WcmObject;
import com.sapportals.wcm.repository.IResource;
import com.sapportals.wcm.repository.ResourceException;
import com.sapportals.wcm.repository.ResourceFactory;
import com.sapportals.wcm.repository.service.IRepositoryServiceFactory;
import com.sapportals.wcm.service.IService;
import com.sapportals.wcm.service.IServiceTypesConst;
import com.sapportals.wcm.service.urlgenerator.IURLGenerator;
import com.sapportals.wcm.util.regex.PathPatternMatcher;
import com.sapportals.wcm.util.uri.URL;
import java.util.*;

/**
 * @todo: Description of the class.
 */
public class GenericObjectTypeHandler extends WcmObject implements IObjectTypeHandler {

  protected String name = null;
  protected Vector resourceTypes = new Vector();
  protected Vector extensions = new Vector();
  protected Vector mimeTypes = new Vector();
  protected Vector paths = new Vector();
  protected Properties properties = new Properties();

  protected String[] extensionArray = null;
//  protected  String[]       mimeTypeArray   = null;
  protected String[] pathArray = null;
  protected IActionList actionList = new ActionList();

  private PathPatternMatcher patternMatcher = null;
  private Integer pathsHashCode;

  private Vector othSourceResourceTypes = new Vector();
  private Vector othTargetResourceTypes = new Vector();
  private Map resourceTypePatternMatcherMap = new HashMap();
  private Map resourceTypeHits = new HashMap();

  /**
   * Construct object of class GenericObjectTypeHandler.
   *
   * @exception WcmException Exception raised in failure situation
   */
  public GenericObjectTypeHandler()
    throws WcmException {
    try {
      IRepositoryServiceFactory sf = ResourceFactory.getInstance().getServiceFactory();

      Vector eVector = new Vector();

      this.initialize("", eVector, eVector, eVector, new ActionList(), new Vector(), new Properties());
    }
    catch (ResourceException ex) {
      com.sap.tc.logging.Location.getLocation(this.getClass()).debugT(ex.getMessage());
    }
  }// constructor

  /**
   * Set the Name attribute of the GenericObjectTypeHandler object.
   *
   * @param name The new Name value
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Set the ActionList attribute of the GenericObjectTypeHandler object.
   *
   * @param actionList The new ActionList value
   */
  public void setActionList(IActionList actionList) {
    this.actionList = actionList;
  }

  /**
   * Set the Properties attribute of the GenericObjectTypeHandler object.
   *
   * @param resourceTypes resourceTypes to be set
   */
  public void setResourceTypes(Vector resourceTypes) {
    this.resourceTypes = (Vector)resourceTypes.clone();

    othSourceResourceTypes = new Vector();
    othTargetResourceTypes = new Vector();
    resourceTypePatternMatcherMap = new HashMap();
    resourceTypeHits = new HashMap();
    Iterator it = this.resourceTypes.iterator();
    while (it.hasNext()) {
      IResourceType irt = (IResourceType)it.next();
      if (irt.getHandling().equals(IResourceType.SOURCE_EXCLUSIVE)) {
        othSourceResourceTypes.add(irt.getResourceType());
      }
      else if (irt.getHandling().equals(IResourceType.TARGET_EXCLUSIVE)) {
        othTargetResourceTypes.add(irt.getResourceType());
      }
      else {
        othSourceResourceTypes.add(irt.getResourceType());
        othTargetResourceTypes.add(irt.getResourceType());
      }
      PathPatternMatcher pm = new PathPatternMatcher(irt.getResourceType());
      resourceTypePatternMatcherMap.put(irt.getResourceType(), pm);
    }
  }

  /**
   * Set the Properties attribute of the GenericObjectTypeHandler object.
   *
   * @param props properties to be set
   */
  public void setProperties(Properties props) {
    this.properties = (Properties)props.clone();
  }

  /**
   * Set the Extensions attribute of the GenericObjectTypeHandler object.
   *
   * @param extensions The new Extensions value
   */
  public void setExtensions(Vector extensions) {
    this.extensions = (Vector)extensions.clone();
    this.extensionArray = this.copyStringVector2Array(this.extensions);
  }

  /**
   * Set the MimeTypes attribute of the GenericObjectTypeHandler object.
   *
   * @param mimeTypes The new MimeTypes value
   */
  public void setMimeTypes(Vector mimeTypes) {
    this.mimeTypes = (Vector)mimeTypes.clone();
//    this.mimeTypeArray = this.copyStringVector2Array(this.mimeTypes);
  }

  /**
   * Set the Paths attribute of the GenericObjectTypeHandler object.
   *
   * @param paths The new Paths value
   */
  public void setPaths(Vector paths) {
    this.paths = (Vector)paths.clone();
    this.pathArray = this.copyStringVector2Array(this.paths);

    // unfortunately this is not really secure because pathArray is only protected
    if (this.pathArray.length > 0) {
      // I have cloned this only because I don't know if the order is sometimes relevant
      Vector helpVector = (Vector)paths.clone();
      Collections.sort(helpVector);
      patternMatcher = new PathPatternMatcher(this.pathArray);
      pathsHashCode = new Integer(helpVector.hashCode());
    }
    else {
      patternMatcher = null;
    }
  }

  public boolean matches(String needle) {
    if (patternMatcher == null) {
      return false;
    }
    else {
      return patternMatcher.matches(needle);
    }
  }

  public boolean matchesResourceType(String testResourceType, String othResourceType) {
    PathPatternMatcher pm
       = (PathPatternMatcher)resourceTypePatternMatcherMap.get(othResourceType);
    if (pm == null) {
      return false;
    }
    else {
      String key = new String(testResourceType + "&_&" + othResourceType);
      Boolean test = (Boolean)resourceTypeHits.get(key);
      if (test == null) {
        boolean matchResult = pm.matches(testResourceType);
        resourceTypeHits.put(key, new Boolean(matchResult));
        return matchResult;
      }
      else if (test.equals(Boolean.TRUE)) {
        return true;
      }
      else {
        return false;
      }
    }
  }

  /**
   * Returns identical values for identical paths arrays.
   *
   * @return
   */
  public Integer getPathsHashCode() {
    return pathsHashCode;
  }


  /**
   * Get the Name attribute of the GenericObjectTypeHandler object.
   *
   * @return The Name value
   * @exception WcmException Exception raised in failure situation
   */
  public String getName()
    throws WcmException {
    return this.name;
  }

  /**
   * Get the ActionList attribute of the GenericObjectTypeHandler object.
   *
   * @return The ActionList value
   */
  public IActionList getActionList() {
    return this.actionList;
  }

  /**
   * Get the URL attribute of the GenericObjectTypeHandler object.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param actionType TBD: Description of the incoming method parameter
   * @return The URL value
   * @exception WcmException Exception raised in failure situation
   */
  public URL getURL(IResource resource, IActionType actionType)
    throws WcmException {
    IAction action = actionList.getAction(actionType);

    if (null != action) {
      IService s = ResourceFactory.getInstance().getServiceFactory().getService(IServiceTypesConst.URLGENERATOR_SERVICE);
      if (s == null) {
        throw new WcmException("URLGenerator is not available");
      }
      String replaced = ((IURLGenerator)s).replaceURIPlaceholders(action.getURLTemplate().toString(), resource);
      return new URL(replaced);
    }
    else {
      throw new WcmException("action not supported!");
    }
  }

  /**
   * Get the URL attribute of the GenericObjectTypeHandler object.
   *
   * @param resource TBD: Description of the incoming method parameter
   * @param actionType TBD: Description of the incoming method parameter
   * @param props TBD: Description of the incoming method parameter
   * @return The URL value
   * @exception WcmException Exception raised in failure situation
   */
  public URL getURL(IResource resource, IActionType actionType, Properties props)
    throws WcmException {
    /**
     * @todo: Implement this com.sapportals.wcm.objecttypehandler.IObjectTypeHandler
     *      method
     */
    throw new java.lang.UnsupportedOperationException("Method getURI() not yet implemented.");
  }

  /**
   * Get the Properties attribute of the GenericObjectTypeHandler object.
   *
   * @return The Properties value
   */
  public Vector getResourceTypes() {
    return this.resourceTypes;
  }

  public Vector getSourceResourceTypes() {
    return this.othSourceResourceTypes;
  }

  public Vector getTargetResourceTypes() {
    return this.othTargetResourceTypes;
  }

  /**
   * Get the Properties attribute of the GenericObjectTypeHandler object.
   *
   * @return The Properties value
   */
  public Properties getProperties() {
    return this.properties;
  }

  /**
   * Get the Extensions attribute of the GenericObjectTypeHandler object.
   *
   * @return The Extensions value
   */
  public String[] getExtensions() {
    return this.extensionArray;
  }

  /**
   * Get the MimeTypes attribute of the GenericObjectTypeHandler object.
   *
   * @return The MimeTypes value
   */
  public Vector getMimeTypes() {
    return this.mimeTypes;
  }

  /**
   * Get the Paths attribute of the GenericObjectTypeHandler object.
   *
   * @return The Paths value
   */
  public String[] getPaths() {
    return this.pathArray;
  }

  /**
   * TBD: Description of the Method.
   *
   * @param name TBD: Description of the incoming method parameter
   * @param extensions TBD: Description of the incoming method parameter
   * @param mimeTypes TBD: Description of the incoming method parameter
   * @param paths TBD: Description of the incoming method parameter
   * @param actionList TBD: Description of the incoming method parameter
   * @param props TBD: Description of the incoming method parameter
   * @param resourceTypes TBD: Description of the incoming method parameter
   * @exception WcmException Exception raised in failure situation
   */
  public void initialize(String name, Vector extensions, Vector mimeTypes, Vector paths, IActionList actionList, Vector resourceTypes, Properties props)
    throws WcmException {
    this.setName(name);
    this.setResourceTypes(resourceTypes);
    this.setExtensions(extensions);
    this.setMimeTypes(mimeTypes);
    this.setPaths(paths);
    this.setActionList(actionList);
    this.setProperties(props);
  }

  /**
   * TBD: Description of the Method.
   *
   * @param stringVector TBD: Description of the incoming method parameter
   * @return TBD: Description of the outgoing return value
   */
  private String[] copyStringVector2Array(Vector stringVector) {
    String[] strArray = new String[stringVector.size()];
    stringVector.toArray(strArray);
    return strArray;
  }

  /**
   * Must be called in startUp() method of an OTH service. Gets a reference to
   * the URLGenerator Service.
   *
   * @exception WcmException Exception raised in failure situation
   */
  public void startUpOTH()
    throws WcmException { }

}// class GenericObjectTypeHandler
