/*
 * Copyright (c) 2002 SAP AG - All Rights Reserved.
 */
package com.sap.tc.cmi.util;

import java.util.EventObject;

/**
 * Event object that describes changes in an {@link ICMIObservableList}. </p>
 *  
 * This type can be called by applications or frameworks using CMI and 
 * implementations of CMI can (but usually have not to) subclass it. </p>
 * 
 * @SAPCMIPart 1
 * @author Frank Weigel
 * @version $Id: //tc/CommonModelInterface/630_VAL_REL/src/_cmi_api/java/com/sap/tc/cmi/util/CMIObservableListEvent.java#1 $
 */
public class CMIObservableListEvent extends EventObject {

  /**
   * Event type when list has been cleared. 
   * Event Object will contain no further useful data.
   */
  public static final int LIST_CLEARED = 0;
  
  /**
   * Event type when a single element has been added.
   * Event Object contains the added element, the index where 
   * the element has been added (fromIndex == toIndex).
   * Implies an index increment by 1 for any existing element 
   * after the insert position.
   */
  public static final int ELEMENT_ADDED = 1;

  /**
   * Event type used when a single element has been changed.
   * Event Object contains the old element, the index where 
   * the element has been changed (fromIndex == toIndex).
   */
  public static final int ELEMENT_CHANGED = 2;

  /**
   * Event type used when a single element has been removed.
   * Event Object contains the old element, the index where 
   * the element has been removed (fromIndex == toIndex).
   * Implies an index decrement by 1 for any existing element 
   * after the remove position.
   */
  public static final int ELEMENT_REMOVED = 3;

  /**
   * Event type when multiple elements have been added en block.
   * Event Object contains the index of the first and the last inserted
   * element. 
   * Implies an index increment by toIndex-fromIndex+1 for any existing element 
   * after the insert position.
   */
  public static final int CONTIGUOUS_ELEMENTS_ADDED = 4;
  
  /**
   * Event type when multiple elements have been removed en block.
   * Event Object contains the index of the first and the last removed
   * element. 
   * Implies an index decrement by toIndex-fromIndex+1 for any existing element 
   * after the insert position.
   * @todo currently no way to determine removed objects
   */
  public static final int CONTIGUOUS_ELEMENTS_REMOVED = 5;

  /**
   * Event type when multiple elements have been removed at not necessarily contiguous positions.
   * Event Object contains no more information.
   * Implies index changes for any existing element after the first remove position.
   * @todo currently no way to determine removed objects/index change
   */
  public static final int MULTIPLE_ELEMENTS_REMOVED = 6;

  /**
   * Constructor for CMIObservableListEvent.
   * @param source
   */
  public CMIObservableListEvent(ICMIObservableList source) {
    super(source);
  }

  /**
   * Creates an event object with the given type and parameters for the given source. </p>
   * 
   * <code>source</code> must not be <code>null</code> and <code>type</code> must be one of the event 
   * types defined in this class. Whether the other parameters have to be set, depends on the event 
   * types and is documented with these types. </p>
   * 
   * This constructor can be called by implementations that use CMIObservableListEvent as it is 
   * as well as by implementations that have to subclass it. </p>
   */
  public CMIObservableListEvent(ICMIObservableList source, int type, Object element, int fromIndex, int toIndex) {
    super(source);
    setData(element, fromIndex, toIndex, type);    
  }
  
  public final ICMIObservableList getObservableList() {
    return (ICMIObservableList) getSource();
  }

  public final int getFromIndex() {
    return fromIndex;
  }
  
  public final int getToIndex() {
    return toIndex;
  }

  public final int getChangeType() {
    return type;
  }
  
  public final Object getElement() {
    return element;
  }

  private int fromIndex;
  
  private int toIndex;
  
  private int type;
  
  private Object element;

  /**
   * Sets the event parameters for this event object. This method is only package 
   * visible to prevent others from modifying event parameters after event creation. 
   * Users outside this package must provide all event parameters at construction 
   * time, using the approp. constructor. </p>
   * 
   * For performance reasons, the list implementations in this package are allowed 
   * to modify an already created event object, but only if they can guarantee, that 
   * such a change doesn't happen during synchronous event processing (could be opened to 
   * subclasses as well). </p>
   */ 
  void setData(Object element, int fromIndex, int toIndex, int type) {
    this.element = element;
    this.fromIndex = fromIndex;
    this.toIndex = toIndex;
    this.type = type;
  }
  
}
