/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.uuid;

import java.io.*;
import java.net.InetAddress;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Date;
import java.util.Random;

/**
 * A Universally Unique Identifier (UUID) is a 128 bit number generated
 * according to an algorithm that is garanteed to be unique in time and
 * space from all other UUIDs. It consists of an IEEE 802 Internet Address
 * and various time stamps to ensure uniqueness. For a complete specification,
 * see ftp://ietf.org/internet-drafts/draft-leach-uuids-guids-01.txt [leach].
 * <p>
 * Copyright (c) SAP AG 2001-2002
 *
 * @version    $Id: //javabas/com.sapportals.wcm/50_SP6_COR/src/java/util/api/com/sapportals/wcm/util/uuid/UUID.java#1 $
 */
public class UUID implements Serializable, Comparable
{
  private final static int UUIDsPerTick = 10000;

  private static byte[] internetAddress;
  private static long lastTime = System.currentTimeMillis();
  private static int uuidsThisTick = UUIDsPerTick;
  private static UUID previousUUID;
  private static Random randomGenerator = new Random(System.currentTimeMillis());
  private static char[] hexDigits =
    { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'a', 'b', 'c', 'd', 'e', 'f' };

  private long time;
  private short clockSequence;
  private byte version = 1;
  private byte[] node = new byte[6];
  private byte[] uuid;

  //---------------------------------------------------------------------

  static {
    try
    {
      internetAddress = InetAddress.getLocalHost().getAddress();
    }
    catch (Exception x)
    {
      //$JL-EXC$ 
      System.out.println("UUID: Can't get host address: " + x);
      x.printStackTrace();
    }
  }

  /** Generate a UUID for this host using version 1 of [leach]. */
  public UUID()
  {
    synchronized (UUID.class)
    {
      this.time = getCurrentTime();

      if (previousUUID == null || nodeChanged(previousUUID))
      {
        // if there is no saved state, or the node address changed,
        // generate a random clock sequence
        this.clockSequence = (short)randomGenerator.nextInt();
        this.node = computeNodeAddress();
      }
      else if (time < previousUUID.getTime())
      {
        // if the clock was turned back, increment the clock sequence
        this.clockSequence++;
        this.node = previousUUID.getNode();
      }
      previousUUID = this;

      this.uuid = genUUID();
    }
  }

  /**
   * Generate a UUID for this host using version 1 of [leach].
   *
   * @param  node  the node to use in the UUID
   */
  public UUID(byte[] node)
  {
    synchronized (UUID.class)
    {
      this.time = getCurrentTime();

      this.node = node;

      this.uuid = genUUID();
    }
  }

  /**
   * Create a UUID from its string representation.
   * Not that getClockSequence, getTime, getVersion and getNode might return
   * nonsense since the internal structure of the UUID will not be checked.
   *
   * @param  str                        The string representation of the uuid
   * @exception  NumberFormatException     if <tt>uuid</tt> is invalid
   */
  public UUID(String str) throws NumberFormatException
  {
    this.uuid = parseUUID(str);

    this.time = this.uuid[7] & 0xf;
    for (int i = 1; i < 8; i++)
    {
      this.time <<= 8;
      this.time |= this.uuid[7 - i];
    }
    this.version = (byte) (this.uuid[7] >> 4);
    this.clockSequence = (short) (((this.uuid[9] & 0x3F) << 8) | this.uuid[8]);
    for (int i = 0; i < 6; i++)
    {
      this.node[i] = this.uuid[10 + i];
    }
  }

  /**
   * Lexically compare this UUID with withUUID. Note: lexical ordering
   * is not temporal ordering.
   *
   * @param  withUUID  the UUID to compare with
   * @return            <ul>
   *    <li>-1 if this UUID is less than withUUID
   *    <li>0 if this UUID is equal to withUUID
   *    <li>1 if this UUID is greater than withUUID
   * </ul>
   */
  public int compareTo(Object withUUID)
  {
    if (!(withUUID instanceof UUID))
    {
      return -1;
    }

    byte[] other = ((UUID)withUUID).getUUID();
    for (int i = 0; i < 16; i++)
    {
      if (this.uuid[i] < other[i])
      {
        return -1;
      }
      else if (this.uuid[i] > other[i])
      {
        return 1;
      }
    }
    return 0;
  }

  /**
   * Get the clock sequence number.
   *
   * @return    the clock sequence number
   */
  public int getClockSequence()
  {
    return clockSequence;
  }

  /**
   * Get the spatially unique portion of the UUID. This is either
   * the 48 bit IEEE 802 host address, or if on is not available, a random
   * number that will not conflict with any IEEE 802 host address.
   *
   * @return    node portion of the UUID
   */
  public byte[] getNode()
  {
    return node;
  }

  /**
   * Get the temporal unique portion of the UUID.
   *
   * @return    the time portion of the UUID
   */
  public long getTime()
  {
    return time;
  }

  /**
   * Get the 128 bit UUID.
   *
   * @return    The uUID value
   */
  public byte[] getUUID()
  {
    return this.uuid;
  }

  /**
   * Get the UUID version number.
   *
   * @return    The version value
   */
  public int getVersion()
  {
    return version;
  }

  /**
   * Compare two UUIDs
   *
   * @param  toUUID  Description of the Parameter
   * @return         true if the UUIDs are equal
   */
  public boolean equals(Object toUUID)
  {
    return compareTo(toUUID) == 0;
  }

  /**
   * Provide a String representation of a UUID as specified in section
   * 3.5 of [leach].
   *
   * @return    Description of the Return Value
   */
  public String toString()
  {
    byte[] uuid = getUUID();

    StringBuffer buffer = new StringBuffer(256);
    for (int i = 0; i < 16; i++)
    {
      buffer.append(hexDigits[(uuid[i] & 0xF0) >> 4]);
      buffer.append(hexDigits[uuid[i] & 0x0F]);
      if (i == 3 || i == 5 || i == 7 || i == 9)
      {
        buffer.append('-');
      }
    }
    return buffer.toString();
  }

  /**
   * Generate a hash code of the uuid.
   *
   * @return    Description of the Return Value
   */
  public int hashCode()
  {
    int hash = 0;
    int i;

    for (i = 0; i < 16; i++)
    {
      hash ^= this.uuid[i];
      hash <<= 2;
    }

    return hash;
  }

  // Changed by Bodo Junglas to genUUID
  /**
   * Generate the 128 bit UUID.
   *
   * @return    Description of the Return Value
   */
  private byte[] genUUID()
  {
    byte[] uuid = new byte[16];
    long t = time;
    for (int i = 0; i < 8; i++)
    {
      uuid[i] = (byte) ((t >> 8 * i) & 0xFF);
      // time
    }
    uuid[7] |= (byte) (version << 4);
    // time hi and version
    uuid[8] = (byte) (clockSequence & 0xFF);
    uuid[9] = (byte) ((clockSequence & 0x3F00) >> 8);
    uuid[9] |= 0x80;
    // clock sequence hi and reserved
    for (int i = 0; i < 6; i++)
    {
      uuid[10 + i] = node[i];
      // node
    }
    return uuid;
  }

  /**
   * Determine if the node changed with respect to previousUUID.
   *
   * @param  previousUUID  the UUID to compare with
   * @return               true if the the previousUUID has a different node than this UUID
   */
  private boolean nodeChanged(UUID previousUUID)
  {
    if (previousUUID != null)
    {
      byte[] previousNode = previousUUID.getNode();
      for (int i = 0; i < previousNode.length; i++)
      {
        if (node[i] != previousNode[i])
        {
          return true;
        }
      }
    }
    return false;
  }

  // Added by Bodo Junglas
  /**
   * Parse a string represention of the uuid.
   *
   * @param  str                     The string representation
   * @return                         The binary representation of the uuid
   * @throws  NumberFormatException  if the string representation is invalid
   */
  private byte[] parseUUID(String str) throws NumberFormatException
  {
    StringBuffer buf = new StringBuffer(32);

    // validate the UUID here:
    for (int i = 0; i < str.length(); i++)
    {
      char ch = str.charAt(i);

      if ((ch >= '0') && (ch <= '9'))
      {
        buf.append(ch);
      }
      else if ((ch >= 'A') && (ch <= 'F'))
      {
        buf.append(ch);
      }
      else if ((ch >= 'a') && (ch <= 'f'))
      {
        buf.append(ch);
      }
      else if ((ch != ' ') && (ch != '-'))
      {
        throw new NumberFormatException("Invalid UUID string");
      }
    }

    if (buf.length() != 32)
    {
      throw new NumberFormatException("Invalid UUID string");
    }

    byte[] uuid = new byte[16];
    for (int i = 0; i < 16; i++)
    {
      uuid[i] =
        (byte) ((Character.digit(buf.charAt(i + i), 16) << 4)
          | Character.digit(buf.charAt(i + i + 1), 16));
    }

    return uuid;
  }

  /**
   * Get a 48 bit cryptographic quality random number to use as the node field
   * of a UUID as specified in section 6.4.1 of version 10 of the WebDAV spec.
   * This is an alternative to the IEEE 802 host address which is not available
   * from Java. The number will not conflict with any IEEE 802 host address because
   * the most significant bit of the first octet is set to 1.
   *
   * @return    a 48 bit number specifying an id for this node
   */
  private static byte[] computeNodeAddress()
  {
    // create a random number by concatenating:
    //    the hash code for the current thread
    //    the current time in milli-seconds
    //    the internet address for this node
    int thread = Thread.currentThread().hashCode();
    long time = System.currentTimeMillis();
    ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
    DataOutputStream out = new DataOutputStream(byteOut);

    try
    {
      out.write(internetAddress);
      out.write(thread);
      out.writeLong(time);
      out.close();
    }
    catch (IOException ignored) {
      //$JL-EXC$ 
      ;
    }
    byte[] rand = byteOut.toByteArray();
    MessageDigest md5 = null;
    try
    {
      md5 = MessageDigest.getInstance("MD5");
      md5.update(rand);
    }
    catch (NoSuchAlgorithmException x)
    {
      //$JL-EXC$ 
      System.err.println("No MD5 algorithm available: " + x);
    }

    // pick the middle 6 bytes of the MD5 digest
    byte[] temp = md5.digest();
    byte[] address = new byte[6];
    for (int i = 0; i < address.length; i++)
    {
      address[i] = temp[i + 5];
    }
    // set the MSB of the first octet to 1 to distinguish from IEEE node addresses
    address[0] = (byte) (address[0] | (byte)0x80);

    return address;
  }

  /**
   * Get the current time compensating for the fact that the real
   * clock resolution may be less than 100ns.
   *
   * @return    the current date and time
   */
  private static long getCurrentTime()
  {
    long now = 0;
    boolean waitForTick = true;
    while (waitForTick)
    {
      now = new Date().getTime() * 10000;
      if (lastTime < now)
      {
        // got a new tick, make sure uuidsPerTick doesn't cause an overrun
        uuidsThisTick = 0;
        waitForTick = false;
      }
      else if (uuidsThisTick < UUIDsPerTick)
      {
        //
        uuidsThisTick++;
        waitForTick = false;
      }
    }
    // add the uuidsThisTick to the time to increase the clock resolution
    now += uuidsThisTick;
    lastTime = now;
    return now;
  }
}
