/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.namespace;

import com.sap.netweaver.bc.rf.util.flyweight.*;

/**
 * Class implementing a namespace-qualified name with a namespace prefix. Please
 * see <a href="http://www.w3.org/TR/1999/REC-xml-names-19990114/#NT-NCName">
 * here</a> .
 *
 * @created 29. Januar 2003
 */

public class Name extends FlyWeight implements IName {
  private final String fullName;
  private final String localName;
  private final String namespace;

  /**
   * Construct name based on given namespace and local name.
   *
   * @param namespace namespace
   * @param localName local name
   */
  protected Name(String namespace, String localName) {
    // Register name only once
    super(
      new StringBuffer("{")
      .append(namespace)
      .append("}")
      .append(localName)
      .toString(),
      false);

    // Store fields
    this.fullName = (String)getFlyWeightId();
    this.localName = localName;
    this.namespace = (namespace == null) ? "" : namespace;
  }

  /**
   * Get name based on given full name, which is the namespace and the local
   * name in {namespace}localName format.
   *
   * @param fullName namespace and local name as full name
   * @return name
   */
  public static IName getName(String fullName) {
    // Retrieve or create flyweight name based on class and full name
    IName name = (IName)getFlyWeight(Name.class, fullName);
    if (name == null) {
      // Check full name and thereby extract namespace and local name
      if (fullName == null) {
        throw new IllegalArgumentException("Name constraint failed: Full name must not be null!");
      }
      if (fullName.charAt(0) != '{') {
        throw new IllegalArgumentException(
          "Name constraint failed: Namespace in full name \""
           + fullName
           + "\" must be in curly brackets (begining bracket is missing)!");
      }
      int rightBracket = fullName.lastIndexOf('}');
      if (rightBracket < 0) {
        throw new IllegalArgumentException(
          "Name constraint failed: Namespace in full name \""
           + fullName
           + "\" must be in curly brackets (ending bracket is missing)!");
      }
      String namespace = fullName.substring(1, rightBracket);
      String localName = fullName.substring(rightBracket + 1);

      // Check contraints
      checkConstraints(namespace, localName);

      // Create name
      try {
        name = new Name(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        if ((name = (IName)getFlyWeight(Name.class, fullName)) == null) {
          throw exception;
        }
      }
    }
    return name;
  }

  /**
   * Get name based on given namespace and local name.
   *
   * @param namespace namespace
   * @param localName local name
   * @return name
   */
  public static IName getName(String namespace, String localName) {
    // Construct full name
    String fullName = new StringBuffer("{")
      .append(namespace)
      .append("}")
      .append(localName)
      .toString();

    // Retrieve or create flyweight name based on class and full name
    IName name = (IName)getFlyWeight(Name.class, fullName);
    if (name == null) {
      // Check contraints
      checkConstraints(namespace, localName);

      // Create name
      try {
        name = new Name(namespace, localName);
      }
      catch (IllegalArgumentException exception) {
        if ((name = (IName)getFlyWeight(Name.class, fullName)) == null) {
          throw exception;
        }
      }
    }
    return name;
  }

  /**
   * Get full name, i.e. including namespace and local name.
   *
   * @return full name, i.e. including namespace and local name
   */
  public final String getFullName() {
    return fullName;
  }

  /**
   * Get local name, i.e. the part behind the namespace.
   *
   * @return local name, i.e. the part behind the namespace.
   */
  public final String getLocalName() {
    return localName;
  }

  /**
   * Get namespace of the name or an empty string if there is no namespace
   * within the name.
   *
   * @return namespace of the name or an empty string if there is no namespace
   *      within the name
   */
  public final String getNamespace() {
    return namespace;
  }

  /**
   * Get namespace of the name or null if there is no namespace within the name,
   * e.g. used when the value is going to be passed to a XML DOM level 2 method.
   *
   * @return namespace of the name or null if there is no namespace within the
   *      name
   */
  public final String getNamespaceOrNull() {
    return "".equals(namespace) ? null : namespace;
  }

  /**
   * Check this name for illegal defintion and characters and throw exception
   * when found.
   *
   * @param namespace namespace
   * @param localName local name
   * @exception IllegalArgumentException when the name contains illegal
   *      characters
   */
  public static void checkConstraints(
    final String namespace,
    final String localName)
    throws IllegalArgumentException {
    // Check namespace
    checkNamespaceConstraints(namespace);

    // Check local name
    checkLocalNameConstraints(localName);
  }

  /**
   * Check namespace for illegal format and throw exception when found.
   *
   * @param namespace namespace
   * @exception IllegalArgumentException when namespace is in illegal format
   */
  public static void checkNamespaceConstraints(final String namespace)
    throws IllegalArgumentException {
    // Check namespace for null or empty string
    if ((namespace != null) && (namespace.length() > 0)) {
      if ((Character.isWhitespace(namespace.charAt(0)))
         || (Character.isWhitespace(namespace.charAt(namespace.length() - 1)))) {
        throw new IllegalArgumentException(
          "Namespace constraint failed: Namespace \""
           + namespace
           + "\" must not begin or end with a whitespace!");
      }
    }
  }

  /**
   * Check local name for illegal format and throw exception when found.
   *
   * @param localName local name
   * @exception IllegalArgumentException when local name is in illegal format
   */
  public static void checkLocalNameConstraints(final String localName)
    throws IllegalArgumentException {
    // Check local name for null or empty string
    if ((localName == null) || (localName.length() < 1)) {
      throw new IllegalArgumentException(
        "Local name constraint failed: Local name being "
         + ((localName == null) ? "null" : "empty")
         + " must not be null or empty!");
    }

    // Check name starting letter and follow-up letters
    char[] chars = localName.toCharArray();
    if ((!Character.isLetter(chars[0])) && (chars[0] != '_')) {
      throw new IllegalArgumentException(
        "Local name constraint failed: First character of local name \""
         + localName
         + "\" must either be letter or underscore!");
    }
    for (int i = 1; i < chars.length; i++) {
      if ((!Character.isLetterOrDigit(chars[i]))
         && (chars[i] != '.')
         && (chars[i] != '-')
         && (chars[i] != '_')
         && (chars[i] != '')) {
        throw new IllegalArgumentException(
          "Local name constraint failed: Non initial characters of local name \""
           + localName
           + "\" must either be letter, digit, dot, combining dot, dash or underscore!");
      }
    }
  }
}
