/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.logging;

import com.sap.tc.logging.*;

import java.io.*;
import java.text.*;

/**
 * Trace formatter formatting logging output.
 *
 * @created 8. Mai 2001
 */
public class LoggingFormatter extends TraceFormatter {
  /**
   * Extract callstack from throwable.
   *
   * @param throwable throwable
   * @return callstack
   */
  public static String extractCallstack(Throwable throwable) {
    StringWriter stringWriter = new StringWriter();
    throwable.printStackTrace(new PrintWriter(stringWriter));
    return stringWriter.toString();
  }

  /**
   * Format log record.
   *
   * @param rec log record
   * @return formatted log record
   */
  public String format(LogRecord rec) {
    String args[] = new String[7];
    int width[] = new int[7];
    int align[] = new int[7];

    SimpleDateFormat dateFormat = new java.text.SimpleDateFormat("yyyy-MM-dd  HH:mm:ss.SSS");
    args[0] = dateFormat.format(rec.getTime());
    width[0] = 25;
    align[0] = com.sap.tc.logging.TraceFormatter.LEFT;
    args[1] = " ";
    align[1] = com.sap.tc.logging.TraceFormatter.MIN;
    String loc = rec.getLocation();
    int end = loc.lastIndexOf("(");
    if (end < 0) {
      end = loc.length();
    }
    end = loc.lastIndexOf(".", end);
    end = loc.lastIndexOf(".", end - 1);
    args[2] =
      loc.substring(end + 1)
       + " ["
       + ((rec.getThread() != null) ? rec.getThread().getName() : rec.getThreadName())
       + "]";
    width[2] = Math.max(80, args[2].length());
    align[2] = com.sap.tc.logging.TraceFormatter.LEFT;
    args[3] = " ";
    align[3] = com.sap.tc.logging.TraceFormatter.MIN;
    args[4] = com.sap.tc.logging.Severity.toString(rec.getSeverity()) + ":";
    align[4] = com.sap.tc.logging.TraceFormatter.MIN;
    args[5] = " ";
    align[5] = com.sap.tc.logging.TraceFormatter.MIN;
    switch (rec.getMsgType()) {
      case com.sap.tc.logging.MsgType.PLAIN:
      {
        args[6] = rec.getMsgClear();
        break;
      }
      case com.sap.tc.logging.MsgType.JAVA:
      {
        args[6] = MessageFormat.format(rec.getMsgClear(), rec.getArgs().toArray());
        break;
      }
    }

    align[6] = com.sap.tc.logging.TraceFormatter.MIN;

    return align(args, width, align);
  }

  public static void main(String[] args) {
    // Check arguments
    if (args.length != 1) {
      System.err.println();
      System.err.println("Usage: java <package>.LoggingFormatter <log file name>");
      System.exit(-1);
    }

    // Open file reader
    BufferedReader reader = null;
    try {
      reader = new BufferedReader(new FileReader(new File(args[0])));
    }
    catch (Exception exception) {
      //$JL-EXC$ 
      System.err.println();
      System.err.println("Failed to open file named " + args[0] + ": " + exception);
      System.exit(-2);
    }

    // Read from file
    LoggingFormatter loggingFormatter = new LoggingFormatter();
    TransientListFormatter listFormatter = new TransientListFormatter();
    StringBuffer msg = new StringBuffer();
    String line;
    LogRecord logRecord;
    try {
      while ((line = reader.readLine()) != null) {
        msg.append(line);
        if ((line.endsWith("#")) && (!line.endsWith("\\#"))) {
          logRecord = listFormatter.parseMsg(msg.toString());
          System.out.print(loggingFormatter.format(logRecord));
          msg.setLength(0);
        }
        else {
          msg.append("\r\n");
        }
      }
    }
    catch (Exception exception) {
      //$JL-EXC$ 
      System.err.println();
      System.err.println("Failed to read file named " + args[0] + ": " + exception);
      System.exit(-3);
    }
  }

  private static class TransientListFormatter extends ListFormatter {
    public TransientListFormatter() {
      super();
    }

    public LogRecord parseMsg(String msg)
      throws ParseException {
      return super.parseMsg(msg);
    }
  }
}
