/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.flyweight;

import java.io.*;

/**
 * Abstract class extending the flyweight pattern by a description. You have to
 * extend this class and make all methods derived from this abstract class final
 * (except for toString())! This is necessary in order to ensure, that no class
 * can be derived from your class overriding and thereby changing the inner
 * mechanism to create an flyweight instance only once, return the same object
 * on clone() etc. It is of greatest importance to ensure that both the id and
 * the flyweight object itself are immutable! Preknown flyweight instances may
 * be added as public final static flyweight members. If you just need to have
 * the predefined flyweight instances, implement an appropriate private
 * constructor, otherwise make it public: <pre>
 *
 * public class Property extends FlyWeightWithDescription
 * {
 *     public final static Property RO = new Property( "RO", "Read Only" );
 *     public final static Property MV = new Property( "MV", "Multi Value" );
 *
 *     private Property ( final Serializable id, final String description )
 *     {
 *         super(  id, description );
 *     }
 * }
 * </pre>
 *
 * @created 20. Januar 2003
 */
public abstract class FlyWeightWithDescription extends FlyWeight {
  // This flyweight instance description
  private final String description;

  /**
   * Construct instance of a flyweight with description.
   *
   * @param id flyweight id
   * @param description flyweight description
   * @param ignoreDuplicateRegistration when true, don't throw
   *      IllegalArgumentException exception when the id is already in use
   */
  protected FlyWeightWithDescription(final Serializable id, final String description, final boolean ignoreDuplicateRegistration) {
    // Call super constructor
    super(id, ignoreDuplicateRegistration);

    // Store description
    this.description = description;
  }

  /**
   * Get description of the flyweight instance.
   *
   * @return description of the flyweight instance
   */
  public String getDescription() {
    // Return description
    return description;
  }

  /**
   * Get descriptive text for this flyweight instance.
   *
   * @return descriptive text for this flyweight instance
   */
  public String toString() {
    // Return description
    return description;
  }
}
