/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.flyweight;

import java.io.*;
import java.util.*;

/**
 * Abstract class implementing the flyweight pattern. You have to extend this
 * class and make all methods derived from this abstract class final (except for
 * toString())! This is necessary in order to ensure, that no class can be
 * derived from your class overriding and thereby changing the inner mechanism
 * to create an flyweight instance only once, return the same object on clone()
 * etc. It is of greatest importance to ensure that both the id and the
 * flyweight object itself are immutable! Preknown flyweight instances may be
 * added as public final static flyweight members. If you just need to have the
 * predefined flyweight instances, implement an appropriate private constructor,
 * otherwise make it public: <pre>
 *
 * public class Property extends FlyWeight
 * {
 *     public final static Property RO = new Property( "RO" );
 *     public final static Property MV = new Property( "MV" );
 *
 *     private Property ( final Serializable id )
 *     {
 *         super( id );
 *     }
 * }
 * </pre>
 *
 * @created 20. Januar 2003
 */
public abstract class FlyWeight implements Cloneable, Serializable {
  // Map of flyweight classes
  private final static Map flyWeightClasses = new HashMap();

  // This flyweight instance id
  private final Serializable id;

  /**
   * Construct instance of a flyweight.
   *
   * @param id flyweight id
   * @param ignoreDuplicateRegistration when true, don't throw
   *      IllegalArgumentException exception when the id is already in use
   * @exception IllegalArgumentException when the id is already in use
   */
  protected FlyWeight(final Serializable id, final boolean ignoreDuplicateRegistration)
    throws IllegalArgumentException {
    // Store id
    this.id = id;

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Get map of instances
      Map flyweightInstances = (Map)flyWeightClasses.get(getClass());
      if (flyweightInstances == null) {
        flyWeightClasses.put(getClass(), flyweightInstances = new HashMap());
      }

      // Check if duplicate registrations should be ignored
      if (ignoreDuplicateRegistration) {
        // Add or replace instance
        flyweightInstances.put(id, this);
      }
      else {
        // Add new instance
        Object oldInstance = flyweightInstances.put(id, this);
        if (oldInstance != null) {
          // Handle already stored flyweight instance
          flyweightInstances.put(id, oldInstance);
          throw new IllegalArgumentException(
            "Failed to add flyweight with id "
             + id
             + " to map of flyweight instances. Id is already in use!");
        }
      }
    }
  }

  /**
   * Get id of the flyweight instance.
   *
   * @return flyweight id
   */
  public Serializable getFlyWeightId() {
    // Return id
    return id;
  }

  /**
   * Get descriptive text for this flyweight instance.
   *
   * @return descriptive text for this flyweight instance
   */
  public String toString() {
    // Return id
    return id.toString();
  }

  /**
   * Get hashcode for this flyweight instance.
   *
   * @return flyweight hashcode
   */
  public int hashCode() {
    // Since no flyweight instances of one flyweight class with
    // the same id are allowed and flyweight instances of
    // different flyweight classes are stored seperately,
    // just return the id to gain some performance
    return id.hashCode();
  }

  /**
   * Compare this and another object for equality. Works only fine when handling
   * objects of classes loaded by same class classloader.
   *
   * @param obj object to be compared against
   * @return result of comparison
   */
  public boolean equals(final Object obj) {
    // Since the super() call to the Object class is slower,
    // just implement the default equals() method directly
    return this == obj;
  }

  /**
   * Clone this flyweight. The clone method will always return the same instance
   * for faster equality comparison. This is not harmful, since the flyweight is
   * immutable.
   *
   * @return this flyweight instance
   */
  public Object clone() {
    // Return this instance - no clones are allowed due to identify check
    return this;
  }

  /**
   * Get flyweight instance by id of the given flyweight class.
   *
   * @param id flyweight id
   * @param flyWeightClass flyweight class
   * @return flyweight instance
   */
  public static FlyWeight getFlyWeight(final Class flyWeightClass, final Serializable id) {
    // Retrieve flyweight based on class and id
    Map flyWeightInstances = (Map)flyWeightClasses.get(flyWeightClass);
    if (flyWeightInstances != null) {
      return (FlyWeight)flyWeightInstances.get(id);
    }
    else {
      return null;
    }
  }

  /**
   * Get flyweight instance by id of the given flyweight class and all sub
   * classes of that class. If, follwoing the inheritance, multiple fyweights
   * have the same id, the first found (behaviour is not deterministic) will be
   * returned.
   *
   * @param id flyweight id
   * @param flyWeightClass flyweight class
   * @return flyweight instance
   */
  public static FlyWeight getFlyWeightInherited(final Class flyWeightClass, final Serializable id) {
    FlyWeight flyWeight = null;

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve flyweight based on class and id
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext() && flyWeight == null; ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass.isAssignableFrom(listedFlyWeightClass)) {
          flyWeight = (FlyWeight)listedFlyWeightInstances.get(id);
        }
      }
    }

    // Return found flyweight
    return flyWeight;
  }

  /**
   * Remove flyweight instance by id of the given flyweight class.
   *
   * @param id flyweight id
   * @param flyWeightClass flyweight class
   * @return flyweight instance removed or null
   */
  public static FlyWeight removeFlyWeight(final Class flyWeightClass, final Serializable id) {
    // Retrieve flyweight based on class and id
    Map flyWeightInstances = (Map)flyWeightClasses.get(flyWeightClass);
    if (flyWeightInstances != null) {
      return (FlyWeight)flyWeightInstances.remove(id);
    }
    else {
      return null;
    }
  }

  /**
   * Remove flyweight instance by id of the given flyweight class and all sub
   * classes of that class. If, follwoing the inheritance, multiple fyweights
   * have the same id, the first found (behaviour is not deterministic) will be
   * removed.
   *
   * @param id flyweight id
   * @param flyWeightClass flyweight class
   * @return flyweight instance removed or null
   */
  public static FlyWeight removeFlyWeightInherited(final Class flyWeightClass, final Serializable id) {
    FlyWeight flyWeight = null;

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve flyweight based on class and id
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext() && flyWeight == null; ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass.isAssignableFrom(listedFlyWeightClass)) {
          flyWeight = (FlyWeight)listedFlyWeightInstances.remove(id);
        }
      }
    }

    // Return found flyweight
    return flyWeight;
  }

  /**
   * Get all so far registered flyweight instances of the given flyweight class.
   * Note that only the so far registered instances can be returned. When for
   * example some instances haven't been registered, because the class wasn't
   * loaded, these instances won't be returned.
   *
   * @param flyWeightClass flyweight class for which all instances should be
   *      looked up
   * @return all so far registered flyweight instances of the given flyweight
   *      class
   */
  public static List getAllFlyWeights(final Class flyWeightClass) {
    List flyWeights = new ArrayList();

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve all flyweights based on class
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext(); ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass == listedFlyWeightClass) {
          flyWeights.addAll(listedFlyWeightInstances.values());
        }
      }
    }

    // Return found flyweights
    return flyWeights;
  }

  /**
   * Get all so far registered flyweight instances of the given flyweight class
   * and all sub classes of that class. Note that only the so far registered
   * instances can be returned. When for example some instances haven't been
   * registered, because the class wasn't loaded, these instances won't be
   * returned.
   *
   * @param flyWeightClass flyweight class (all sub classes of that class will
   *      also be checked) for which all instances should be looked up
   * @return all so far registered flyweight instances of the given flyweight
   *      class and all sub classes of that class
   */
  public static List getAllFlyWeightsInherited(final Class flyWeightClass) {
    List flyWeights = new ArrayList();

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve all flyweights based on class
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext(); ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass.isAssignableFrom(listedFlyWeightClass)) {
          flyWeights.addAll(listedFlyWeightInstances.values());
        }
      }
    }

    // Return found flyweights
    return flyWeights;
  }

  /**
   * Remove all so far registered flyweight instances of the given flyweight
   * class. Note that only the so far registered instances can be removed. When
   * for example some instances haven't been registered, because the class
   * wasn't loaded, these instances won't be removed.
   *
   * @param flyWeightClass flyweight class for which all instances should be
   *      removed
   * @return all so far removed flyweight instances of the given flyweight class
   */
  public static List removeAllFlyWeights(final Class flyWeightClass) {
    List flyWeights = new ArrayList();

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve all flyweights based on class
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext(); ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass == listedFlyWeightClass) {
          flyWeights.addAll(listedFlyWeightInstances.values());
          mapEntry.setValue(new HashMap());
        }
      }
    }

    // Return found flyweights
    return flyWeights;
  }

  /**
   * Remove all so far registered flyweight instances of the given flyweight
   * class and all sub classes of that class. Note that only the so far
   * registered instances can be removed. When for example some instances
   * haven't been registered, because the class wasn't loaded, these instances
   * won't be removed.
   *
   * @param flyWeightClass flyweight class (all sub classes of that class will
   *      also be checked) for which all instances should be removed
   * @return all so far removed flyweight instances of the given flyweight class
   *      and all sub classes of that class
   */
  public static List removeAllFlyWeightsInherited(final Class flyWeightClass) {
    List flyWeights = new ArrayList();

    // Synchronize on static map of flyweight classes
    synchronized (flyWeightClasses) {
      // Retrieve all flyweights based on class
      for (Iterator listedFlyWeightClassesIter = flyWeightClasses.entrySet().iterator(); listedFlyWeightClassesIter.hasNext(); ) {
        Map.Entry mapEntry = (Map.Entry)listedFlyWeightClassesIter.next();
        Class listedFlyWeightClass = (Class)mapEntry.getKey();
        Map listedFlyWeightInstances = (Map)mapEntry.getValue();
        if (flyWeightClass.isAssignableFrom(listedFlyWeightClass)) {
          flyWeights.addAll(listedFlyWeightInstances.values());
          mapEntry.setValue(new HashMap());
        }
      }
    }

    // Return found flyweights
    return flyWeights;
  }
}
