/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.exception;

import java.util.*;

/**
 * Implements a resource accessor wrapping the standard exception resource
 * accessor class ({@link com.sap.localization.ResourceAccessor). This class
 * also allows to access messages from resource bundles accessed by arbitrary
 * class loaders, caches them and all accessed resource bundles.
 *
 * @created 14. April 2003
 */
public class ResourceAccessor extends com.sap.localization.ResourceAccessor {
  private final ClassLoader resourceBundleClassLoader;
  private final String resourceBundleName;
  private final Map resourceBundles;

  private final static Map resourceBundleClassLoaders = new HashMap();

  private final static Locale defaultLocale = Locale.getDefault();// Performance improvement: Used frequently when no locale is supplied


  /**
   * Construct object of class ResourceAccessor.
   *
   * @param resourceBundleClassLoader class loader having access to the resource
   *      bundle
   * @param resourceBundleName resource bundle name used to look up resource
   *      bundle
   */
  private ResourceAccessor(
    ClassLoader resourceBundleClassLoader,
    String resourceBundleName) {
    super(null);
    this.resourceBundleClassLoader = resourceBundleClassLoader;
    this.resourceBundleName = resourceBundleName;
    this.resourceBundles = new HashMap();
  }


  /**
   * Get the messageText attribute of the ResourceAccessor object.
   *
   * @param locale locale for localized message
   * @param key resource identifier for localized message
   * @return The messageText value
   */
  public String getMessageText(Locale locale, String key) {
    String message = null;
    if (key != null) {
      if (locale == null) {
        locale = defaultLocale;
      }
      ResourceBundle resourceBundle =
        (ResourceBundle)resourceBundles.get(locale);
      if (resourceBundle == null) {
        resourceBundles.put(
          locale,
          resourceBundle =
          ResourceBundle.getBundle(
          resourceBundleName,
          locale,
          resourceBundleClassLoader));
        if (resourceBundle != null) {
          message = resourceBundle.getString(key);
        }
      }
      else {
        message = resourceBundle.getString(key);
      }
    }
    return message;
  }


  /**
   * Get class loader having access to the resource bundle.
   *
   * @return class loader having access to the resource bundle
   */
  public ClassLoader getResourceBundleClassLoader() {
    return resourceBundleClassLoader;
  }


  /**
   * Get resource bundle name used to look up resource bundle.
   *
   * @return resource bundle name used to look up resource bundle
   */
  public String getResourceBundleName() {
    return resourceBundleName;
  }


  /**
   * Create resource message.
   *
   * @param key resource identifier for localized message
   * @return resource message
   */
  public ResourceMessage getResourceMessage(String key) {
    return new ResourceMessage(this, key);
  }


  /**
   * Create resource message.
   *
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @return resource message
   */
  public ResourceMessage getResourceMessage(String key, Object[] args) {
    return new ResourceMessage(this, key, args);
  }


  /**
   * Get the instance attribute of the ResourceAccessor class.
   *
   * @param resourceBundleClassLoader class loader having access to the resource
   *      bundle
   * @param resourceBundleName resource bundle name used to look up resource
   *      bundle
   * @return The instance value
   */
  public static synchronized ResourceAccessor getInstance(
    ClassLoader resourceBundleClassLoader,
    String resourceBundleName) {
    Map resourceBundleNames =
      (Map)resourceBundleClassLoaders.get(resourceBundleClassLoader);
    if (resourceBundleNames == null) {
      resourceBundleClassLoaders.put(
        resourceBundleClassLoader,
        resourceBundleNames = new HashMap());
    }
    ResourceAccessor resourceAccessor =
      (ResourceAccessor)resourceBundleNames.get(resourceBundleName);
    if (resourceAccessor == null) {
      resourceBundleNames.put(
        resourceBundleName,
        resourceAccessor =
        new ResourceAccessor(
        resourceBundleClassLoader,
        resourceBundleName));
    }
    return resourceAccessor;
  }
}
