/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.exception;

/**
 * Class implementing a unchecked base exception deriving from the exception
 * framework unchecked base exception, but wrapping it due to the bad
 * localization implementation and bad name scheme in that localization package
 * and the fact that the package seems to be totally misnamed as
 * com.sap.localization being just a localization package for the exception
 * framework and due to the mandatory usage of localized messages for all kind
 * of exceptions. This class also allows to contain messages not necesssarily
 * being localized in order to support easy exception creation.
 *
 * @created 19. Mrz 2003
 */
public class BaseRuntimeException extends com.sap.exception.BaseRuntimeException {
  // Definition of default auto exception log behaviour,
  // used when no user behaviour is supplied during exception construction
  private final static boolean LOG_EXCEPTION_ON_CREATION = true;

  /**
   * Construct object of class BaseRuntimeException.
   */
  public BaseRuntimeException() {
    this(LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(boolean logExceptionOnCreation) {
    super();
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param rootCause throwable object which caused this exception
   */
  public BaseRuntimeException(Throwable rootCause) {
    this(rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(Throwable rootCause, boolean logExceptionOnCreation) {
    super(rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   */
  public BaseRuntimeException(String message) {
    this(message, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   */
  public BaseRuntimeException(String message, Object[] args) {
    this(message, args, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  public BaseRuntimeException(String message, Throwable rootCause) {
    this(message, rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  public BaseRuntimeException(String message, Object[] args, Throwable rootCause) {
    this(message, args, rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(String message, boolean logExceptionOnCreation) {
    super(new ResourceMessage(message));
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(
    String message,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(message, args));
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(
    String message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(message), rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  public BaseRuntimeException(
    String message,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(message, args), rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   */
  protected BaseRuntimeException(ResourceAccessor resourceAccessor, String key) {
    this(resourceAccessor, key, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args) {
    this(resourceAccessor, key, args, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause) {
    this(resourceAccessor, key, rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause) {
    this(resourceAccessor, key, args, rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(resourceAccessor, key));
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(resourceAccessor, key, args));
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(resourceAccessor, key), rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param resourceAccessor resource accessor
   * @param key resource identifier for localized message
   * @param args message arguments inserted by MessageFormat means
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(
    ResourceAccessor resourceAccessor,
    String key,
    Object[] args,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(new ResourceMessage(resourceAccessor, key, args), rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   */
  protected BaseRuntimeException(ResourceMessage message) {
    this(message, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   */
  protected BaseRuntimeException(ResourceMessage message, Throwable rootCause) {
    this(message, rootCause, LOG_EXCEPTION_ON_CREATION);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(ResourceMessage message, boolean logExceptionOnCreation) {
    super(message);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Construct object of class BaseRuntimeException.
   *
   * @param message not localized message
   * @param rootCause throwable object which caused this exception
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  protected BaseRuntimeException(
    ResourceMessage message,
    Throwable rootCause,
    boolean logExceptionOnCreation) {
    super(message, rootCause);
    logExceptionOnCreation(logExceptionOnCreation);
  }

  /**
   * Log the exception now during its construction when the given parameter is
   * true.
   *
   * @param logExceptionOnCreation when true log the exception now during its
   *      construction
   */
  private void logExceptionOnCreation(boolean logExceptionOnCreation) {
    // Check given parameter
    if (logExceptionOnCreation) {
      // Log the exception now during its construction
      log();
    }
  }
}
