/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.event;

import com.sap.netweaver.bc.rf.util.logging.*;

import com.sap.tc.logging.*;

/**
 * ReceiverEntry implements a data structure used by the broker for each
 * registered receiver and event template combination. It stores the receiver,
 * the event template, its receiver mode and the so far collected but not yet
 * sent events. Every asynchronous receiver gets its own event queue and sender
 * thread.
 *
 * @created 23. Januar 2003
 */
public class ReceiverEntry {
  private static Location logger = Location.getLocation(com.sap.netweaver.bc.rf.util.event.ReceiverEntry.class);

  // Maximum size of queue for not sent events
  private final static int QUEUE_MAX_SIZE = 1000;

  // Event sender thread when needed
  private EventSenderThread thread = null;

  // Event queue when needed
  private EventQueue queue = null;

  // Event receiver
  private IEventReceiver receiver;

  // Event template this entry reacts on
  private IEvent template;

  // Event mapper if supplied
  private IEventMapper mapping;

  // List of hold events
  private IEventList events;

  // Actual state of the event receiver
  private ReceiverMode state;

  // TBD: Unknown member
  private boolean collect;

  // True when a separate thread and queue should be used
  private boolean async = false;

  // Priority this receiver has compared to others
  private int priority = 0;

  /**
   * Construct instance based on parameters.
   *
   * @param receiver event receiver
   * @param template event template
   * @param mapping event mapper
   * @param priority event priority
   * @param async true when a separate thread and queue should be used
   */
  ReceiverEntry(IEventReceiver receiver, IEvent template, IEventMapper mapping, int priority, boolean async) {
    this.receiver = receiver;
    this.template = template;
    this.mapping = mapping;
    this.events = new EventList();
    this.state = ReceiverMode.SENDING;
    this.collect = false;
    this.async = async;
    this.priority = priority;

    if (async) {
      this.queue = new EventQueue(ReceiverEntry.QUEUE_MAX_SIZE);
      this.thread = new EventSenderThread(this.queue, this.receiver);
    }
  }

  /**
   * returns true when a separate thread and queue is be used.
   *
   * @return true when a separate thread and queue is be used
   */
  public boolean isAsync() {
    return this.async;
  }

  /**
   * Get event receiver priority.
   *
   * @return event receiver priority
   */
  int getPriority() {
    return this.priority;
  }

  /**
   * Get event receiver.
   *
   * @return event receiver
   */
  IEventReceiver getReceiver() {
    return this.receiver;
  }

  /**
   * Get event template.
   *
   * @return event template
   */
  IEvent getTemplate() {
    return this.template;
  }

  /**
   * Overwrite toString() for debugging purposes.
   *
   * @return string representation
   */
  public String toString() {
    if (this.template == null) {
      return this.receiver.toString() + ":null";
    }
    else {
      return this.receiver.toString() + ":" + this.template.getDescription();
    }
  }

  /**
   * Overwrite equals() for list-search and contains() purposes.
   *
   * @param other object to compare this instance against
   * @return true when equals
   */
  public boolean equals(Object other) {
    if (other instanceof ReceiverEntry) {
      return ((this.receiver == ((ReceiverEntry)other).receiver)
         && ((this.template == null) || (this.template.isLike(((ReceiverEntry)other).template))));
    }
    else {
      return false;
    }
  }

  /**
   * Send event to receiver, i.e. calls a receivers received() method or enqueue
   * event if the event receiver is asynchronous.
   *
   * @param event event
   */
  void doSend(IEvent event) {
    if (!event.isLike(this.template)) {
      return;
    }

    // Event mapping stuff...
    IEvent sendEvent = null;
    if (this.mapping == null) {
      sendEvent = event;
    }
    else {
      try {
        sendEvent = this.mapping.map(event);
      }
      catch (Throwable t) {
        logger.errorT("doSend(179)", "Exception in event mapper: " + LoggingFormatter.extractCallstack(t));
      }
    }
    if (sendEvent == null) {
      return;
    }

    if (!this.async) {
      try {
        this.receiver.received(sendEvent);
      }
      catch (Throwable t) {
        logger.errorT("doSend(191)", "Exception in event handler: " + LoggingFormatter.extractCallstack(t));
      }
      if (logger.beDebug()) {
        logger.debugT("doSend(194)", "received() method returned: " + receiver + ", event=" + sendEvent);
      }
    }
    else {
      // Asynchronous events: put into queue
      if (!this.queue.enqueue(sendEvent)) {
        logger.errorT("doSend(200)", "Failed to enqueue event: " + sendEvent.getDescription()
           + "for receiver " + this.receiver + LoggingFormatter.extractCallstack(new Exception()));
      }
    }
  }

  /**
   * Send all hold events.
   */
  void sendHoldEvents() {
    IEventListIterator iterator = this.events.listIterator();
    while (iterator.hasNext()) {
      doSend(iterator.next());
    }
    this.events = new EventList();
    this.collect = false;
  }

  /**
   * TBD: Unknown function.
   *
   * @param collect TBD: Unknown function
   */
  void setSendHoldEvents(boolean collect) {
    this.collect = collect;
  }

  /**
   * TBD: Unknown function.
   *
   * @param event event to be hold
   */
  void holdEvent(IEvent event) {

    if (!event.isLike(this.template)) {
      return;
    }

    IEvent sendEvent;
    if (this.mapping == null) {
      sendEvent = event;
    }
    else {
      sendEvent = this.mapping.map(event);
    }
    if (sendEvent == null) {
      return;
    }

    this.events.addLast(sendEvent);

  }

  /**
   * Get list all hold events.
   *
   * @return list of all hold events
   */
  IEventList getHoldEvents() {
    return this.events;
  }

  /**
   * Remove all hold events.
   *
   * @return list of all hold events
   */
  IEventList clearHoldEvents() {
    IEventList events = this.events;
    this.events = new EventList();
    return events;
  }

  /**
   * Compute state for the receiver based on state change.
   *
   * @param state state change
   */
  void setState(ReceiverMode state) {
    boolean sendhold = ((this.state.equals(ReceiverMode.HOLD))
       && (state.equals(ReceiverMode.SENDING)) && (this.collect));
    /*
     * SENDING   -> SENDING    => SENDING
     * SUSPEND   -> SENDING    => SENDING
     * HOLD      -> SENDING    => SENDING
     * SENDING   -> SUSPEND    => SUSPEND
     * SUSPEND   -> SUSPEND    => SUSPEND
     * HOLD      -> SUSPEND    => SUSPEND
     * SENDING   -> HOLD       => HOLD
     * SUSPEND   -> HOLD       => HOLD
     * HOLD      -> HOLD       => HOLD
     */
    this.state = state;
    if (sendhold) {
      sendHoldEvents();
    }
  }

  /**
   * Get state for the receiver.
   *
   * @return state for the receiver
   */
  ReceiverMode getState() {
    return this.state;
  }
}
