/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.event;

import com.sap.netweaver.bc.rf.util.exception.*;

/**
 * The event broker is a mediator between event senders ({@link IEventSender})
 * and receivers ({@link IEventReceiver}). All senders and receivers must
 * register themselfs with a broker. All receivers will receive events from all
 * senders but it is possible to register for different event "templates" (event
 * class and type). Usually a sender will instantiate its own broker and
 * register itself.
 *
 * @author Markus Breitenfelder
 * @created 23. Januar 2003
 */
public interface IEventBroker {
  public final static int PRIO_MAX = 1;
  public final static int PRIO_MIN = 99;

  /**
   * Sends the specified event to all receivers that have registered for this
   * kind of event.
   *
   * @param event event to be sent
   * @param sender event sender instance sending the event
   * @exception BaseException when the event sender is not registered
   */
  public void send(IEvent event, IEventSender sender)
    throws BaseException;

  /**
   * Registers an event sender instance.
   *
   * @param sender event sender to be registered
   * @exception BaseException when the registration fails
   */
  public void register(IEventSender sender)
    throws BaseException;

  /**
   * Registers an event receiver. The receiver will only receive events which
   * match the template event. This depends on the matching rules implemented in
   * the events {@link IEvent#isLike} method. Usually the event's instance and
   * type are relevant. It is allowed to register the same event receiver
   * instance several times with different templates.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @exception BaseException when the registration fails
   */
  public void register(IEventReceiver receiver, IEvent template)
    throws BaseException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has an additional {@link IEventMapper}
   * argument, i.e. that the mapper will be called prior to giving the event to
   * the receiver.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @exception BaseException when the registration fails
   * @see IEventMapper
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapper)
    throws BaseException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has two additional arguments for priority
   * and asynchronous events. <p>
   *
   * The priority controls the order of multible receivers which have registered
   * for the same event(s). Receivers with higher priority (smaller values) will
   * receive an event before receivers with lower priority (greater values). <p>
   *
   * An event receiver can choose to receive events asynchronously. This means
   * that a dedicated event queue and sender thread is created for each receiver
   * and template. The event sender will not be blocked, that means the send()
   * method puts the event into the queue and returns immediatelly.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception BaseException when the registration fails
   */
  public void register(IEventReceiver receiver, IEvent template, int priority, boolean async)
    throws BaseException;

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent,
   * IEventMapper)} and {@link #register(IEventReceiver, IEvent, int, boolean)}
   * method.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception BaseException when the registration fails
   * @see IEventMapper
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapper, int priority, boolean async)
    throws BaseException;

  /**
   * Unregisteres an event sender instance.
   *
   * @param sender event sender to be unregistered
   * @exception BaseException when the unregistration fails
   */
  public void unregister(IEventSender sender)
    throws BaseException;

  /**
   * Unregisteres an event receiver instance.
   *
   * @param receiver event receiver to be unregistered
   * @param template event template on which the event receiver will no longer
   *      receive events
   * @exception BaseException when the unregistration fails
   */
  public void unregister(IEventReceiver receiver, IEvent template)
    throws BaseException;

  /**
   * Suspend the event receiver. The broker will stop delivering any events
   * until {@link #resume} is called.
   *
   * @param receiver event receiver
   */
  public void suspend(IEventReceiver receiver);

  /**
   * Resumes a previously suspended event receiver. The broker will continue
   * delivering events to the receiver.
   *
   * @param receiver event receiver
   */
  public void resume(IEventReceiver receiver);

  /**
   * TBD: Unknown function.
   *
   * @param receiver TBD: Unknown parameter
   * @param collect TBD: Unknown parameter
   */
  public void hold(IEventReceiver receiver, boolean collect);

  /**
   * Get the receiver mode of the receiver, i.e. off, sending and so on.
   *
   * @param receiver event receiver
   * @return receiver mode of the receiver
   * @see ReceiverMode
   */
  public ReceiverMode getMode(IEventReceiver receiver);

  /**
   * Get all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public IEventList getHoldEvents(IEventReceiver receiver);

  /**
   * Remove all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public IEventList clearHoldEvents(IEventReceiver receiver);

  /**
   * Returns a list of event templates for all possible events this broker can
   * send (determined by querying all registered senders). The event instances
   * contained in the list are "templates", that means the event parameter is
   * not relevant.
   *
   * @return list of event templates for all possible events this broker can
   *      send
   */
  public IEventList getEvents();
}
