/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.event;

import java.util.*;

/**
 * Class implementing mutable list of {@link IEvent} instances.
 *
 * @author Markus Breitenfelder
 * @created 23. Januar 2003
 */
public class EventList implements IEventList {
  private LinkedList list = null;

  /**
   * Construct a list containing no events.
   */
  public EventList() {
    this.list = new LinkedList();
  }

  /**
   * Constructs a list containing the events of the specified list, in the order
   * they are returned by the list's iterator.
   *
   * @param list event list
   */
  public EventList(EventList list) {
    this.list = new LinkedList(list.list);
  }


  /**
   * Get number of events in list.
   *
   * @return number of events in list
   */
  public int size() {
    return this.list.size();
  }

  /**
   * Check if list is empty.
   *
   * @return true if list is empty
   */
  public boolean isEmpty() {
    return this.list.isEmpty();
  }

  /**
   * Get event by index.
   *
   * @param index index
   * @return event
   */
  public IEvent get(int index) {
    return (IEvent)this.list.get(index);
  }

  /**
   * Set event by index.
   *
   * @param index index
   * @param event event
   * @return event priorily stored at the given index
   */
  public IEvent set(int index, IEvent event) {
    return (IEvent)this.list.set(index, event);
  }

  /**
   * Add event to event list.
   *
   * @param event event
   * @return true if event was added successfully
   */
  public boolean add(IEvent event) {
    return this.list.add(event);
  }

  /**
   * Add event to the beginning of the list.
   *
   * @param event event
   */
  public void addFirst(IEvent event) {
    this.list.addFirst(event);
  }

  /**
   * Add event to the end of the list.
   *
   * @param event event
   */
  public void addLast(IEvent event) {
    this.list.addLast(event);
  }

  /**
   * Add event into list at specified index, moving all following events one
   * element further.
   *
   * @param index index
   * @param event event
   */
  public void add(int index, IEvent event) {
    this.list.add(index, event);
  }

  /**
   * Add all events from list to list.
   *
   * @param list list of events
   * @return true if all events were added successfully
   */
  public boolean addAll(IEventList list) {
    return this.list.addAll(((EventList)list).list);
  }

  /**
   * Remove event at specified index, moving all following events one element
   * closer.
   *
   * @param index index
   * @return event priorily stored at the given index
   */
  public IEvent remove(int index) {
    return (IEvent)this.list.remove(index);
  }

  /**
   * Remove all events from list.
   */
  public void clear() {
    this.list.clear();
  }

  /**
   * Get an iterator of the elements in this list (in proper sequence).
   *
   * @return list iterator
   */
  public IEventListIterator listIterator() {
    return new EventListIterator(this.list.listIterator());
  }

  /**
   * Returns a list iterator of the elements in this list (in proper sequence),
   * starting at the specified position in the list. The specified index
   * indicates the first element that would be returned by an initial call to
   * the next method. An initial call to the previous method would return the
   * element with the specified index minus one.
   *
   * @param index index
   * @return list iterator
   */
  public IEventListIterator listIterator(int index) {
    return new EventListIterator(this.list.listIterator(index));
  }

  /**
   * Get string representation of the list.
   *
   * @return string representation of the list
   */
  public String toString() {
    return this.list.toString();
  }
}
