/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.event;

import com.sap.netweaver.bc.rf.util.exception.*;

import com.sap.tc.logging.*;

import java.util.*;

/**
 * Default class implementation of an event broker.
 *
 * @author Markus Breitenfelder
 * @created 23. Januar 2003
 * @see IEventBroker
 */
public class EventBroker implements IEventBroker {

  private static Location logger = Location.getLocation(com.sap.netweaver.bc.rf.util.event.EventBroker.class);

  // All registered senders
  protected final HashMap senderMap;

  // List of receiver entries, in priority order
  protected List receivers;

  /**
   * Construct object of class EventBroker.
   */
  public EventBroker() {
    this.senderMap = new HashMap();
    this.receivers = Collections.EMPTY_LIST;
  }

  /**
   * Sends the specified event to all receivers that have registered for this
   * kind of event.
   *
   * @param event event to be sent
   * @param sender event sender instance sending the event
   * @exception BaseException when the event sender is not registered
   */
  public void send(IEvent event, IEventSender sender)
    throws BaseException {
    if (event != null) {

      if (!this.senderMap.containsKey(((Object)sender).toString())) {
        throw new BaseException("Event sender is not registered: " + ((Object)sender).toString());
      }

      List rl = getReceivers();
      for (int i = 0, n = rl.size(); i < n; ++i) {
        ReceiverEntry entry = (ReceiverEntry)rl.get(i);
        if (entry.getState().equals(ReceiverMode.SENDING)) {
          entry.doSend(event);
        }
        else if (entry.getState().equals(ReceiverMode.HOLD)) {
          entry.holdEvent(event);
        }
      }
    }
  }

  /**
   * Registers an event sender instance.
   *
   * @param sender event sender to be registered
   * @exception BaseException when the registration fails
   */
  public void register(IEventSender sender)
    throws BaseException {
    synchronized (this.senderMap) {
      if (!this.senderMap.containsKey(((Object)sender).toString())) {
        this.senderMap.put(((Object)sender).toString(), sender);
      }
      else {
        throw new BaseException("Sender is already registered: " + ((Object)sender).toString());
      }
    }
  }

  /**
   * Registers an event receiver. The receiver will only receive events which
   * match the template event. This depends on the matching rules implemented in
   * the events {@link IEvent#isLike} method. Usually the event's instance and
   * type are relevant. It is allowed to register the same event receiver
   * instance several times with different templates.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @exception BaseException when the registration fails
   */
  public void register(IEventReceiver receiver, IEvent template)
    throws BaseException {
    this.register(receiver, template, null, IEventBroker.PRIO_MIN, false);
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has an additional {@link IEventMapper}
   * argument, i.e. that the mapper will be called prior to giving the event to
   * the receiver.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @exception BaseException when the registration fails
   * @see IEventMapper
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapper)
    throws BaseException {
    this.register(receiver, template, mapper, IEventBroker.PRIO_MIN, false);
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent)}
   * method. This regstration method has two additional arguments for priority
   * and asynchronous events. <p>
   *
   * The priority controls the order of multible receivers which have registered
   * for the same event(s). Receivers with higher priority (smaller values) will
   * receive an event before receivers with lower priority (greater values). <p>
   *
   * An event receiver can choose to receive events asynchronously. This means
   * that a dedicated event queue and sender thread is created for each receiver
   * and template. The event sender will not be blocked, that means the send()
   * method puts the event into the queue and returns immediatelly.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception BaseException when the registration fails
   */
  public void register(IEventReceiver receiver, IEvent template, int priority, boolean async)
    throws BaseException {
    this.register(receiver, template, null, priority, async);
  }

  /**
   * See the general contract of the {@link #register(IEventReceiver, IEvent,
   * IEventMapper)} and {@link #register(IEventReceiver, IEvent, int, boolean)}
   * method.
   *
   * @param receiver event receiver to be registered
   * @param template event template on which the event receiver will receive
   *      events
   * @param mapper event mapper to be called prior to giving the event to the
   *      receiver
   * @param priority priority of this receiver and template
   * @param async true when the receiver whould receive the events on a seperate
   *      thread
   * @exception BaseException when the registration fails
   * @see IEventMapper
   */
  public void register(IEventReceiver receiver, IEvent template, IEventMapper mapper, int priority, boolean async)
    throws BaseException {
    ReceiverEntry entry = new ReceiverEntry(receiver, template, mapper, priority, async);
    if (addReceiver(entry)) {
      logger.debugT("register(179)", "Receiver registered: " + ((Object)receiver).toString() +
        ", event=" + template + ", =" + mapper + ", prio=" + priority + ", async=" + async);
    }
    else {
      logger.debugT("register(183)", "Receiver is already registered for event(s): " + ((Object)receiver).toString() +
        ", event=" + template + ", =" + mapper + ", prio=" + priority + ", async=" + async);
    }
  }

  /**
   * Unregisteres an event sender instance.
   *
   * @param sender event sender to be unregistered
   * @exception BaseException when the unregistration fails
   */
  public void unregister(IEventSender sender)
    throws BaseException {
    synchronized (this.senderMap) {
      this.senderMap.remove(((Object)sender).toString());
    }
  }

  /**
   * Unregisteres an event receiver instance.
   *
   * @param receiver event receiver to be unregistered
   * @param template event template on which the event receiver will no longer
   *      receive events
   * @exception BaseException when the unregistration fails
   */
  public void unregister(IEventReceiver receiver, IEvent template)
    throws BaseException {
    removeReceiver(receiver);
  }

  /**
   * Suspend the event receiver. The broker will stop delivering any events
   * until {@link #resume} is called.
   *
   * @param receiver event receiver
   */
  public void suspend(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setState(ReceiverMode.SUSPEND);
      return;
    }
  }

  /**
   * Resumes a previously suspended event receiver. The broker will continue
   * delivering events to the receiver.
   *
   * @param receiver event receiver
   */
  public void resume(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setState(ReceiverMode.SENDING);
      return;
    }
  }

  /**
   * TBD: Unknown function.
   *
   * @param receiver TBD: Unknown parameter
   * @param collect TBD: Unknown parameter
   */
  public void hold(IEventReceiver receiver, boolean collect) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      entry.setSendHoldEvents(collect);
      entry.setState(ReceiverMode.HOLD);
      return;
    }
  }

  /**
   * Get the receiver mode of the receiver, i.e. off, sending and so on.
   *
   * @param receiver event receiver
   * @return receiver mode of the receiver
   * @see ReceiverMode
   */
  public ReceiverMode getMode(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.getState();
    }
    else {
      return ReceiverMode.OFF;
    }
  }

  /**
   * Get all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public IEventList getHoldEvents(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.getHoldEvents();
    }
    else {
      return null;
    }
  }

  /**
   * Remove all hold but not yet sent events for the given receiver.
   *
   * @param receiver event receiver
   * @return all hold but not yet sent events for the given receiver
   */
  public IEventList clearHoldEvents(IEventReceiver receiver) {
    ReceiverEntry entry = searchEntry(receiver);
    if (entry != null) {
      return entry.clearHoldEvents();
    }
    else {
      return null;
    }
  }

  /**
   * Returns a list of event templates for all possible events this broker can
   * send (determined by querying all registered senders). The event instances
   * contained in the list are "templates", that means the event parameter is
   * not relevant.
   *
   * @return list of event templates for all possible events this broker can
   *      send
   */
  public IEventList getEvents() {
    IEventList list = new EventList();
    synchronized (this.senderMap) {
      Iterator it = this.senderMap.values().iterator();
      while (it.hasNext()) {
        IEventSender sender = (IEventSender)it.next();
        IEventList l = sender.getEvents();
        if (l != null) {
          list.addAll(l);
        }
      }
    }
    return list;
  }

  /**
   * Get all registered receiver entries.
   *
   * @return all registered receiver entries
   */
  protected synchronized List getReceivers() {
    return this.receivers;
  }

  /**
   * Add receiver entry.
   *
   * @param entry receiver entry to be added
   * @return true when successful
   */
  private synchronized boolean addReceiver(ReceiverEntry entry) {
    if (!this.receivers.contains(entry)) {
      List nrec = new ArrayList(this.receivers.size() + 1);
      nrec.addAll(this.receivers);
      nrec.add(entry);
      Collections.sort(nrec, new PrioComparator());
      this.receivers = nrec;
      return true;
    }
    else {
      return false;
    }
  }

  /**
   * Remove receiver entry.
   *
   * @param receiver receiver to be removed
   */
  private synchronized void removeReceiver(IEventReceiver receiver) {
    List rl = new ArrayList(this.receivers);
    boolean modified = false;
    for (int i = 0, n = rl.size(); i < n; ++i) {
      ReceiverEntry entry = (ReceiverEntry)rl.get(i);
      if (entry.getReceiver() == receiver) {
        rl.remove(i);
        --i;
        modified = true;
      }
    }
    if (modified) {
      this.receivers = rl;
    }
  }

  /**
   * Search receiver entry for receiver.
   *
   * @param receiver receiver
   * @return receiver entry
   */
  private ReceiverEntry searchEntry(IEventReceiver receiver) {
    List rl = getReceivers();
    for (int i = 0, n = rl.size(); i < n; ++i) {
      ReceiverEntry entry = (ReceiverEntry)rl.get(i);
      if (entry.getReceiver() == receiver) {
        return entry;
      }
    }
    return null;
  }

  /**
   * Class comparing two receivers for priority.
   *
   * @created 23. Januar 2003
   */
  private class PrioComparator implements Comparator {
    /**
     * Compare two receivers for priority.
     *
     * @param o1 receiver to be compared for priority
     * @param o2 receiver to be compared for priority
     * @return priority order
     */
    public int compare(Object o1, Object o2) {
      ReceiverEntry e1 = (ReceiverEntry)o1;
      ReceiverEntry e2 = (ReceiverEntry)o2;
      if ((e1.isAsync() && e2.isAsync()) || (!e1.isAsync() && !e2.isAsync())) {
        // if both rec. are sync. or async. compare the priority
        return e1.getPriority() - e2.getPriority();
      }
      else {
        if (e1.isAsync()) {
          // async. receivers must always receive events after sync. receivers
          return 1;
        }
        else {
          return -1;
        }
      }
    }
  }
}
