/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.event;

import com.sap.netweaver.bc.rf.util.exception.*;

import java.util.*;

/**
 * Abstract base class implementation of an event mapper.
 *
 * @author Paul Goetz
 * @created 23. Januar 2003
 * @see IEventMapper
 */
public abstract class AbstractEventMapper
   implements IEventMapper {
  // System id for this mapper
  protected String m_ID = null;

  // Properties for this mapper
  protected Properties m_Properties = null;

  protected final static String CONFIG_NAME_TAG = "name";

  /**
   * Create a mapper with a specific id and the given custom properties.
   *
   * @param id id
   * @param properties properties
   * @exception BaseException Exception raised in failure situation
   */
  protected AbstractEventMapper(String id,
    Properties properties)
    throws BaseException {
    m_ID = id;
    m_Properties = properties;
  }

  /**
   * Get the mapper's unique system id.
   *
   * @return mapper's unique system id
   */
  public String getID() {
    return m_ID;
  }

  /**
   * Get the mapper's displayable name.
   *
   * @return mapper's displayable name
   */
  public String getDescription() {

    return m_Properties.getProperty(CONFIG_NAME_TAG, m_ID);
  }

  /**
   * Get the mapper's localized displayable name.
   *
   * @param locale locale
   * @return mapper's localized displayable name
   */
  public String getDescription(Locale locale) {

    return getDescription();
  }

  /**
   * Map or transform an event prior to sending it to the receiver from one kind
   * to another which then is sent to the receiver. It might be a good idea to
   * put the orginal event in the user defined parameter of the new mapped and
   * sent event.
   *
   * @param event event to be mapped
   * @return mapped event
   */
  public abstract IEvent map(IEvent event);

  /**
   * Returns a list of mapped event templates for all possible events this
   * mapper can send. The event instances contained in the list are "templates",
   * that means the event parameter is not relevant. An implementation of this
   * method should call the broker and ask it to call all its senders to get the
   * list of all possible event templates which can possibly be handed over to a
   * mapper/receiver. Then the mapper should check, which of these event
   * templates are supported and can be mapped and should return these mapped
   * event templates in a list.
   *
   * @param broker broker to be asked for all possible events of all of its
   *      senders
   * @return list of event templates for all possible events this mapper can
   *      return in conjunction with the broker
   */
  public IEventList getEvents(IEventBroker broker) {
    IEventList events = new EventList();

    if (broker == null) {
      return events;
    }

    IEventList brokerEvents = broker.getEvents();
    IEventListIterator brokerEventIterator = brokerEvents.listIterator();
    while (brokerEventIterator.hasNext()) {
      IEvent mappedEvent = map(brokerEventIterator.next());
      if (mappedEvent != null) {
        events.add(mappedEvent);
      }
    }

    return events;
  }
}
