/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.enum;

import com.sap.netweaver.bc.rf.util.flyweight.*;

import java.io.*;
import java.util.*;

/**
 * Abstract class implementing a type-safe enumeration. You have to extend this
 * class, add your public final static enumeration members and implement an
 * appropriate private constructor: <pre>
 * public class Status extends Enum
 * {
 *     public final static Status SUCCESS = new Status( 1 );
 *     public final static Status FAILURE = new Status( 2 );
 *
 *     private A( final int id )
 *     {
 *         super( id );
 *     }
 * }
 * </pre>
 *
 * @created 20. Januar 2003
 */
public abstract class Enum extends FlyWeight implements Cloneable, Comparable, Serializable {
  // This enum instance id
  private final int id;

  /**
   * Construct instance of an enumeration.
   *
   * @param id enumeration id
   * @param ignoreDuplicateRegistration when true, don't throw
   *      IllegalArgumentException exception when the id is already in use
   */
  protected Enum(final int id, final boolean ignoreDuplicateRegistration) {
    // Call flyweight constructor
    super(new Integer(id), ignoreDuplicateRegistration);

    // Store id
    this.id = id;
  }

  /**
   * Get id of the enumeration instance.
   *
   * @return enumeration id
   */
  public final int getEnumId() {
    // Return id
    return id;
  }

  /**
   * Get descriptive text for this enumeration instance.
   *
   * @return descriptive text for this enumeration instance
   */
  public String toString() {
    // Return id
    return Integer.toString(id);
  }

  /**
   * Get hashcode for this enumeration instance.
   *
   * @return enumeration hashcode
   */
  public final int hashCode() {
    // Since no enum instances of one enum class with
    // the same id are allowed and enum instances of
    // different enum classes are stored seperately,
    // just return the id to gain some performance
    return id;
  }

  /**
   * Compare this and another object for equality. Works only fine when handling
   * objects of classes loaded by same class classloader.
   *
   * @param obj object to be compared against
   * @return result of comparison
   */
  public final boolean equals(final Object obj) {
    // Since the super() call to the Object class is slower,
    // just implement the default equals() method directly
    return this == obj;
  }

  /**
   * Clone this enumeration. The clone method will always return the same
   * instance for faster equality comparison. This is not harmful, since the
   * enumeration is immutable.
   *
   * @return this enumeration instance
   */
  public final Object clone() {
    // Return this instance - no clones are allowed due to identify check
    return this;
  }

  /**
   * Compare this and another object for order.
   *
   * @param obj object to be compared against
   * @return result of comparison
   */
  public final int compareTo(final Object obj) {
    // Compare other object with this enum - throw exception when of wrong type
    return id - ((Enum)obj).id;
  }

  /**
   * Get enumeration instance by id of the given enumeration class.
   *
   * @param enumClass enumeration class
   * @param id enumeration id
   * @return enumeration instance
   */
  public final static Enum getEnum(final Class enumClass, final int id) {
    // Call flyweight member
    return (Enum)getFlyWeight(enumClass, new Integer(id));
  }

  /**
   * Get enumeration instance by id of the given enumeration class and all sub
   * classes of that class. If, follwoing the inheritance, multiple enumerations
   * have the same id, the first found (behaviour is not deterministic) will be
   * returned.
   *
   * @param enumClass enumeration class
   * @param id enumeration id
   * @return enumeration instance
   */
  public final static Enum getEnumInherited(final Class enumClass, final int id) {
    // Call flyweight member
    return (Enum)getFlyWeightInherited(enumClass, new Integer(id));
  }

  /**
   * Get all so far registered enumeration instances of the given enumeration
   * class. Note that only the so far registered instances can be returned. When
   * for example some instances haven't been registered, because their class
   * wasn't loaded, these instances won't be returned.
   *
   * @param enumClass enumeration class
   * @return all so far registered enumeration instances of the given
   *      enumeration class
   */
  public static List getAllEnums(final Class enumClass) {
    // Call flyweight member
    return getAllFlyWeights(enumClass);
  }

  /**
   * Get all so far registered enumeration instances of the given enumeration
   * class and all sub classes of that class. Note that only the so far
   * registered instances can be returned. When for example some instances
   * haven't been registered, because their class wasn't loaded, these instances
   * won't be returned.
   *
   * @param enumClass enumeration class
   * @return all so far registered enumeration instances of the given
   *      enumeration class and all sub classes of that class
   */
  public static List getAllEnumsInherited(final Class enumClass) {
    // Call flyweight member
    return getAllFlyWeightsInherited(enumClass);
  }
}
