/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.context;

/**
 * <p>
 *
 * Interface describing a context factory, i.e. a factory used to get and set a
 * context (some available information). Retrieving the context should be
 * possible without the need for arguments, e.g. by looking into a thread local
 * variable.</p> <p>
 *
 * A class implementing this interface must implement the singleton pattern and
 * must register with the context factory registry before any context is
 * requested from that registry. This is automatically done, when subclassing
 * the abstract context factory classes.</p> <p>
 *
 * There are two different kinds of context factories. One, the independent one,
 * that supports all optional write operations and another one, the dependent
 * one, that depends on another context factory and therefore doesn't support
 * the optional write operations, since they would need to modify other contexts
 * in an unexpected way:</p> <p>
 *
 * <pre>
 * Mandatory read operation:
 *     public boolean isAvailable();
 *     public int size();
 *     public Object getTypeUnsafeContext() throws ContextUnavailableException;
 *     public Object getTypeUnsafeContextOrNullIfUnavailable();
 *
 * Optional write operations:
 *     public void setTypeUnsafeContext( Object context ) throws UnsupportedOperationException;
 *     public void removeTypeUnsafeContext( ) throws UnsupportedOperationException;
 *     public Object popTypeUnsafeContext() throws ContextUnavailableException, UnsupportedOperationException;
 *     public void pushTypeUnsafeContext( Object context ) throws UnsupportedOperationException;
 * </pre> <p>
 *
 * All context factories may implement additional type-safe versions of the
 * above methods. Therefore the signatures contain the word "TypeUnsafe" in the
 * name.</p> <p>
 *
 * The dependent context factories are used to hide the computation of
 * information normally spread over multiple code spots and are therefore to be
 * prefered over own code doing the computation based on independent context
 * factories. </p> <p>
 *
 * Here an example for an independent context factory:</p> <pre>
 * public class IndependentUserContextFactory extends AbstractIndependentThreadContextFactory
 * {
 *     private static IndependentUserContextFactory inst = new IndependentUserContextFactory();
 *
 *     public static IndependentUserContextFactory getInstance()
 *     {
 *         return inst;
 *     }
 *
 *     public User getContext() throws ContextUnavailableException
 *     {
 *         return ( User )inst.getTypeUnsafeContext();
 *     }
 *
 *     public void setContext( User context )
 *     {
 *         inst.setTypeUnsafeContext( context );
 *     }
 *
 *     public void removeContext( )
 *     {
 *         inst.removeTypeUnsafeContext( );
 *     }
 *
 *     public User popContext() throws ContextUnavailableException
 *     {
 *         return ( User )inst.popTypeUnsafeContext();
 *     }
 *
 *     public void pushContext( User context )
 *     {
 *         inst.pushTypeUnsafeContext( context );
 *     }
 * }
 * </pre> <p>
 *
 * Here an example for a dependent context factory:</p> <pre>
 * public class DependentLocaleContextFactory extends AbstractDependentContextFactory
 * {
 *     private static DependentLocaleContextFactory inst = new DependentLocaleContextFactory();
 *
 *     public static DependentLocaleContextFactory getInstance()
 *     {
 *         return inst;
 *     }
 *
 *     public Object getTypeUnsafeContext() throws ContextUnavailableException
 *     {
 *         return IndependentUserContextFactory.getInstance().getContext().getLocale();
 *     }
 *
 *     public Locale getContext() throws ContextUnavailableException
 *     {
 *         return IndependentUserContextFactory.getInstance().getContext().getLocale();
 *     }
 * }
 * </pre>
 *
 * @created 20. Januar 2003
 */
public interface IContextFactory {
  /**
   * Checks if there is at least one actual context stacked.
   *
   * @return flag indicating whether there is at least one actual context
   *      stacked
   */
  public boolean isAvailable();

  /**
   * Get the number of context entries in the stack of contexts.
   *
   * @return number of context entries in the stack of contexts
   */
  public int size();

  /**
   * Get actual context from stack of contexts.
   *
   * @return actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   */
  public Object getTypeUnsafeContext()
    throws ContextUnavailableException;

  /**
   * Get actual context from stack of contexts. Return null if none is availble
   * and don't throw a ContextUnavailableException.
   *
   * @return actual context on the context stack
   */
  public Object getTypeUnsafeContextOrNullIfUnavailable();

  /**
   * Replace actual context with new one in stack of contexts (optional
   * operation). If no context is stored, this one will be added.
   *
   * @param context new actual context replacing the old one on the stack
   * @exception UnsupportedOperationException when the context factory isn't
   *      supporting this writing optional operation
   */
  public void setTypeUnsafeContext(Object context)
    throws UnsupportedOperationException;

  /**
   * Remove actual context including all other stacked contexts completely
   * (optional operation).
   *
   * @exception UnsupportedOperationException when the context factory isn't
   *      supporting this writing optional operation
   */
  public void removeTypeUnsafeContext()
    throws UnsupportedOperationException;

  /**
   * Get and remove actual context from the stack of contexts (optional
   * operation).
   *
   * @return last actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   * @exception UnsupportedOperationException when the context factory isn't
   *      supporting this writing optional operation
   */
  public Object popTypeUnsafeContext()
    throws ContextUnavailableException, UnsupportedOperationException;

  /**
   * Add new context to the top of the stack of contexts (optional operation).
   *
   * @param context new actual context
   * @exception UnsupportedOperationException when the context factory isn't
   *      supporting this writing optional operation
   */
  public void pushTypeUnsafeContext(Object context)
    throws UnsupportedOperationException;
}
