/*
 * Copyright (c) 2003 by SAP AG. All Rights Reserved.
 *
 * SAP, mySAP, mySAP.com and other SAP products and
 * services mentioned herein as well as their respective
 * logos are trademarks or registered trademarks of
 * SAP AG in Germany and in several other countries all
 * over the world. MarketSet and Enterprise Buyer are
 * jointly owned trademarks of SAP AG and Commerce One.
 * All other product and service names mentioned are
 * trademarks of their respective companies.
 *
 * @version $Id$
 */

package com.sap.netweaver.bc.rf.util.context;

import java.util.*;

/**
 * Context factory registry holding all registered context factories and
 * providing means to get and set a context. In order to access a context
 * through this class, each context factory needs to register with this
 * registry. This must be done prior to any call trying to access a context for
 * that context factory. Beware of the dangers registering a context factory in
 * the static initializer. No one ensures, that the static initializer is called
 * before a context for the context factory is requested by the type unsafe API
 * of this registry.
 *
 * @created 20. Januar 2003
 * @see IContextFactory
 */
public class ContextFactoryRegistry {
  private static Map contextFactories = new HashMap();

  /**
   * Register context factory instance.
   *
   * @param contextFactory context factory instance to be registered
   * @exception ContextFactoryRegistrationError when the context factory is
   *      already registered with the context factory registry
   */
  public static void registerContextFactory(IContextFactory contextFactory)
    throws ContextFactoryRegistrationError {
    // Check argument
    if (contextFactory == null) {
      throw new IllegalArgumentException("Factory to be registered may not be null!");
    }

    // Synchronize on static map of factories
    synchronized (contextFactories) {
      // Add new factory
      Object oldFactory =
        contextFactories.put(
        contextFactory.getClass().getName(),
        contextFactory);
      if (oldFactory != null) {
        // Handle already stored factory
        contextFactories.put(
          oldFactory.getClass().getName(),
          oldFactory);
        throw new ContextFactoryRegistrationError(
          (IContextFactory)oldFactory);
      }
    }
  }

  /**
   * Unregister context factory by context factory class (name).
   *
   * @param contextFactoryClass context factory class (name) to be unregistered
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static void unregisterContextFactory(Class contextFactoryClass)
    throws ContextFactoryRegistrationError {
    // Check argument
    if (contextFactoryClass == null) {
      throw new IllegalArgumentException("Factory to be unregistered may not be null!");
    }

    // Synchronize on static map of factories
    synchronized (contextFactories) {
      // Remove old factory
      if (contextFactories.remove(contextFactoryClass.getName()) == null) {
        // Handle not stored factory
        throw new IllegalArgumentException(
          "Failed to unregister context factory with class name"
           + contextFactoryClass.getName()
           + ". There is no context factory registered with this class name!");
      }
    }
  }

  /**
   * Unregister context factory by context factory instance.
   *
   * @param contextFactory context factory to be unregistered
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static void unregisterContextFactory(IContextFactory contextFactory)
    throws ContextFactoryRegistrationError {
    // Check argument
    if (contextFactory == null) {
      throw new IllegalArgumentException("Factory to be unregistered may not be null!");
    }

    // Delegate call
    unregisterContextFactory(contextFactory.getClass());
  }

  /**
   * Internally look up context factory based on it's class (name).
   *
   * @param contextFactoryClass context factory class used to look up context
   *      factory
   * @return context factory looked up
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  private static IContextFactory getContextFactory(Class contextFactoryClass)
    throws ContextFactoryRegistrationError {
    IContextFactory contextFactory =
      (IContextFactory)contextFactories.get(
      contextFactoryClass.getName());
    if (contextFactory == null) {
      throw new ContextFactoryRegistrationError(contextFactoryClass);
    }
    return contextFactory;
  }

  /**
   * Get actual context from stack of contexts.
   *
   * @param contextFactoryClass context factory class used to look up context
   *      factory
   * @return actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static Object getTypeUnsafeContext(Class contextFactoryClass)
    throws ContextUnavailableException, ContextFactoryRegistrationError {
    return getContextFactory(contextFactoryClass).getTypeUnsafeContext();
  }

  /**
   * Get actual context from stack of contexts. Return null if none is availble
   * and don't throw a ContextUnavailableException.
   *
   * @param contextFactoryClass TBD: Description of the incoming method
   *      parameter
   * @return actual context on the context stack
   */
  public Object getTypeUnsafeContextOrNullIfUnavailable(Class contextFactoryClass) {
    return getContextFactory(contextFactoryClass).getTypeUnsafeContextOrNullIfUnavailable();
  }

  /**
   * Replace actual context with new one in stack of contexts. If no context is
   * stored, this one will be added.
   *
   * @param context new actual context replacing the old one on the stack
   * @param contextFactoryClass context factory class used to look up context
   *      factory
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static void setTypeUnsafeContext(
    Class contextFactoryClass,
    Object context)
    throws ContextFactoryRegistrationError {
    getContextFactory(contextFactoryClass).setTypeUnsafeContext(context);
  }

  /**
   * Get and remove actual context from the stack of contexts.
   *
   * @param contextFactoryClass context factory class used to look up context
   *      factory
   * @return last actual context on the context stack
   * @exception ContextUnavailableException when no (more) context is available
   *      on the context stack
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static Object popTypeUnsafeContext(Class contextFactoryClass)
    throws ContextUnavailableException, ContextFactoryRegistrationError {
    return getContextFactory(contextFactoryClass).popTypeUnsafeContext();
  }

  /**
   * Add new context to the top of the stack of contexts.
   *
   * @param context new actual context
   * @param contextFactoryClass context factory class used to look up context
   *      factory
   * @exception ContextFactoryRegistrationError when the context factory isn't
   *      registered with the context factory registry
   */
  public static void pushTypeUnsafeContext(
    Class contextFactoryClass,
    Object context)
    throws ContextFactoryRegistrationError {
    getContextFactory(contextFactoryClass).pushTypeUnsafeContext(context);
  }
}
